/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Juergen Schumacher, Andreas Weber, Drazen Cindric, Andreas Schank (all Attensity Europe GmbH) - initial
 * implementation
 **********************************************************************************************************************/
package org.eclipse.smila.jobmanager.test;

import java.util.Map;

import org.eclipse.smila.datamodel.AnyMap;

/**
 * Tests for Job Run Canceling.
 * 
 */
public class TestJobRunUsingStores extends JobManagerTestBase {

  private static final String TEST_STORE = "test";

  private static final String OTHER_STORE = "other";

  /** {@inheritDoc} */
  @Override
  protected void setUp() throws Exception {
    super.setUp();
    _objectStoreService.ensureStore(TEST_STORE);
    _objectStoreService.ensureStore(OTHER_STORE);
  }

  /**
   * no jobs defined. must deliver zero jobs.
   */
  public void testNoJobRuns() throws Exception {
    // be sure that no job runs exist.   
    assertTrue(_jobRunEngine.getJobRunsUsingStore(TEST_STORE).isEmpty());
  }

  /**
   * two jobs defined but not started. must deliver zero jobs.
   */
  public void testTwoJobsNotStarted() throws Exception {
    final String jobName = "testJob";
    final String workerParameterValue = "test";
    final AnyMap parameters = createJobParameters(workerParameterValue, TEST_STORE, TEST_STORE);
    addJob(jobName, "testWorkflow", parameters);

    final String jobName2 = "testJob2";
    final AnyMap parameters2 = createJobParameters(workerParameterValue, OTHER_STORE, OTHER_STORE);
    addJob(jobName2, "testWorkflow", parameters2);

    assertTrue(_jobRunEngine.getJobRunsUsingStore(TEST_STORE).isEmpty());
  }

  /**
   * two jobs defined but the other one is started. must deliver zero jobs.
   */
  public void testTwoJobsOnlyOtherStarted() throws Exception {
    final String jobName = "testJob";
    final String workerParameterValue = "test";
    final AnyMap parameters = createJobParameters(workerParameterValue, TEST_STORE, OTHER_STORE);
    addJob(jobName, "testWorkflow", parameters);

    final String jobName2 = "testJob2";
    final AnyMap parameters2 = createJobParameters(workerParameterValue, OTHER_STORE, OTHER_STORE);
    addJob(jobName2, "testWorkflow", parameters2);

    final String jobId2 = startJob(jobName2);
    assertTrue(_jobRunEngine.getJobRunsUsingStore(TEST_STORE).isEmpty());
    _jobRunEngine.finishJob(jobName2, jobId2);
  }

  /**
   * two jobs defined (one using the test store) and both started. must deliver one job.
   */
  public void testTwoJobsBothStarted() throws Exception {
    final String jobName = "testJob";
    final String workerParameterValue = "test";
    final AnyMap parameters = createJobParameters(workerParameterValue, TEST_STORE, OTHER_STORE);
    addJob(jobName, "testWorkflow", parameters);

    final String jobName2 = "testJob2";
    final AnyMap parameters2 = createJobParameters(workerParameterValue, OTHER_STORE, OTHER_STORE);
    addJob(jobName2, "testWorkflow", parameters2);

    final String jobId = startJob(jobName);
    final String jobId2 = startJob(jobName2);
    final Map<String, String> jobRunMap = _jobRunEngine.getJobRunsUsingStore(TEST_STORE);
    assertEquals(jobId, jobRunMap.get(jobName));
    assertEquals(1, jobRunMap.size());
    _jobRunEngine.finishJob(jobName, jobId);
    _jobRunEngine.finishJob(jobName2, jobId2);
  }

  /**
   * three jobs defined (two using tested store) and one of those using tested store started. must deliver one job.
   */
  public void testThreeJobsOneMatchingStarted() throws Exception {
    final String jobName = "testJob";
    final String workerParameterValue = "test";
    final AnyMap parameters = createJobParameters(workerParameterValue, TEST_STORE, OTHER_STORE);
    addJob(jobName, "testWorkflow", parameters);

    final String jobName2 = "testJob2";
    final AnyMap parameters2 = createJobParameters(workerParameterValue, OTHER_STORE, OTHER_STORE);
    addJob(jobName2, "testWorkflow", parameters2);

    final String jobName3 = "testJob3";
    final AnyMap parameters3 = createJobParameters(workerParameterValue, TEST_STORE, OTHER_STORE);
    addJob(jobName3, "testWorkflow", parameters3);

    final String jobId = startJob(jobName);
    final String jobId2 = startJob(jobName2);
    final Map<String, String> jobRunMap = _jobRunEngine.getJobRunsUsingStore(TEST_STORE);
    assertEquals(jobId, jobRunMap.get(jobName));
    assertEquals(1, jobRunMap.size());
    _jobRunEngine.finishJob(jobName, jobId);
    _jobRunEngine.finishJob(jobName2, jobId2);
  }

  /**
   * three jobs defined (two using tested store) and all started. must deliver two jobs.
   */
  public void testThreeJobsAllStarted() throws Exception {
    final String jobName = "testJob";
    final String workerParameterValue = "test";
    final AnyMap parameters = createJobParameters(workerParameterValue, TEST_STORE, OTHER_STORE);
    addJob(jobName, "testWorkflow", parameters);

    final String jobName2 = "testJob2";
    final AnyMap parameters2 = createJobParameters(workerParameterValue, OTHER_STORE, OTHER_STORE);
    addJob(jobName2, "testWorkflow", parameters2);

    final String jobName3 = "testJob3";
    final AnyMap parameters3 = createJobParameters(workerParameterValue, TEST_STORE, OTHER_STORE);
    addJob(jobName3, "testWorkflow", parameters3);

    final String jobId = startJob(jobName);
    final String jobId2 = startJob(jobName2);
    final String jobId3 = startJob(jobName3);
    final Map<String, String> jobRunMap = _jobRunEngine.getJobRunsUsingStore(TEST_STORE);
    assertEquals(jobId, jobRunMap.get(jobName));
    assertEquals(jobId3, jobRunMap.get(jobName3));
    assertEquals(2, jobRunMap.size());
    // OTHER_STORE also used in all jobs
    final Map<String, String> jobRunMapOther = _jobRunEngine.getJobRunsUsingStore(OTHER_STORE);
    assertEquals(jobId, jobRunMapOther.get(jobName));
    assertEquals(jobId2, jobRunMapOther.get(jobName2));
    assertEquals(jobId3, jobRunMapOther.get(jobName3));
    assertEquals(3, jobRunMapOther.size());
    _jobRunEngine.finishJob(jobName, jobId);
    _jobRunEngine.finishJob(jobName2, jobId2);
    _jobRunEngine.finishJob(jobName3, jobId3);
  }

  /**
   * three jobs defined (two using tested store) and all started. must deliver two jobs.
   */
  public void testThreeJobsAllStartedOneOfThemAlreadyFinishedOneCancelled() throws Exception {
    final String jobName = "testJob";
    final String workerParameterValue = "test";
    final AnyMap parameters = createJobParameters(workerParameterValue, TEST_STORE, OTHER_STORE);
    addJob(jobName, "testWorkflow", parameters);

    final String jobName2 = "testJob2";
    final AnyMap parameters2 = createJobParameters(workerParameterValue, OTHER_STORE, OTHER_STORE);
    addJob(jobName2, "testWorkflow", parameters2);

    final String jobName3 = "testJob3";
    final AnyMap parameters3 = createJobParameters(workerParameterValue, TEST_STORE, OTHER_STORE);
    addJob(jobName3, "testWorkflow", parameters3);

    final String jobId = startJob(jobName);
    final String jobId2 = startJob(jobName2);
    final String jobId3 = startJob(jobName3);

    _jobRunEngine.finishJob(jobName, jobId);
    _jobRunEngine.cancelJob(jobName2, jobId2);
    final Map<String, String> jobRunMapOther = _jobRunEngine.getJobRunsUsingStore(OTHER_STORE);
    assertEquals(jobId3, jobRunMapOther.get(jobName3));
    assertEquals(1, jobRunMapOther.size());
    _jobRunEngine.finishJob(jobName3, jobId3);
  }

  /**
   * three jobs defined (two using tested store) and all started. must deliver two jobs.
   */
  public void testThreeJobsAllStartedTwoRestarted() throws Exception {
    final String jobName = "testJob";
    final String workerParameterValue = "test";
    final AnyMap parameters = createJobParameters(workerParameterValue, TEST_STORE, OTHER_STORE);
    addJob(jobName, "testWorkflow", parameters);

    final String jobName2 = "testJob2";
    final AnyMap parameters2 = createJobParameters(workerParameterValue, OTHER_STORE, OTHER_STORE);
    addJob(jobName2, "testWorkflow", parameters2);

    final String jobName3 = "testJob3";
    final AnyMap parameters3 = createJobParameters(workerParameterValue, TEST_STORE, OTHER_STORE);
    addJob(jobName3, "testWorkflow", parameters3);

    String jobId = startJob(jobName);
    String jobId2 = startJob(jobName2);
    final String jobId3 = startJob(jobName3);

    _jobRunEngine.finishJob(jobName, jobId);
    jobId = _jobRunEngine.startJob(jobName);
    _jobRunEngine.cancelJob(jobName2, jobId2);
    jobId2 = _jobRunEngine.startJob(jobName2);
    final Map<String, String> jobRunMap = _jobRunEngine.getJobRunsUsingStore(TEST_STORE);
    assertEquals(jobId, jobRunMap.get(jobName));
    assertEquals(jobId3, jobRunMap.get(jobName3));
    assertEquals(2, jobRunMap.size());
    // OTHER_STORE also used in all jobs
    final Map<String, String> jobRunMapOther = _jobRunEngine.getJobRunsUsingStore(OTHER_STORE);
    assertEquals(jobId, jobRunMapOther.get(jobName));
    assertEquals(jobId2, jobRunMapOther.get(jobName2));
    assertEquals(jobId3, jobRunMapOther.get(jobName3));
    assertEquals(3, jobRunMapOther.size());
    _jobRunEngine.finishJob(jobName, jobId);
    _jobRunEngine.finishJob(jobName2, jobId2);
    _jobRunEngine.finishJob(jobName3, jobId3);
  }
}
