/*********************************************************************************************************************
 * Copyright (c) 2008, 2012 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
**********************************************************************************************************************/
package org.eclipse.smila.jobmanager.test;

import java.util.List;

import junit.framework.TestCase;

import org.eclipse.smila.common.definitions.ParameterDefinition;
import org.eclipse.smila.common.exceptions.InvalidDefinitionException;
import org.eclipse.smila.datamodel.Any.ValueType;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.DataFactory;

public class TestParameterDefinition extends TestCase {

  public void testParseSimpleDefinition() throws Exception {
    final AnyMap any = DataFactory.DEFAULT.createAnyMap();
    any.put("name", "simple");
    final ParameterDefinition param = new ParameterDefinition(any);
    assertParameter(param, "simple", ValueType.STRING, false, false, false, false);
  }

  public void testParseAnyParameter() throws Exception {
    final AnyMap any = DataFactory.DEFAULT.createAnyMap();
    any.put("name", "anyParam");
    any.put("type", "any");
    final ParameterDefinition param = new ParameterDefinition(any);
    assertParameter(param, "anyParam", null, false, false, false, false);
  }

  public void testParseStringParameter() throws Exception {
    final AnyMap any = DataFactory.DEFAULT.createAnyMap();
    any.put("name", "stringParam");
    any.put("type", "string");
    final ParameterDefinition param = new ParameterDefinition(any);
    assertParameter(param, "stringParam", ValueType.STRING, false, false, false, false);
  }

  public void testParseLongParameter() throws Exception {
    final AnyMap any = DataFactory.DEFAULT.createAnyMap();
    any.put("name", "longParam");
    any.put("type", "long");
    final ParameterDefinition param = new ParameterDefinition(any);
    assertParameter(param, "longParam", ValueType.LONG, false, false, false, false);
  }

  public void testParseOptionalDefinition() throws Exception {
    final AnyMap any = DataFactory.DEFAULT.createAnyMap();
    any.put("name", "optional");
    any.put("optional", true);
    ParameterDefinition param = new ParameterDefinition(any);
    assertParameter(param, "optional", ValueType.STRING, true, false, false, false);
    any.put("optional", "true");
    param = new ParameterDefinition(any);
    assertParameter(param, "optional", ValueType.STRING, true, false, false, false);
  }

  public void testParseMultiDefinition() throws Exception {
    final AnyMap any = DataFactory.DEFAULT.createAnyMap();
    any.put("name", "multi");
    any.put("multi", true);
    ParameterDefinition param = new ParameterDefinition(any);
    assertParameter(param, "multi", ValueType.STRING, false, true, false, false);
    any.put("multi", "true");
    param = new ParameterDefinition(any);
    assertParameter(param, "multi", ValueType.STRING, false, true, false, false);
  }

  public void testParseDefinitionWithValues() throws Exception {
    final AnyMap any = DataFactory.DEFAULT.createAnyMap();
    any.put("name", "valueParam");
    any.add("values", DataFactory.DEFAULT.createStringValue("value1"));
    any.add("values", DataFactory.DEFAULT.createStringValue("value2"));
    final ParameterDefinition param = new ParameterDefinition(any);
    assertParameter(param, "valueParam", ValueType.STRING, false, false, false, true);
    assertEquals("value1", param.getValues().get(0).asString());
    assertEquals("value2", param.getValues().get(1).asString());
  }

  public void testParseMapParameter() throws Exception {
    final AnyMap any = DataFactory.DEFAULT.createAnyMap();
    any.put("name", "mapParam");
    any.put("type", "map");
    AnyMap entry = DataFactory.DEFAULT.createAnyMap();
    entry.put("name", "entry1");
    any.add("entries", entry);
    entry = DataFactory.DEFAULT.createAnyMap();
    entry.put("name", "entry2");
    entry.put("type", "double");
    entry.put("optional", true);
    entry.put("multi", true);
    any.add("entries", entry);
    final ParameterDefinition param = new ParameterDefinition(any);
    assertParameter(param, "mapParam", ValueType.MAP, false, false, true, false);
    final List<ParameterDefinition> entries = param.getEntries();
    assertParameter(entries.get(0), "entry1", ValueType.STRING, false, false, false, false);
    assertParameter(entries.get(1), "entry2", ValueType.DOUBLE, true, true, false, false);
  }

  public void testMissingName() throws Exception {
    final AnyMap any = DataFactory.DEFAULT.createAnyMap();
    any.put("type", "string");
    try {
      new ParameterDefinition(any);
    } catch (final InvalidDefinitionException ex) {
      ;
    }
  }

  public void testInvalidTypeName() throws Exception {
    final AnyMap any = DataFactory.DEFAULT.createAnyMap();
    any.put("type", "unknown-type");
    assertInvalidConfigException(any);
  }

  public void testInvalidSequenceType() throws Exception {
    final AnyMap any = DataFactory.DEFAULT.createAnyMap();
    any.put("type", "seq");
    assertInvalidConfigException(any);
  }

  public void testInvalidOptionalFlag() throws Exception {
    final AnyMap any = DataFactory.DEFAULT.createAnyMap();
    any.put("optional", "true");
    assertInvalidConfigException(any);
  }

  public void testInvalidMultiFlag() throws Exception {
    final AnyMap any = DataFactory.DEFAULT.createAnyMap();
    any.put("multi", "true");
    assertInvalidConfigException(any);
  }

  public void testInvalidTypeForValues() throws Exception {
    final AnyMap any = DataFactory.DEFAULT.createAnyMap();
    any.add("values", DataFactory.DEFAULT.createStringValue("value1"));
    any.add("values", DataFactory.DEFAULT.createStringValue("value2"));
    any.put("type", "map");
    assertInvalidConfigException(any);
    any.put("type", "any");
    assertInvalidConfigException(any);
  }

  public void testInvalidValues() throws Exception {
    final AnyMap any = DataFactory.DEFAULT.createAnyMap();
    any.add("values", DataFactory.DEFAULT.createStringValue("value1"));
    any.add("values", DataFactory.DEFAULT.createStringValue("value2"));
    any.put("type", "long");
    assertInvalidConfigException(any);
  }

  public void testInvalidTypeForEntries() throws Exception {
    final AnyMap any = DataFactory.DEFAULT.createAnyMap();
    final AnyMap entry = DataFactory.DEFAULT.createAnyMap();
    entry.put("name", "entry1");
    any.add("entries", entry);
    any.put("type", "string");
    assertInvalidConfigException(any);
    any.put("type", "any");
    assertInvalidConfigException(any);
  }

  private void assertParameter(final ParameterDefinition param, final String name, final ValueType type,
    final boolean isOptional, final boolean isMulti, final boolean hasEntries, final boolean hasValues) {
    assertEquals(name, param.getName());
    assertEquals(type, param.getType());
    assertEquals(isOptional, param.isOptional());
    assertEquals(isMulti, param.isMulti());
    assertEquals(hasEntries, param.getEntries() != null);
    assertEquals(hasValues, param.getValues() != null);
  }

  private void assertInvalidConfigException(final AnyMap any) {
    any.put("name", "error");
    try {
      new ParameterDefinition(any);
    } catch (final InvalidDefinitionException ex) {
      ;
    }
  }
}
