/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Juergen Schumacher, Andreas Weber, Drazen Cindric, Andreas Schank (all Attensity Europe GmbH) - initial
 * implementation
 **********************************************************************************************************************/
package org.eclipse.smila.jobmanager.test;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.Map.Entry;

import junit.framework.TestCase;

import org.eclipse.smila.common.definitions.AccessAny;
import org.eclipse.smila.common.definitions.ParameterDefinition;
import org.eclipse.smila.common.exceptions.InvalidDefinitionException;
import org.eclipse.smila.datamodel.Any;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.AnySeq;
import org.eclipse.smila.jobmanager.definitions.WorkerDefinition;
import org.eclipse.smila.jobmanager.definitions.WorkerDefinition.Mode;
import org.eclipse.smila.taskworker.input.InputMode;
import org.eclipse.smila.taskworker.output.OutputMode;

/**
 * Tests for validation in worke definition parsing.
 * 
 */
public class TestWorkerDefinition extends TestCase {
  /**
   * Tests creating a worker with a missing name. must fail.
   * 
   * @throws Exception
   *           unhandled exception.
   */
  public void testMissingName() throws Exception {
    final AnyMap workerAny = AccessAny.FACTORY.createAnyMap();
    workerAny.put("worker-name", "worker");
    final AnySeq inputAny = AccessAny.FACTORY.createAnySeq();
    final AnyMap input1 = AccessAny.FACTORY.createAnyMap();
    input1.put("name", "insertedRecords");
    input1.put("type", "recordBulks");
    inputAny.add(input1);
    workerAny.put("input", inputAny);
    try {
      new WorkerDefinition(workerAny);
      fail("should not work");
    } catch (final Exception ex) {
      assertTrue("wrong exception caught: " + ex.toString(), ex instanceof InvalidDefinitionException);
    }
  }

  /**
   * Tests creating a worker with an invalid name. must fail.
   * 
   * @throws Exception
   *           unhandled exception.
   */
  public void testInvalidName() throws Exception {
    final AnyMap workerAny = AccessAny.FACTORY.createAnyMap();
    workerAny.put("name", "worker+/");
    final AnySeq inputAny = AccessAny.FACTORY.createAnySeq();
    final AnyMap input1 = AccessAny.FACTORY.createAnyMap();
    input1.put("name", "insertedRecords");
    input1.put("type", "recordBulks");
    inputAny.add(input1);
    workerAny.put("input", inputAny);
    try {
      new WorkerDefinition(workerAny);
      fail("should not work");
    } catch (final Exception ex) {
      assertTrue("wrong exception caught: " + ex.toString(), ex instanceof InvalidDefinitionException);
    }
  }

  /**
   * Tests missing input name. must fail.
   * 
   * @throws Exception
   *           unhandled exception.
   */
  public void testMissingInputName() throws Exception {
    final AnyMap workerAny = AccessAny.FACTORY.createAnyMap();
    workerAny.put("name", "worker");
    final AnySeq inputAny = AccessAny.FACTORY.createAnySeq();
    final AnyMap input1 = AccessAny.FACTORY.createAnyMap();
    input1.put("input-name", "insertedRecords");
    input1.put("type", "recordBulks");
    inputAny.add(input1);
    workerAny.put("input", inputAny);
    try {
      new WorkerDefinition(workerAny);
      fail("should not work");
    } catch (final Exception ex) {
      assertTrue("wrong exception caught: " + ex.toString(), ex instanceof InvalidDefinitionException);
    }
  }

  /**
   * test missing output name. must fail.
   * 
   * @throws Exception
   *           unhandled exception.
   */
  public void testMissingOutputName() throws Exception {
    final AnyMap workerAny = AccessAny.FACTORY.createAnyMap();
    workerAny.put("name", "worker");
    final AnySeq inputAny = AccessAny.FACTORY.createAnySeq();
    final AnyMap input1 = AccessAny.FACTORY.createAnyMap();
    input1.put("output-name", "insertedRecords");
    input1.put("type", "recordBulks");
    inputAny.add(input1);
    workerAny.put("output", inputAny);
    try {
      new WorkerDefinition(workerAny);
      fail("should not work");
    } catch (final Exception ex) {
      assertTrue("wrong exception caught: " + ex.toString(), ex instanceof InvalidDefinitionException);
    }
  }

  /**
   * tests wrong worker modes. must fail.
   * 
   * @throws Exception
   *           unhandled exception.
   */
  public void testWrongWorkerMode() throws Exception {
    final AnyMap workerAny = AccessAny.FACTORY.createAnyMap();
    workerAny.put("name", "worker");
    final AnySeq inputAny = AccessAny.FACTORY.createAnySeq();
    final AnyMap input1 = AccessAny.FACTORY.createAnyMap();
    input1.put("name", "insertedRecords");
    input1.put("type", "recordBulks");
    inputAny.add(input1);
    workerAny.put("input", inputAny);
    final AnySeq workerModesAny = AccessAny.FACTORY.createAnySeq();
    workerModesAny.add("appendable");
    workerAny.put("modes", workerModesAny);
    try {
      new WorkerDefinition(workerAny);
      fail("should not work");
    } catch (final Exception ex) {
      assertTrue("wrong exception caught: " + ex.toString(), ex instanceof InvalidDefinitionException);
    }
  }

  /**
   * test wrong input mode. must fail.
   * 
   * @throws Exception
   *           unhandled exception.
   */
  public void testWrongInputMode() throws Exception {
    final AnyMap workerAny = AccessAny.FACTORY.createAnyMap();
    workerAny.put("name", "worker");
    final AnySeq inputAny = AccessAny.FACTORY.createAnySeq();
    final AnyMap input1 = AccessAny.FACTORY.createAnyMap();
    input1.put("name", "insertedRecords");
    input1.put("type", "recordBulks");
    final AnySeq inputModesAny = AccessAny.FACTORY.createAnySeq();
    inputModesAny.add("quamifier");
    input1.put("modes", inputModesAny);
    inputAny.add(input1);
    workerAny.put("input", inputAny);
    try {
      new WorkerDefinition(workerAny);
      fail("should not work");
    } catch (final Exception ex) {
      assertTrue("wrong exception caught: " + ex.toString(), ex instanceof InvalidDefinitionException);
    }
  }

  /**
   * test wrong output mode. must fail.
   * 
   * @throws Exception
   *           unhandled exception.
   */
  public void testWrongOutputMode() throws Exception {
    final AnyMap workerAny = AccessAny.FACTORY.createAnyMap();
    workerAny.put("name", "worker");
    final AnySeq inputAny = AccessAny.FACTORY.createAnySeq();
    final AnyMap output1 = AccessAny.FACTORY.createAnyMap();
    output1.put("name", "insertedRecords");
    output1.put("type", "recordBulks");
    final AnySeq outputModesAny = AccessAny.FACTORY.createAnySeq();
    outputModesAny.add("apendable");
    output1.put("modes", outputModesAny);
    inputAny.add(output1);
    workerAny.put("output", inputAny);
    try {
      new WorkerDefinition(workerAny);
      fail("should not work");
    } catch (final Exception ex) {
      assertTrue("wrong exception caught: " + ex.toString(), ex instanceof InvalidDefinitionException);
    }
  }

  /**
   * tests all worker modes. must not fail.
   * 
   * @throws Exception
   *           unhandled exception.
   */
  public void testAllWorkerModes() throws Exception {
    final AnyMap workerAny = AccessAny.FACTORY.createAnyMap();
    workerAny.put("name", "worker");
    final AnySeq inputAny = AccessAny.FACTORY.createAnySeq();
    final AnyMap input1 = AccessAny.FACTORY.createAnyMap();
    input1.put("name", "insertedRecords");
    input1.put("type", "recordBulks");
    inputAny.add(input1);
    workerAny.put("input", inputAny);
    final AnySeq workerModesAny = AccessAny.FACTORY.createAnySeq();
    workerModesAny.add("bulkSource");
    workerModesAny.add("autoCommit");
    workerModesAny.add("runAlways");
    workerModesAny.add("requestsCompletion");
    workerAny.put("modes", workerModesAny);

    final WorkerDefinition worker = new WorkerDefinition(workerAny);
    final Collection<Mode> modes = worker.getModes();
    assertEquals(4, modes.size());
    assertTrue(modes.contains(Mode.AUTOCOMMIT));
    assertTrue(modes.contains(Mode.BULKSOURCE));
    assertTrue(modes.contains(Mode.RUNALWAYS));
    assertTrue(modes.contains(Mode.REQUESTSCOMPLETION));
  }

  /**
   * test all input modes. must not fail.
   * 
   * @throws Exception
   *           unhandled exception.
   */
  public void testAllInputModes() throws Exception {
    final AnyMap workerAny = AccessAny.FACTORY.createAnyMap();
    workerAny.put("name", "worker");
    final AnySeq inputAny = AccessAny.FACTORY.createAnySeq();
    final AnyMap input1 = AccessAny.FACTORY.createAnyMap();
    input1.put("name", "insertedRecords");
    input1.put("type", "recordBulks");
    final AnySeq inputModesAny = AccessAny.FACTORY.createAnySeq();
    inputModesAny.add("qualifier");
    input1.put("modes", inputModesAny);
    inputAny.add(input1);
    workerAny.put("input", inputAny);
    new WorkerDefinition(workerAny);
  }

  /**
   * test all output modes. must not fail.
   * 
   * @throws Exception
   *           unhandled exception.
   */
  public void testAllOutputModes() throws Exception {
    final AnyMap workerAny = AccessAny.FACTORY.createAnyMap();
    workerAny.put("name", "worker");
    final AnySeq inputAny = AccessAny.FACTORY.createAnySeq();
    final AnyMap output1 = AccessAny.FACTORY.createAnyMap();
    output1.put("name", "insertedRecords");
    output1.put("type", "recordBulks");
    final AnySeq outputModesAny = AccessAny.FACTORY.createAnySeq();
    outputModesAny.add("appendable");
    outputModesAny.add("optional");
    outputModesAny.add("multiple");
    output1.put("modes", outputModesAny);
    inputAny.add(output1);
    workerAny.put("output", inputAny);
    new WorkerDefinition(workerAny);
  }

  /**
   * test goups in input and output slots. must not fail.
   * 
   * @throws Exception
   *           unhandled exception.
   */
  public void testSlotGroups() throws Exception {
    final AnyMap workerAny = AccessAny.FACTORY.createAnyMap();
    workerAny.put("name", "worker");
    final AnySeq outputAny = AccessAny.FACTORY.createAnySeq();
    final AnyMap output1 = AccessAny.FACTORY.createAnyMap();
    output1.put("name", "insertedRecords");
    output1.put("type", "recordBulks");
    final AnySeq outputModesAny = AccessAny.FACTORY.createAnySeq();
    outputModesAny.add("appendable");
    outputModesAny.add("optional");
    output1.put("modes", outputModesAny);
    output1.put("group", "group1");
    outputAny.add(output1);

    final AnyMap output2 = AccessAny.FACTORY.createAnyMap();
    output2.put("name", "insertedRecords");
    output2.put("type", "recordBulks");
    final AnySeq outputModesAny2 = AccessAny.FACTORY.createAnySeq();
    outputModesAny2.add("appendable");
    outputModesAny2.add("optional");
    output2.put("modes", outputModesAny2);
    output2.put("group", "group2");
    outputAny.add(output2);

    workerAny.put("output", outputAny);

    final AnySeq inputAny = AccessAny.FACTORY.createAnySeq();
    final AnyMap input = AccessAny.FACTORY.createAnyMap();
    input.put("name", "insertedRecords");
    input.put("type", "recordBulks");
    inputAny.add(input);
    workerAny.put("input", inputAny);

    final WorkerDefinition wd = new WorkerDefinition(workerAny);
    final AnyMap reconvertedWorkerDef = wd.toAny();

    assertEquals(workerAny, reconvertedWorkerDef);
  }

  /**
   * tests working worker definition. must not fail.
   * 
   * @throws Exception
   *           unhandled exception.
   */
  public void testWorkingDefinition() throws Exception {
    final AnyMap workerAny = AccessAny.FACTORY.createAnyMap();
    workerAny.put("name", "worker");
    final AnySeq workerModesAny = AccessAny.FACTORY.createAnySeq();
    workerModesAny.add("autoCommit");
    workerAny.put("modes", workerModesAny);
    final AnySeq parametersAny = AccessAny.FACTORY.createAnySeq();
    final AnyMap indexParam = AccessAny.FACTORY.createAnyMap();
    indexParam.put("name", "index");
    parametersAny.add(indexParam);
    final AnyMap mergeParam = AccessAny.FACTORY.createAnyMap();
    mergeParam.put("name", "mergeParameters");
    parametersAny.add(mergeParam);
    workerAny.put("parameters", parametersAny);

    final AnySeq inputAny = AccessAny.FACTORY.createAnySeq();
    final AnyMap input1 = AccessAny.FACTORY.createAnyMap();
    input1.put("name", "insertedRecords");
    input1.put("type", "recordBulks");
    final AnySeq inputModesAny = AccessAny.FACTORY.createAnySeq();
    inputModesAny.add("qualifier");
    input1.put("modes", inputModesAny);
    inputAny.add(input1);
    workerAny.put("input", inputAny);

    final AnySeq outputAny = AccessAny.FACTORY.createAnySeq();
    final AnyMap output1 = AccessAny.FACTORY.createAnyMap();
    output1.put("name", "deletedRecords");
    output1.put("type", "indexDeletes");
    final AnySeq outputModesAny = AccessAny.FACTORY.createAnySeq();
    outputModesAny.add("appendable");
    outputModesAny.add("optional");
    output1.put("modes", outputModesAny);
    outputAny.add(output1);
    workerAny.put("output", outputAny);

    final WorkerDefinition wdef = new WorkerDefinition(workerAny);
    assertEquals("name does not match", workerAny.getStringValue("name"), wdef.getName());
    assertTrue("modes do not match", wdef.getModes().contains(WorkerDefinition.Mode.AUTOCOMMIT));
    assertTrue("parameters do not match", wdef.getMandatoryParameterNames().contains("index"));
    assertTrue("parameters do not match", wdef.getMandatoryParameterNames().contains("mergeParameters"));
    assertEquals("input name does not match", workerAny.getSeq("input").getMap(0).getStringValue("name"), wdef
      .getInput().toArray(new WorkerDefinition.Input[1])[0].getName());
    assertEquals("input type does not match", workerAny.getSeq("input").getMap(0).getStringValue("type"), wdef
      .getInput().toArray(new WorkerDefinition.Input[1])[0].getType());
    assertTrue("input modes do not match", wdef.getInput().toArray(new WorkerDefinition.Input[1])[0].getModes()
      .contains(InputMode.QUALIFIER));
    assertEquals("output name does not match", workerAny.getSeq("output").getMap(0).getStringValue("name"), wdef
      .getOutput().toArray(new WorkerDefinition.Output[1])[0].getName());
    assertEquals("output type does not match", workerAny.getSeq("output").getMap(0).getStringValue("type"), wdef
      .getOutput().toArray(new WorkerDefinition.Output[1])[0].getType());
    assertTrue("output modes do not match", wdef.getOutput().toArray(new WorkerDefinition.Output[1])[0].getModes()
      .contains(OutputMode.APPENDABLE));
    assertTrue("output modes do not match", wdef.getOutput().toArray(new WorkerDefinition.Output[1])[0].getModes()
      .contains(OutputMode.OPTIONAL));

  }

  /**
   * tests working worker definition with optional parameter. must not fail.
   * 
   * @throws Exception
   *           unhandled exception.
   */
  public void testOptionalParametersWorkingDefinition() throws Exception {
    final AnyMap workerAny = AccessAny.FACTORY.createAnyMap();
    workerAny.put("name", "worker");
    final AnySeq workerModesAny = AccessAny.FACTORY.createAnySeq();
    workerModesAny.add("autoCommit");
    workerAny.put("modes", workerModesAny);
    final AnySeq parametersAny = AccessAny.FACTORY.createAnySeq();
    final AnyMap indexParam = AccessAny.FACTORY.createAnyMap();
    indexParam.put("name", "index");
    parametersAny.add(indexParam);
    final AnyMap mergeParam = AccessAny.FACTORY.createAnyMap();
    mergeParam.put("name", "mergeParameters");
    parametersAny.add(mergeParam);
    final AnyMap optionalParam = AccessAny.FACTORY.createAnyMap();
    optionalParam.put("name", "optionalParameter");
    optionalParam.put("optional", true);
    parametersAny.add(optionalParam);
    workerAny.put("parameters", parametersAny);

    final AnySeq inputAny = AccessAny.FACTORY.createAnySeq();
    final AnyMap input1 = AccessAny.FACTORY.createAnyMap();
    input1.put("name", "insertedRecords");
    input1.put("type", "recordBulks");
    final AnySeq inputModesAny = AccessAny.FACTORY.createAnySeq();
    inputModesAny.add("qualifier");
    input1.put("modes", inputModesAny);
    inputAny.add(input1);
    workerAny.put("input", inputAny);

    final AnySeq outputAny = AccessAny.FACTORY.createAnySeq();
    final AnyMap output1 = AccessAny.FACTORY.createAnyMap();
    output1.put("name", "deletedRecords");
    output1.put("type", "indexDeletes");
    final AnySeq outputModesAny = AccessAny.FACTORY.createAnySeq();
    outputModesAny.add("appendable");
    outputModesAny.add("optional");
    output1.put("modes", outputModesAny);
    outputAny.add(output1);
    workerAny.put("output", outputAny);

    final WorkerDefinition wdef = new WorkerDefinition(workerAny);
    assertEquals("name does not match", workerAny.getStringValue("name"), wdef.getName());
    assertTrue("modes do not match", wdef.getModes().contains(WorkerDefinition.Mode.AUTOCOMMIT));
    assertTrue("parameters do not match", wdef.getMandatoryParameterNames().contains("index"));
    assertTrue("parameters do not match", wdef.getMandatoryParameterNames().contains("mergeParameters"));
    assertFalse("optional parameter listed as mandatory",
      wdef.getMandatoryParameterNames().contains("optionalParameter"));
    final Collection<String> paramNames = new ArrayList<String>();
    for (final ParameterDefinition paramEntry : wdef.getParameters()) {
      paramNames.add(paramEntry.getName());
    }
    assertEquals(3, paramNames.size());
    assertTrue("parameter not found", paramNames.contains("index"));
    assertTrue("parameter not found", paramNames.contains("mergeParameters"));
    assertTrue("optional parameter not found", paramNames.contains("optionalParameter"));
    assertEquals("input name does not match", workerAny.getSeq("input").getMap(0).getStringValue("name"), wdef
      .getInput().toArray(new WorkerDefinition.Input[1])[0].getName());
    assertEquals("input type does not match", workerAny.getSeq("input").getMap(0).getStringValue("type"), wdef
      .getInput().toArray(new WorkerDefinition.Input[1])[0].getType());
    assertTrue("input modes do not match", wdef.getInput().toArray(new WorkerDefinition.Input[1])[0].getModes()
      .contains(InputMode.QUALIFIER));
    assertEquals("output name does not match", workerAny.getSeq("output").getMap(0).getStringValue("name"), wdef
      .getOutput().toArray(new WorkerDefinition.Output[1])[0].getName());
    assertEquals("output type does not match", workerAny.getSeq("output").getMap(0).getStringValue("type"), wdef
      .getOutput().toArray(new WorkerDefinition.Output[1])[0].getType());
    assertTrue("output modes do not match", wdef.getOutput().toArray(new WorkerDefinition.Output[1])[0].getModes()
      .contains(OutputMode.APPENDABLE));
    assertTrue("output modes do not match", wdef.getOutput().toArray(new WorkerDefinition.Output[1])[0].getModes()
      .contains(OutputMode.OPTIONAL));
  }

  /**
   * tests worker definition with invalid parameters. must not fail.
   * 
   * @throws Exception
   *           unhandled exception.
   */
  public void testInvalidParametersWorkingDefinition() throws Exception {
    final AnyMap workerAny = AccessAny.FACTORY.createAnyMap();
    workerAny.put("name", "worker");
    final AnySeq workerModesAny = AccessAny.FACTORY.createAnySeq();
    workerModesAny.add("autoCommit");
    workerAny.put("modes", workerModesAny);
    final AnySeq parametersAny = AccessAny.FACTORY.createAnySeq();
    final AnyMap indexParam = AccessAny.FACTORY.createAnyMap();
    indexParam.put("name", "index");
    parametersAny.add(indexParam);
    final AnyMap mergeParam = AccessAny.FACTORY.createAnyMap();
    mergeParam.put("name", "mergeParameters");
    parametersAny.add(mergeParam);
    final AnyMap optionalParam = AccessAny.FACTORY.createAnyMap();
    // wrong parameter.., missing "name" attribute
    optionalParam.put("noname", "optionalParameter");
    optionalParam.put("optional", true);
    parametersAny.add(optionalParam);
    workerAny.put("parameters", parametersAny);

    final AnySeq inputAny = AccessAny.FACTORY.createAnySeq();
    final AnyMap input1 = AccessAny.FACTORY.createAnyMap();
    input1.put("name", "insertedRecords");
    input1.put("type", "recordBulks");
    final AnySeq inputModesAny = AccessAny.FACTORY.createAnySeq();
    inputModesAny.add("qualifier");
    input1.put("modes", inputModesAny);
    inputAny.add(input1);
    workerAny.put("input", inputAny);

    final AnySeq outputAny = AccessAny.FACTORY.createAnySeq();
    final AnyMap output1 = AccessAny.FACTORY.createAnyMap();
    output1.put("name", "deletedRecords");
    output1.put("type", "indexDeletes");
    final AnySeq outputModesAny = AccessAny.FACTORY.createAnySeq();
    outputModesAny.add("appendable");
    outputModesAny.add("optional");
    output1.put("modes", outputModesAny);
    outputAny.add(output1);
    workerAny.put("output", outputAny);

    WorkerDefinition wdef;
    try {
      wdef = new WorkerDefinition(workerAny);
      fail("invalid definituion, exceptione xpected.");
    } catch (final InvalidDefinitionException e) {
      ;// ok
    }
    // now correct name again
    optionalParam.put("name", "optionalParameter");
    // but invalid new parameter (no map)
    parametersAny.add("string");
    try {
      wdef = new WorkerDefinition(workerAny);
      fail("invalid definituion, exceptione xpected.");
    } catch (final InvalidDefinitionException e) {
      ;// ok
    }
    parametersAny.clear();
    parametersAny.add(indexParam);
    parametersAny.add(mergeParam);
    parametersAny.add(optionalParam);
    wdef = new WorkerDefinition(workerAny);
    assertEquals("name does not match", workerAny.getStringValue("name"), wdef.getName());
    assertTrue("modes do not match", wdef.getModes().contains(WorkerDefinition.Mode.AUTOCOMMIT));
    assertTrue("parameters do not match", wdef.getMandatoryParameterNames().contains("index"));
    assertTrue("parameters do not match", wdef.getMandatoryParameterNames().contains("mergeParameters"));
    assertFalse("optional parameter listed as mandatory",
      wdef.getMandatoryParameterNames().contains("optionalParameter"));
    final Collection<String> paramNames = new ArrayList<String>();
    for (final ParameterDefinition paramEntry : wdef.getParameters()) {
      paramNames.add(paramEntry.getName());
    }
    assertEquals(3, paramNames.size());
    assertTrue("parameter not found", paramNames.contains("index"));
    assertTrue("parameter not found", paramNames.contains("mergeParameters"));
    assertTrue("optional parameter not found", paramNames.contains("optionalParameter"));
    assertEquals("input name does not match", workerAny.getSeq("input").getMap(0).getStringValue("name"), wdef
      .getInput().toArray(new WorkerDefinition.Input[1])[0].getName());
    assertEquals("input type does not match", workerAny.getSeq("input").getMap(0).getStringValue("type"), wdef
      .getInput().toArray(new WorkerDefinition.Input[1])[0].getType());
    assertTrue("input modes do not match", wdef.getInput().toArray(new WorkerDefinition.Input[1])[0].getModes()
      .contains(InputMode.QUALIFIER));
    assertEquals("output name does not match", workerAny.getSeq("output").getMap(0).getStringValue("name"), wdef
      .getOutput().toArray(new WorkerDefinition.Output[1])[0].getName());
    assertEquals("output type does not match", workerAny.getSeq("output").getMap(0).getStringValue("type"), wdef
      .getOutput().toArray(new WorkerDefinition.Output[1])[0].getType());
    assertTrue("output modes do not match", wdef.getOutput().toArray(new WorkerDefinition.Output[1])[0].getModes()
      .contains(OutputMode.APPENDABLE));
    assertTrue("output modes do not match", wdef.getOutput().toArray(new WorkerDefinition.Output[1])[0].getModes()
      .contains(OutputMode.OPTIONAL));

  }

  /**
   * tests working worker definition conversion from Any and to Any. must not fail.
   * 
   * @throws Exception
   *           unhandled exception.
   */
  public void testWorkingDefinitionFromAndToAny() throws Exception {
    final AnyMap workerAny = AccessAny.FACTORY.createAnyMap();
    workerAny.put("name", "worker");
    final AnySeq workerModesAny = AccessAny.FACTORY.createAnySeq();
    workerModesAny.add("autoCommit");
    workerModesAny.add("deleteInput");
    final AnySeq parametersAny = AccessAny.FACTORY.createAnySeq();
    final AnyMap indexParam = AccessAny.FACTORY.createAnyMap();
    indexParam.put("name", "index");
    parametersAny.add(indexParam);
    final AnyMap mergeParam = AccessAny.FACTORY.createAnyMap();
    mergeParam.put("name", "mergeParameters");
    parametersAny.add(mergeParam);
    workerAny.put("parameters", parametersAny);

    final AnySeq inputAny = AccessAny.FACTORY.createAnySeq();
    final AnyMap input1 = AccessAny.FACTORY.createAnyMap();
    input1.put("name", "insertedRecords");
    input1.put("type", "recordBulks");
    final AnySeq inputModesAny = AccessAny.FACTORY.createAnySeq();
    inputModesAny.add("qualifier");
    input1.put("modes", inputModesAny);
    inputAny.add(input1);
    workerAny.put("input", inputAny);

    final AnySeq outputAny = AccessAny.FACTORY.createAnySeq();
    final AnyMap output1 = AccessAny.FACTORY.createAnyMap();
    output1.put("name", "deletedRecords");
    output1.put("type", "indexDeletes");
    final AnySeq outputModesAny = AccessAny.FACTORY.createAnySeq();
    outputModesAny.add("appendable");
    outputModesAny.add("optional");
    output1.put("modes", outputModesAny);
    outputAny.add(output1);
    workerAny.put("output", outputAny);

    final WorkerDefinition workerDefinition = new WorkerDefinition(workerAny);
    final AnyMap reengineeredAny = workerDefinition.toAny();

    assertTrue("Worker Definition Any objects differ.", checkAnies(workerAny, reengineeredAny));
  }

  /**
   * tests worker parameter ranges.
   */
  public void testWorkerParameterRanges() throws Exception {
    final AnyMap workerAny = AccessAny.FACTORY.createAnyMap();
    workerAny.put("name", "worker");
    final AnySeq parametersAny = AccessAny.FACTORY.createAnySeq();
    final AnyMap indexParam = AccessAny.FACTORY.createAnyMap();
    indexParam.put("name", "index");
    indexParam.put("range", "indexName");
    parametersAny.add(indexParam);
    final AnyMap jobParam = AccessAny.FACTORY.createAnyMap();
    jobParam.put("name", "job");
    jobParam.put("range", "jobName");
    parametersAny.add(jobParam);
    final AnyMap otherParam = AccessAny.FACTORY.createAnyMap();
    otherParam.put("name", "otherParameter");
    parametersAny.add(otherParam);
    workerAny.put("parameters", parametersAny);

    final WorkerDefinition wDef = new WorkerDefinition(workerAny);

    List<ParameterDefinition> noRangeParam = wDef.getParametersbyRange(null);
    assertEquals(1, noRangeParam.size());
    assertEquals(otherParam, noRangeParam.get(0).toAny(true));
    noRangeParam = wDef.getParametersbyRange("");
    assertEquals(1, noRangeParam.size());
    assertEquals(otherParam, noRangeParam.get(0).toAny(true));
    final List<ParameterDefinition> jobNameParam = wDef.getParametersbyRange(jobParam.getStringValue("range"));
    assertEquals(1, jobNameParam.size());
    assertEquals(jobParam, jobNameParam.get(0).toAny(true));
    final List<ParameterDefinition> indexNameParam = wDef.getParametersbyRange(indexParam.getStringValue("range"));
    assertEquals(1, indexNameParam.size());
    assertEquals(indexParam, indexNameParam.get(0).toAny(true));
    assertTrue(wDef.getParametersbyRange("noRange").isEmpty());
    final List<ParameterDefinition> allPar = wDef.getParameters();
    assertTrue(allPar.containsAll(noRangeParam));
    assertTrue(allPar.containsAll(jobNameParam));
    assertTrue(allPar.containsAll(indexNameParam));
  }

  /**
   * Utility method to check two Any objects for equity.
   * 
   * @param expected
   *          Expected Any.
   * @param actual
   *          The Any to check.
   * @return true, if the Anies are equal, false if not (in most cases the assertion will prevent this method to return
   *         false...)
   * @throws Exception
   *           Exceptions when accessing the Any objects.
   */
  private boolean checkAnies(final org.eclipse.smila.datamodel.Any expected, final Any actual) throws Exception {
    if (expected == actual) {
      return true;
    }
    assertFalse("one is null, the other not.", expected == null || actual == null);
    assertEquals("discriminators differ for any " + expected.toString(), expected.getValueType(),
      actual.getValueType());
    assertEquals("different any sizes.", expected.size(), actual.size());

    if (expected.isSeq()) {
      final AnySeq expectedSeq = (AnySeq) expected;
      final AnySeq actualSeq = (AnySeq) actual;

      for (int i = 0; i < expectedSeq.size(); i++) {
        assertTrue("List entry " + expected.toString() + " differs.",
          checkAnies(expectedSeq.get(i), actualSeq.get(i)));
      }
      return true;
    } else if (expected.isMap()) {
      final AnyMap expectedMap = (AnyMap) expected;
      final AnyMap actualMap = (AnyMap) actual;
      final Iterator<Entry<String, Any>> iterator = expectedMap.entrySet().iterator();
      while (iterator.hasNext()) {
        final Entry<String, Any> entry = iterator.next();
        assertTrue("Map entry " + entry.getKey() + " differs.",
          checkAnies(entry.getValue(), actualMap.get(entry.getKey())));
      }
      return true;
    } else {
      return expected.equals(actual);
    }
  }
}
