/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Juergen Schumacher, Andreas Weber, Drazen Cindric, Andreas Schank (all Attensity Europe GmbH) - initial
 * implementation
 **********************************************************************************************************************/
package org.eclipse.smila.jobmanager.test;

import java.util.Arrays;

import junit.framework.TestCase;

import org.eclipse.smila.common.definitions.AccessAny;
import org.eclipse.smila.common.exceptions.InvalidDefinitionException;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.AnySeq;
import org.eclipse.smila.jobmanager.definitions.JobRunMode;
import org.eclipse.smila.jobmanager.definitions.WorkflowDefinition;
import org.eclipse.smila.jobmanager.exceptions.InvalidConfigException;

/**
 * Tests for validation in job parsing.
 * 
 */
public class TestWorkflowDefinition extends TestCase {
  /**
   * Tests workflow with missing name. Must fail.
   * 
   * @throws Exception
   *           Any exception that happens.
   */
  public void testMissingName() throws Exception {
    final AnyMap workflowAny = AccessAny.FACTORY.createAnyMap();
    workflowAny.put("workflow-name", "job");
    final AnyMap parametersAny = AccessAny.FACTORY.createAnyMap();
    parametersAny.put("indexName", "index");
    workflowAny.put("parameters", parametersAny);
    final AnyMap startActionAny = AccessAny.FACTORY.createAnyMap();
    startActionAny.put("worker", "input-worker");
    final AnyMap inputParametersAny = AccessAny.FACTORY.createAnyMap();
    inputParametersAny.put("index", "${indexName}");
    startActionAny.put("parameters", inputParametersAny);
    final AnyMap outputAny = AccessAny.FACTORY.createAnyMap();
    outputAny.put("output", "insertBucket");
    startActionAny.put("output", outputAny);
    workflowAny.put("startAction", startActionAny);
    try {
      new WorkflowDefinition(workflowAny);
      fail("should not work");
    } catch (final Exception ex) {
      assertTrue("wrong exception caught: " + ex.toString(), ex instanceof InvalidDefinitionException);
    }
  }

  /**
   * Tests workflow with invalid name. Must fail.
   * 
   * @throws Exception
   *           Any exception that happens.
   */
  public void testInvalidName() throws Exception {
    final AnyMap workflowAny = AccessAny.FACTORY.createAnyMap();
    workflowAny.put("name", "+job/");
    final AnyMap parametersAny = AccessAny.FACTORY.createAnyMap();
    parametersAny.put("indexName", "index");
    workflowAny.put("parameters", parametersAny);
    final AnyMap startActionAny = AccessAny.FACTORY.createAnyMap();
    startActionAny.put("worker", "input-worker");
    final AnyMap inputParametersAny = AccessAny.FACTORY.createAnyMap();
    inputParametersAny.put("index", "${indexName}");
    startActionAny.put("parameters", inputParametersAny);
    final AnyMap outputAny = AccessAny.FACTORY.createAnyMap();
    outputAny.put("output", "insertBucket");
    startActionAny.put("output", outputAny);
    workflowAny.put("startAction", startActionAny);
    try {
      new WorkflowDefinition(workflowAny);
      fail("should not work");
    } catch (final Exception ex) {
      assertTrue("wrong exception caught: " + ex.toString(), ex instanceof InvalidDefinitionException);
    }
  }

  /**
   * Tests workflow with missing startAction.
   * 
   * @throws Exception
   *           Exception while trying to access jobmanager or creating objects.
   */
  public void testMissingStartAction() throws Exception {
    /*
     * { "name": "testMissingStartAction", "actions": [ { "worker": "input-worker", "parameters": { "index":
     * "${indexName}" }, "output": { "output": "insertBucket", "deletedRecords": "indexDeletesBucket" } } ] }
     */
    final AnyMap workflowAny = AccessAny.FACTORY.createAnyMap();
    workflowAny.put("name", "testMissingStartAction");
    final AnySeq actionsListAny = AccessAny.FACTORY.createAnySeq();
    final AnyMap actionAny = AccessAny.FACTORY.createAnyMap();
    actionAny.put("worker", "input-worker");
    final AnyMap parametersAny = AccessAny.FACTORY.createAnyMap();
    parametersAny.put("index", "index");
    actionAny.put("parameters", parametersAny);
    final AnyMap outputAny = AccessAny.FACTORY.createAnyMap();
    outputAny.put("output", "insertBucket");
    actionAny.put("output", outputAny);
    actionsListAny.add(actionAny);
    workflowAny.put("actions", actionsListAny);
    try {
      new WorkflowDefinition(workflowAny);
      fail("should fail.");
    } catch (final Exception ex) {
      assertTrue("Wrong exception caught: " + ex, ex instanceof InvalidConfigException);
    }
  }

  /**
   * Tests missing worker name in startAction.
   * 
   * @throws Exception
   *           Exception when converting to or from Any.
   */
  public void testMissingWorkerNameInStartAction() throws Exception {
    final AnyMap workflowAny = AccessAny.FACTORY.createAnyMap();
    workflowAny.put("name", "testMissingWorkerNameInStartAction");
    final AnyMap parametersAny = AccessAny.FACTORY.createAnyMap();
    parametersAny.put("indexName", "index");
    workflowAny.put("parameters", parametersAny);

    final AnyMap startActionAny = AccessAny.FACTORY.createAnyMap();
    final AnyMap inputParametersAny = AccessAny.FACTORY.createAnyMap();
    inputParametersAny.put("index", "${indexName}");
    startActionAny.put("parameters", inputParametersAny);
    final AnyMap outputAny = AccessAny.FACTORY.createAnyMap();
    outputAny.put("output", "transBucket");
    startActionAny.put("output", outputAny);
    workflowAny.put("startAction", startActionAny);

    final AnySeq actionsAny = AccessAny.FACTORY.createAnySeq();
    // intermediate worker
    final AnyMap intermediateActionAny = AccessAny.FACTORY.createAnyMap();
    intermediateActionAny.put("worker", "intermediate-worker");
    final AnyMap intermediateInputAny = AccessAny.FACTORY.createAnyMap();
    intermediateInputAny.put("input", "transBucket");
    intermediateActionAny.put("input", intermediateInputAny);
    final AnyMap intermediateOutputAny = AccessAny.FACTORY.createAnyMap();
    intermediateOutputAny.put("output", "processedBucket");
    intermediateActionAny.put("output", intermediateOutputAny);
    actionsAny.add(intermediateActionAny);

    // final worker
    final AnyMap finalActionAny = AccessAny.FACTORY.createAnyMap();
    finalActionAny.put("worker", "final-worker");
    final AnyMap finalInputAny = AccessAny.FACTORY.createAnyMap();
    finalInputAny.put("input", "processedBucket");
    finalActionAny.put("input", finalInputAny);
    final AnyMap finalOutputAny = AccessAny.FACTORY.createAnyMap();
    finalOutputAny.put("output", "finalBucket");
    finalActionAny.put("output", finalOutputAny);
    actionsAny.add(finalActionAny);

    workflowAny.put("actions", actionsAny);

    try {
      new WorkflowDefinition(workflowAny);
      fail("missing worker in startAction should prevent the WorkflowDefinition from being constructed.");
    } catch (final Exception e) {
      assertTrue("wrong exception caught: " + e.toString(), e instanceof InvalidDefinitionException);
    }
  }

  /**
   * Tests missing worker name in actions.
   * 
   * @throws Exception
   *           Exception when converting to or from Any.
   */
  public void testMissingWorkerNameInActions() throws Exception {
    final AnyMap workflowAny = AccessAny.FACTORY.createAnyMap();
    workflowAny.put("name", "testMissingWorkerNameInActions");
    final AnyMap parametersAny = AccessAny.FACTORY.createAnyMap();
    parametersAny.put("indexName", "index");
    workflowAny.put("parameters", parametersAny);

    final AnyMap startActionAny = AccessAny.FACTORY.createAnyMap();
    startActionAny.put("worker", "input-worker");
    final AnyMap inputParametersAny = AccessAny.FACTORY.createAnyMap();
    inputParametersAny.put("index", "${indexName}");
    startActionAny.put("parameters", inputParametersAny);
    final AnyMap outputAny = AccessAny.FACTORY.createAnyMap();
    outputAny.put("output", "transBucket");
    startActionAny.put("output", outputAny);
    workflowAny.put("startAction", startActionAny);

    final AnySeq actionsAny = AccessAny.FACTORY.createAnySeq();
    // intermediate worker
    final AnyMap intermediateActionAny = AccessAny.FACTORY.createAnyMap();
    intermediateActionAny.put("worker", "intermediate-worker");
    final AnyMap intermediateInputAny = AccessAny.FACTORY.createAnyMap();
    intermediateInputAny.put("input", "transBucket");
    intermediateActionAny.put("input", intermediateInputAny);
    final AnyMap intermediateOutputAny = AccessAny.FACTORY.createAnyMap();
    intermediateOutputAny.put("output", "processedBucket");
    intermediateActionAny.put("output", intermediateOutputAny);
    actionsAny.add(intermediateActionAny);

    // final worker
    final AnyMap finalActionAny = AccessAny.FACTORY.createAnyMap();
    // this one is missing: finalActionAny.put("worker", "final-worker");
    final AnyMap finalInputAny = AccessAny.FACTORY.createAnyMap();
    finalInputAny.put("input", "processedBucket");
    finalActionAny.put("input", finalInputAny);
    final AnyMap finalOutputAny = AccessAny.FACTORY.createAnyMap();
    finalOutputAny.put("output", "finalBucket");
    finalActionAny.put("output", finalOutputAny);
    actionsAny.add(finalActionAny);

    workflowAny.put("actions", actionsAny);

    try {
      new WorkflowDefinition(workflowAny);
      fail("missing worker in actions section should prevent the WorkflowDefinition from being constructed.");
    } catch (final Exception e) {
      assertTrue("wrong exception caught: " + e.toString(), e instanceof InvalidDefinitionException);
    }
  }

  /**
   * Tests working workflow.
   * 
   * @throws Exception
   *           Exception while trying to access jobmanager or creating objects.
   */
  public void testWorkingDefinition() throws Exception {
    final AnyMap workflowAny = AccessAny.FACTORY.createAnyMap();
    workflowAny.put("name", "testWorkingDefinition");
    final AnyMap parametersAny = AccessAny.FACTORY.createAnyMap();
    parametersAny.put("indexName", "index");
    workflowAny.put("parameters", parametersAny);

    final AnyMap startActionAny = AccessAny.FACTORY.createAnyMap();
    startActionAny.put("worker", "input-worker");
    final AnyMap inputParametersAny = AccessAny.FACTORY.createAnyMap();
    inputParametersAny.put("index", "${indexName}");
    startActionAny.put("parameters", inputParametersAny);
    final AnyMap outputAny = AccessAny.FACTORY.createAnyMap();
    outputAny.put("output", "insertBucket");

    startActionAny.put("output", outputAny);
    workflowAny.put("startAction", startActionAny);
    final WorkflowDefinition workflowDefinition = new WorkflowDefinition(workflowAny);
    assertNotNull(workflowDefinition);
    assertEquals("name is different", workflowAny.getStringValue("name"), workflowDefinition.getName());
    assertEquals("parameters are different", workflowAny.getMap("parameters").getStringValue("indexName"),
      workflowDefinition.getParameters().getStringValue("indexName"));
    assertEquals("worker name is different", workflowAny.getMap("startAction").getStringValue("worker"),
      workflowDefinition.getStartAction().getWorker());
    assertEquals("startAction parameter is different", workflowAny.getMap("startAction").getMap("parameters")
      .getStringValue("index"), workflowDefinition.getStartAction().getParameters().getStringValue("index"));
    assertEquals("startAction output bucket is different", workflowAny.getMap("startAction").getMap("output")
      .getStringValue("output"), workflowDefinition.getStartAction().getOutput().get("output"));
    assertEquals("startAction output bucket is different", workflowAny.getMap("startAction").getMap("output")
      .getStringValue("deletedRecords"), workflowDefinition.getStartAction().getOutput().get("deletedRecords"));
    assertNull(workflowDefinition.getJobRunModes());
    assertNull(workflowDefinition.getDefaultJobRunMode());
  }

  /**
   * Tests the conversion to and from Any.
   * 
   * @throws Exception
   *           Exception when converting to or from Any.
   */
  public void testWorkingDefinitionToAndFromAny() throws Exception {
    final AnyMap workflowAny = AccessAny.FACTORY.createAnyMap();
    workflowAny.put("name", "testWorkingDefinitionToAndFromAny");
    final AnySeq modesAny = workflowAny.getSeq("modes", true);
    modesAny.add("runOnce");
    modesAny.add("standard");
    final AnyMap parametersAny = workflowAny.getMap("parameters", true);
    parametersAny.put("indexName", "index");

    final AnyMap startActionAny = workflowAny.getMap("startAction", true);
    startActionAny.put("worker", "input-worker");
    final AnyMap inputParametersAny = startActionAny.getMap("parameters", true);
    inputParametersAny.put("index", "${indexName}");

    final AnyMap outputAny = startActionAny.getMap("output", true);
    outputAny.put("output", "transBucket");

    workflowAny.put("additionalInfo1", "additional info1");
    workflowAny.put("additionalInfo2", "additional info2");

    final AnySeq actionsAny = workflowAny.getSeq("actions", true);
    // intermediate worker
    final AnyMap intermediateActionAny = AccessAny.FACTORY.createAnyMap();
    actionsAny.add(intermediateActionAny);
    intermediateActionAny.put("worker", "intermediate-worker");
    final AnyMap intermediateInputAny = intermediateActionAny.getMap("input", true);
    intermediateInputAny.put("input", "transBucket");

    final AnyMap intermediateOutputAny = AccessAny.FACTORY.createAnyMap();
    intermediateOutputAny.put("output", "processedBucket");
    intermediateActionAny.put("output", intermediateOutputAny);

    // final worker
    final AnyMap finalActionAny = AccessAny.FACTORY.createAnyMap();
    actionsAny.add(finalActionAny);
    finalActionAny.put("worker", "final-worker");
    final AnyMap finalInputAny = finalActionAny.getMap("input", true);
    finalInputAny.put("input", "processedBucket");
    finalActionAny.put("input", finalInputAny);
    final AnyMap finalOutputAny = finalActionAny.getMap("output", true);
    finalOutputAny.put("output", "finalBucket");

    final WorkflowDefinition workflowDefinition = new WorkflowDefinition(workflowAny);
    assertNotNull(workflowDefinition);
    final AnyMap reEngineeredAny = workflowDefinition.toAny(true);
    assertEquals("Input and output Any objects do not match", workflowAny, reEngineeredAny);

    workflowAny.remove("additionalInfo1");
    workflowAny.remove("additionalInfo2");
    final AnyMap anyWithoutDetails = workflowDefinition.toAny(false);
    assertEquals("Input and output Any objects do not match", workflowAny, anyWithoutDetails);
  }

  /** try parsing a workflow with a single allowed job run mode. */
  public void testSetJobRunMode() throws Exception {
    final AnyMap workflowAny = AccessAny.FACTORY.createAnyMap();
    workflowAny.put("name", "testSetJobRunMode");
    final AnyMap startActionAny = workflowAny.getMap("startAction", true);
    startActionAny.put("worker", "input-worker");
    final AnySeq modesAny = workflowAny.getSeq("modes", true);
    modesAny.add("runOnce");

    final WorkflowDefinition workflowDefinition = new WorkflowDefinition(workflowAny);
    assertNotNull(workflowDefinition);
    assertEquals("name is different", workflowAny.getStringValue("name"), workflowDefinition.getName());
    assertEquals("worker name is different", workflowAny.getMap("startAction").getStringValue("worker"),
      workflowDefinition.getStartAction().getWorker());
    assertEquals(Arrays.asList(JobRunMode.RUNONCE), workflowDefinition.getJobRunModes());
    assertEquals(JobRunMode.RUNONCE, workflowDefinition.getDefaultJobRunMode());
  }

  /** try parsing a workflow with a two allowed job run modes (non-standard order). */
  public void testSetJobRunModes() throws Exception {
    final AnyMap workflowAny = AccessAny.FACTORY.createAnyMap();
    workflowAny.put("name", "testSetJobRunMode");
    final AnyMap startActionAny = workflowAny.getMap("startAction", true);
    startActionAny.put("worker", "input-worker");
    final AnySeq modesAny = workflowAny.getSeq("modes", true);
    modesAny.add("runOnce");
    modesAny.add("standard");

    final WorkflowDefinition workflowDefinition = new WorkflowDefinition(workflowAny);
    assertNotNull(workflowDefinition);
    assertEquals("name is different", workflowAny.getStringValue("name"), workflowDefinition.getName());
    assertEquals("worker name is different", workflowAny.getMap("startAction").getStringValue("worker"),
      workflowDefinition.getStartAction().getWorker());
    assertEquals(Arrays.asList(JobRunMode.RUNONCE, JobRunMode.STANDARD), workflowDefinition.getJobRunModes());
    assertEquals(JobRunMode.RUNONCE, workflowDefinition.getDefaultJobRunMode());
  }

  /** try rejecting a workflow with an invalid allowed job run mode. */
  public void testInvalidJobRunMode() throws Exception {
    final AnyMap workflowAny = AccessAny.FACTORY.createAnyMap();
    workflowAny.put("name", "testSetJobRunMode");
    final AnyMap startActionAny = workflowAny.getMap("startAction", true);
    startActionAny.put("worker", "input-worker");
    final AnySeq modesAny = workflowAny.getSeq("modes", true);
    modesAny.add("doesNotRun");

    try {
      new WorkflowDefinition(workflowAny);
      fail("should not work");
    } catch (final InvalidDefinitionException ex) {
      ; // OK
    }
  }

}
