/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Juergen Schumacher, Andreas Weber, Drazen Cindric, Andreas Schank (all Attensity Europe GmbH) - initial
 * implementation
 **********************************************************************************************************************/
package org.eclipse.smila.jobmanager.definitions;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;

import org.eclipse.smila.common.definitions.AccessAny;
import org.eclipse.smila.common.definitions.DefinitionBase;
import org.eclipse.smila.common.exceptions.InvalidDefinitionException;
import org.eclipse.smila.datamodel.Any;
import org.eclipse.smila.datamodel.Any.ValueType;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.AnySeq;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.jobmanager.exceptions.InvalidConfigException;

/**
 * Definition of a workflow. JSON format:
 * 
 * <pre>
 * {
 *   "name": "indexUpdate",
 *   "parameters":
 *     {
 *       "paramName": "paramValue"
 *     },
 *   "startAction":
 *     {
 *        "worker": "sfrIndexOptimizer",
 *        "parameters":
 *           {
 *              "index": "indexName"
 *           },
 *        "input":
 *           {
 *              "partitionsToMerge": "indexPartitionsBucket"
 *           },
 *        "output":
 *           {
 *              "newPartition": "indexPartitionsBucket"
 *           }
 *     },
 *   "actions": [
 *        {
 *           "worker": "sfrRecordDeleter",
 *           "parameters":
 *              {
 *                  "index": "indexName"
 *              },
 *           "input":
 *              {
 *                  "partition": "indexPartitionsBucket",
 *                  "deletedRecords": "indexDeletesBucket"
 *              }
 *        } ]
 * }
 * </pre>
 */
public class WorkflowDefinition extends DefinitionBase {

  /** name of JSON property for lists of workflows. */
  public static final String KEY_WORKFLOWS = "workflows";

  /** name of JSON property for parameters. */
  public static final String KEY_PARAMETERS = "parameters";

  /** name of JSON property for parameter value. */
  public static final String KEY_VALUE = "value";

  /** name of JSON property for startAction. */
  public static final String KEY_START_ACTION = "startAction";

  /** name of JSON property for actions. */
  public static final String KEY_ACTIONS = "actions";

  /** name of JSON property for modes. */
  public static final String KEY_MODES = "modes";

  /** parameters are optional and describe a list of parameters to be declared for this workflow. */
  private final AnyMap _parameters;

  /** optional list of allowed job run modes. */
  private final List<JobRunMode> _modes;

  /** startAction is optional. */
  private final WorkflowAction _startAction;

  /** actions are optional, but at least one action or startAction must be defined. */
  private final Collection<WorkflowAction> _actions;

  /**
   * parse workflow definition from Any object.
   * 
   * @param workflow
   *          see class comment for format of Any object
   * @throws Exception
   *           An exception that occurred while converting the object.
   */
  public WorkflowDefinition(final AnyMap workflow) throws Exception {
    super(workflow);
    _parameters = workflow.getMap(KEY_PARAMETERS);

    final AnySeq modesAny = workflow.getSeq(KEY_MODES);
    try {
      _modes = JobRunMode.parseModes(modesAny);
    } catch (final InvalidDefinitionException ex) {
      throw new InvalidDefinitionException("Workflow '" + _name + "': " + ex.getMessage());
    }

    // start action
    final AnyMap startActionAny = (AnyMap) AccessAny.get(workflow, KEY_START_ACTION, ValueType.MAP);
    if (startActionAny == null) {
      throw new InvalidConfigException("Workflow " + _name + " does not define a startAction.");
    }
    _startAction = new WorkflowAction(startActionAny);

    // all actions:
    final AnySeq actionsAny = (AnySeq) AccessAny.get(workflow, KEY_ACTIONS, ValueType.SEQ);
    if (actionsAny != null) {
      _actions = new ArrayList<WorkflowAction>();
      for (final Any actionAny : actionsAny) {
        if (actionAny.isMap()) {
          _actions.add(new WorkflowAction((AnyMap) actionAny));
        }
      }
    } else {
      _actions = null;
    }
  }

  /**
   * Parse on single workflow from an Any object containing a workflow description:
   * 
   * <pre>
   * {
   * // see class comment for workflow format.
   * }
   * </pre>
   * 
   * @param workflowAny
   *          workflow as Any.
   * @return parsed workflow definition.
   * @throws InvalidDefinitionException
   *           error parsing Any.
   */
  public static WorkflowDefinition parseWorkflow(final AnyMap workflowAny) throws InvalidDefinitionException {
    WorkflowDefinition workflow = null;
    try {
      workflow = new WorkflowDefinition(workflowAny);
    } catch (final InvalidDefinitionException ex) {
      throw ex;
    } catch (final Exception ex) {
      throw new InvalidDefinitionException("Invalid any structure", ex);
    }
    return workflow;
  }

  /**
   * @return the parameters
   */
  public AnyMap getParameters() {
    return _parameters;
  }

  /** get allowed job run modes for this workflow: either the configured ones or null, if not configured. */
  public List<JobRunMode> getJobRunModes() {
    return _modes;
  }

  /** return the default job run mode for this workflow: either the first of the configured list of modes or null. */
  public JobRunMode getDefaultJobRunMode() {
    if (_modes == null) {
      return null;
    }
    return getJobRunModes().get(0);
  }

  /**
   * 
   * @return the startAction
   */
  public WorkflowAction getStartAction() {
    return _startAction;
  }

  /**
   * @return the actions. (Does not include start action)
   */
  public Collection<WorkflowAction> getActions() {
    return _actions;
  }

  /**
   * Returns object information as an Any representation.
   * 
   * @param includingAdditionalAttributes
   *          'true' if also any additional information in the AnyMap should be returned, 'false' if only the (minimal
   *          set of) relevant information should be returned.
   * @return Any object describing this workflow definition.
   */
  @Override
  public AnyMap toAny(final boolean includingAdditionalAttributes) {
    try {
      final AnyMap workflowAny = super.toAny(includingAdditionalAttributes);
      if (!includingAdditionalAttributes) {
        // be sure to overide our parameters
        if (_modes != null) {
          workflowAny.put(KEY_MODES, JobRunMode.toAny(_modes));
        }
        if (_parameters != null) {
          workflowAny.put(KEY_PARAMETERS, _parameters);
        }
        if (_startAction != null) {
          workflowAny.put(KEY_START_ACTION, _startAction.toAny());
        }
        if (_actions != null) {
          final AnySeq actionsAny = DataFactory.DEFAULT.createAnySeq();
          for (final WorkflowAction action : _actions) {
            actionsAny.add(action.toAny());
          }
          workflowAny.put(KEY_ACTIONS, actionsAny);
        }
      }
      return workflowAny;
    } catch (final Exception ex) {
      throw new IllegalStateException("Failed to create Any object for workflow " + _name, ex);
    }
  }

  /**
   * Returns the names of all buckets this workflow references.
   * 
   * @return The referenced buckets.
   */
  public Collection<String> getReferencedBuckets() {
    final Collection<String> result = new HashSet<String>();

    // first the bucket names for the input/output buckets
    final Collection<WorkflowAction> actions = new ArrayList<WorkflowAction>();
    if (_startAction != null) {
      actions.add(_startAction);
    }
    if (_actions != null) {
      actions.addAll(_actions);
    }
    for (final WorkflowAction workflowAction : actions) {
      if (workflowAction.getInput() != null) {
        result.addAll(workflowAction.getInput().values());
      }
      if (workflowAction.getOutput() != null) {
        result.addAll(workflowAction.getOutput().values());
      }
    }
    return result;
  }

  /**
   * Returns the names of all workers this workflow references.
   * 
   * @return The referenced workers.
   */
  public Collection<String> getReferencedWorkers() {
    final Collection<String> result = new HashSet<String>();

    if (_startAction != null) {
      result.add(_startAction.getWorker());
    }
    if (_actions != null) {
      for (final WorkflowAction workflowAction : _actions) {
        result.add(workflowAction.getWorker());
      }
    }
    return result;
  }
}
