/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Drazen Cindric (Attensity Europe GmbH) - initial implementation
 **********************************************************************************************************************/
package org.eclipse.smila.objectstore.filesystem.test;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStreamWriter;
import java.io.StringWriter;
import java.util.Collection;
import java.util.Properties;

import org.apache.commons.io.IOUtils;
import org.eclipse.core.runtime.Platform;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.objectstore.NoSuchObjectException;
import org.eclipse.smila.objectstore.NoSuchStoreException;
import org.eclipse.smila.objectstore.ObjectStoreException;
import org.eclipse.smila.objectstore.ObjectStoreService;
import org.eclipse.smila.objectstore.StoreObject;
import org.eclipse.smila.objectstore.StoreOutputStream;
import org.eclipse.smila.objectstore.filesystem.SimpleObjectStoreService;
import org.eclipse.smila.test.DeclarativeServiceTestCase;
import org.eclipse.smila.utils.config.ConfigUtils;
import org.eclipse.smila.utils.workspace.WorkspaceHelper;

/**
 * Test class for {@link SimpleObjectStoreService}.
 * 
 * @author drazen
 * 
 */
public class TestSimpleObjectStoreService extends DeclarativeServiceTestCase {

  /** The store name for testing. */
  private static final String STORE_NAME = "testStore";

  /** The object prefix for testing. */
  private static final String OBJECT_PREFIX = "object";

  /** The test data. */
  private static final String TEST_DATA = "this is a test: ";

  /** The service. */
  protected SimpleObjectStoreService _service;

  /** the root of the object store. */
  private File _rootStorePath;

  /** the base path to the (visible) object store. */
  private File _visibleStorePath;

  /** {@inheritDoc} */
  @Override
  protected void setUp() throws Exception {
    final ObjectStoreService lookup = getService(ObjectStoreService.class);
    assertNotNull(lookup);
    assertTrue(lookup instanceof SimpleObjectStoreService);
    _service = (SimpleObjectStoreService) lookup;
    _rootStorePath = WorkspaceHelper.createWorkingDir(SimpleObjectStoreService.BUNDLE_ID);
    InputStream configFileStream = null;
    try {
      configFileStream =
        ConfigUtils.getConfigStream(SimpleObjectStoreService.BUNDLE_ID, "objectstoreservice.properties");
      final Properties props = new Properties();
      props.load(configFileStream);
      if (props.containsKey(SimpleObjectStoreService.PROPERTY_ROOT_PATH)) {
        final String storeRootPath = props.getProperty(SimpleObjectStoreService.PROPERTY_ROOT_PATH);
        if (storeRootPath != null && !"".equals(storeRootPath)) {
          _rootStorePath = new File(storeRootPath);
        }
      }
    } finally {
      if (configFileStream != null) {
        IOUtils.closeQuietly(configFileStream);
      }
    }
    _visibleStorePath = new File(_rootStorePath, "objectstore");
  }

  /**
   * Tests if the {@link SimpleObjectStoreService} is available.
   * 
   * @throws Exception
   */
  public void testService() throws Exception {
    assertFalse(_service.existsStore("dummyStore"));
  }

  /**
   * Tests if given store names are valid.
   * 
   * @throws Exception
   */
  public void testValidStoreName() throws Exception {
    assertTrue(_service.isValidStoreName("23fahA66HHjkls456"));
    assertFalse(_service.isValidStoreName("!abc"));
    assertFalse(_service
      .isValidStoreName("01234567890123456789012345678901234567890123456789012345678901234567890123456789"
        + "01234567890123456789012345678901234567890123456789012345678901234567890123456789"
        + "01234567890123456789012345678901234567890123456789012345678901234567890123456789"
        + "01234567890123456789012345678901234567890123456789012345678901234567890123456789"));
  }

  /**
   * Adds stores, test if they are available and removes them with checking that they are gone after removing.
   * 
   * @throws Exception
   */
  public void testGetAndRemoveStoreNames() throws Exception {
    _service.ensureStore("test1");
    _service.ensureStore("test2");
    _service.ensureStore("test3");
    _service.ensureStore("test4");
    _service.ensureStore("test5");
    Collection<String> storeNames = _service.getStoreNames();
    assertTrue(storeNames.contains("test1"));
    assertTrue(storeNames.contains("test2"));
    assertTrue(storeNames.contains("test3"));
    assertTrue(_service.existsStore("test1"));
    _service.removeStore("test1");
    _service.removeStore("test2");
    _service.removeStore("test3");
    storeNames = _service.getStoreNames();
    assertFalse(storeNames.contains("test1"));
    assertFalse(storeNames.contains("test2"));
    assertFalse(storeNames.contains("test3"));
    assertTrue(storeNames.contains("test4"));
    assertTrue(storeNames.contains("test5"));
    assertFalse(_service.existsStore("test1"));
    _service.removeAllStores();
    storeNames = _service.getStoreNames();
    assertFalse(storeNames.contains("test4"));
    assertFalse(storeNames.contains("test5"));
    assertTrue(_service.getStoreNames().isEmpty());
  }

  /**
   * Tests if ensure, create and exists work properly.
   * 
   * @throws Exception
   */
  public void testEnsureCreateExistsStore() throws Exception {
    _service.removeStore(STORE_NAME);
    _service.createStore(STORE_NAME, null);
    try {
      _service.createStore(STORE_NAME, null);
      fail("Should fail, because store already exists.");
    } catch (final ObjectStoreException ose) {
      assertNotNull(ose);
    }
    _service.ensureStore(STORE_NAME);
    assertTrue(_service.existsStore(STORE_NAME));
    _service.removeStore(STORE_NAME);
    assertFalse(_service.existsStore(STORE_NAME));
    final String platform = Platform.getOS();
    if (Platform.OS_LINUX.equals(platform)) {
      _visibleStorePath.setReadOnly();
      try {
        _service.createStore(STORE_NAME, null);
        fail("Should not work, store can not be written.");
      } catch (final ObjectStoreException ose) {
        assertNotNull(ose);
      }
      try {
        _service.ensureStore(STORE_NAME);
        fail("Should not work, store can not be written.");
      } catch (final ObjectStoreException ose) {
        assertNotNull(ose);
      }
      _visibleStorePath.setWritable(true);
    }
    final String storeName = STORE_NAME + "new";
    _service.ensureStore(storeName);
    assertTrue(_service.existsStore(storeName));
    _service.removeStore(storeName);
    assertFalse(_service.existsStore(storeName));
  }

  /**
   * Puts some objects into a store and reads them.
   * 
   * @throws Exception
   */
  public void testPutGetAppendObjects() throws Exception {
    final String appendData = " with some more data";
    _service.removeStore(STORE_NAME);
    _service.createStore(STORE_NAME, null);
    for (int i = 0; i < 10; i++) {
      _service.putObject(STORE_NAME, OBJECT_PREFIX + i, getStringAsBytes(TEST_DATA + i));
    }
    try {
      _service.getObject(STORE_NAME, "notExistingId");
    } catch (final ObjectStoreException ose) {
      assertNotNull(ose);
    }
    for (int i = 0; i < 10; i++) {
      _service.existsObject(STORE_NAME, OBJECT_PREFIX + i);
      assertEquals(TEST_DATA + i, new String(_service.getObject(STORE_NAME, OBJECT_PREFIX + i)));
    }
    // append to existing
    _service.appendToObject(STORE_NAME, OBJECT_PREFIX + 2, getStringAsBytes(appendData));
    assertEquals(TEST_DATA + 2 + appendData, new String(_service.getObject(STORE_NAME, OBJECT_PREFIX + 2)));
    // append to non existing
    _service.appendToObject(STORE_NAME, OBJECT_PREFIX + 11, getStringAsBytes(appendData));
    assertEquals(appendData, new String(_service.getObject(STORE_NAME, OBJECT_PREFIX + 11)));
    assertEquals(appendData, getStringFromInputStream(_service.readObject(STORE_NAME, OBJECT_PREFIX + 11)));
    _service.removeStore(STORE_NAME);
  }

  /**
   * Test for write objects.
   * 
   * @throws Exception
   */
  public void testWriteObjects() throws Exception {
    _service.removeStore(STORE_NAME);
    _service.createStore(STORE_NAME, null);
    _service.putObject(STORE_NAME, OBJECT_PREFIX + 999, getStringAsBytes(TEST_DATA + 999));
    // overwrite with write object
    StoreOutputStream storeOutputStream = _service.writeObject(STORE_NAME, OBJECT_PREFIX + 999);
    storeOutputStream.write(getStringAsBytes("This is the overwritten text"));
    storeOutputStream.close();
    assertEquals("This is the overwritten text", new String(_service.getObject(STORE_NAME, OBJECT_PREFIX + 999)));
    // new object with write
    storeOutputStream = _service.writeObject(STORE_NAME, OBJECT_PREFIX + 555);
    storeOutputStream.write(getStringAsBytes(TEST_DATA));
    // not visible yet
    try {
      _service.getObject(STORE_NAME, OBJECT_PREFIX + 555);
      fail("Object should not be visible.");
    } catch (final NoSuchObjectException nsoe) {
      assertNotNull(nsoe);
    }
    storeOutputStream.close();
    assertEquals(TEST_DATA, new String(_service.getObject(STORE_NAME, OBJECT_PREFIX + 555)));
    // new object with write
    storeOutputStream = _service.writeObject(STORE_NAME, OBJECT_PREFIX + 777);
    storeOutputStream.write(getStringAsBytes(TEST_DATA));
    storeOutputStream.abort();
    // not visible
    try {
      _service.getObject(STORE_NAME, OBJECT_PREFIX + 777);
      fail("Object should not be visible.");
    } catch (final NoSuchObjectException nsoe) {
      assertNotNull(nsoe);
    }
    // should not work after abort
    try {
      storeOutputStream.write(100);
      fail("Write after abort should not work!");
    } catch (final IOException ioe) {
      assertNotNull(ioe);
    }
    _service.removeStore(STORE_NAME);
  }

  /**
   * Puts some objects into a store and reads the object infos.
   * 
   * @throws Exception
   */
  public void testPutAndGetObjectInfos() throws Exception {
    final String anotherPrefix = "another";
    _service.removeStore(STORE_NAME);
    _service.createStore(STORE_NAME, null);
    for (int i = 0; i < 10; i++) {
      _service.putObject(STORE_NAME, OBJECT_PREFIX + i, getStringAsBytes(TEST_DATA + i));
    }
    try {
      _service.getObjectInfo(STORE_NAME, "notExistingId");
    } catch (final ObjectStoreException ose) {
      assertNotNull(ose);
    }
    for (int i = 0; i < 10; i++) {
      final StoreObject objectInfo = _service.getObjectInfo(STORE_NAME, OBJECT_PREFIX + i);
      assertEquals(OBJECT_PREFIX + i, objectInfo.getId());
      assertEquals((TEST_DATA + i).length(), objectInfo.getSize());
    }
    try {
      _service.getStoreObjectInfos("notExistingStore");
    } catch (final NoSuchStoreException nsse) {
      assertNotNull(nsse);
    }
    Collection<StoreObject> objectInfos = _service.getStoreObjectInfos(STORE_NAME, "notExistingPrefix");
    assertTrue(objectInfos.isEmpty());
    for (int i = 0; i < 10; i++) {
      _service.putObject(STORE_NAME, anotherPrefix + i, getStringAsBytes(TEST_DATA + i));
    }
    int numberOfObjects = 0;
    objectInfos = _service.getStoreObjectInfos(STORE_NAME);
    for (final StoreObject objectInfo : objectInfos) {
      assertTrue(objectInfo.getId().startsWith(OBJECT_PREFIX) || objectInfo.getId().startsWith(anotherPrefix));
      numberOfObjects++;
    }
    assertEquals(20, numberOfObjects);
    numberOfObjects = 0;
    objectInfos = _service.getStoreObjectInfos(STORE_NAME, anotherPrefix);
    for (final StoreObject objectInfo : objectInfos) {
      assertTrue(objectInfo.getId().startsWith(anotherPrefix));
      numberOfObjects++;
    }
    assertEquals(10, numberOfObjects);
    _service.clearStore(STORE_NAME);
    objectInfos = _service.getStoreObjectInfos(STORE_NAME);
    assertTrue(objectInfos.isEmpty());
    _service.removeStore(STORE_NAME);
  }

  /**
   * Puts some objects into a store and reads the object infos.
   * 
   * @throws Exception
   */
  public void testPutAndCountObjects() throws Exception {
    final String anotherPrefix = "another";
    _service.removeStore(STORE_NAME);
    try {
      _service.countStoreObjects("notExistingStore", "");
    } catch (final NoSuchStoreException nsse) {
      assertNotNull(nsse);
    }
    _service.createStore(STORE_NAME, null);
    assertEquals(0, _service.countStoreObjects(STORE_NAME, ""));
    for (int i = 0; i < 10; i++) {
      _service.putObject(STORE_NAME, OBJECT_PREFIX + i, getStringAsBytes(TEST_DATA + i));
    }
    final Collection<StoreObject> objectInfos = _service.getStoreObjectInfos(STORE_NAME, "notExistingPrefix");
    assertTrue(objectInfos.isEmpty());
    for (int i = 0; i < 10; i++) {
      _service.putObject(STORE_NAME, anotherPrefix + i, getStringAsBytes(TEST_DATA + i));
    }
    assertEquals(20, _service.countStoreObjects(STORE_NAME, ""));
    assertEquals(10, _service.countStoreObjects(STORE_NAME, anotherPrefix));
    _service.clearStore(STORE_NAME);
    assertEquals(0, _service.countStoreObjects(STORE_NAME, ""));
    _service.removeStore(STORE_NAME);
  }

  /**
   * Puts some objects into a store with a folder hierarchy and reads the object infos.
   * 
   * @throws Exception
   */
  public void testPutAndGetObjectInfosWithFolderHierarchy() throws Exception {

    _service.removeStore(STORE_NAME);
    _service.createStore(STORE_NAME, null);
    String folderPrefix = "";
    for (int i = 0; i < 5; i++) {
      for (int j = 0; j < 10; j++) {
        _service.putObject(STORE_NAME, folderPrefix + OBJECT_PREFIX + j, getStringAsBytes(TEST_DATA + j));
      }
      folderPrefix = "folder" + i + "/";
    }
    _service.putObject(STORE_NAME, "foldera/b/c" + OBJECT_PREFIX + "abc", getStringAsBytes(TEST_DATA + "abc"));

    Collection<String> prefixes = _service.getPrefixes(STORE_NAME, "folder");
    assertEquals(5, prefixes.size());
    for (final String prefix : prefixes) {
      assertTrue(prefix.startsWith("folder"));
      assertTrue(prefix.endsWith("/"));
    }

    prefixes = _service.getPrefixes(STORE_NAME, "folder1");
    assertEquals(1, prefixes.size());
    for (final String prefix : prefixes) {
      assertTrue(prefix.startsWith("folder1/"));
      assertTrue(prefix.endsWith("/"));
    }

    prefixes = _service.getPrefixes(STORE_NAME, "folder1/");
    assertEquals(10, prefixes.size());
    for (final String prefix : prefixes) {
      assertTrue(prefix.startsWith("folder1/"));
      assertFalse(prefix.endsWith("/"));
    }

    prefixes = _service.getPrefixes(STORE_NAME, "folder1/" + OBJECT_PREFIX);
    assertEquals(10, prefixes.size());
    for (final String prefix : prefixes) {
      assertTrue(prefix.startsWith("folder1/" + OBJECT_PREFIX));
      assertFalse(prefix.endsWith("/"));
    }

    prefixes = _service.getPrefixes(STORE_NAME, "foldera/b");
    assertEquals(1, prefixes.size());
    for (final String prefix : prefixes) {
      assertTrue(prefix.startsWith("foldera/b/"));
      assertTrue(prefix.endsWith("/"));
    }

    prefixes = _service.getPrefixes(STORE_NAME, "foldera/b/c");
    assertEquals(1, prefixes.size());
    for (final String prefix : prefixes) {
      assertTrue(prefix.startsWith("foldera/b/c"));
      assertFalse(prefix.endsWith("/"));
    }

    prefixes = _service.getPrefixes(STORE_NAME, null);
    assertEquals(15, prefixes.size());
    for (final String prefix : prefixes) {
      assertTrue(prefix.startsWith(OBJECT_PREFIX) || prefix.startsWith("folder"));
    }

    prefixes = _service.getPrefixes(STORE_NAME, "notexistingprefix");
    assertEquals(0, prefixes.size());

    prefixes = _service.getPrefixes(STORE_NAME, "not/existing/prefix");
    assertEquals(0, prefixes.size());

    _service.clearStore(STORE_NAME);
    prefixes = _service.getPrefixes(STORE_NAME, null);
    assertTrue(prefixes.isEmpty());
    _service.removeStore(STORE_NAME);
  }

  /**
   * Puts some objects into a store and reads them.
   * 
   * @throws Exception
   */
  public void testGetStoreInfo() throws Exception {
    _service.removeStore(STORE_NAME);
    _service.createStore(STORE_NAME, null);
    for (int i = 0; i < 10; i++) {
      _service.putObject(STORE_NAME, OBJECT_PREFIX + i, getStringAsBytes(TEST_DATA + i));
    }
    AnyMap storeInfo = _service.getStoreInfo(STORE_NAME, false);
    assertEquals(STORE_NAME, storeInfo.getStringValue(SimpleObjectStoreService.KEY_STORE_NAME));
    assertTrue(storeInfo.getLongValue(SimpleObjectStoreService.KEY_OBJECT_COUNT) == 10);
    assertTrue(storeInfo.getLongValue(SimpleObjectStoreService.KEY_SIZE) > 0);
    assertNull(storeInfo.getMap(SimpleObjectStoreService.KEY_OBJECTS));
    storeInfo = _service.getStoreInfo(STORE_NAME, true);
    assertEquals(10, storeInfo.getSeq(SimpleObjectStoreService.KEY_OBJECTS).size());
    _service.removeStore(STORE_NAME);
  }

  /**
   * Puts some objects into a store and reads them.
   * 
   * @throws Exception
   */
  public void testRemoveObjects() throws Exception {
    _service.removeStore(STORE_NAME);
    _service.createStore(STORE_NAME, null);
    for (int i = 0; i < 10; i++) {
      _service.putObject(STORE_NAME, OBJECT_PREFIX + i, getStringAsBytes(TEST_DATA + i));
    }
    AnyMap storeInfo = _service.getStoreInfo(STORE_NAME, false);
    assertTrue(storeInfo.getLongValue(SimpleObjectStoreService.KEY_OBJECT_COUNT) == 10);
    for (int i = 0; i < 10; i++) {
      _service.removeObject(STORE_NAME, OBJECT_PREFIX + i);
    }
    storeInfo = _service.getStoreInfo(STORE_NAME, false);
    assertTrue(storeInfo.getLongValue(SimpleObjectStoreService.KEY_OBJECT_COUNT) == 0);
    _service.clearStore(STORE_NAME);
    // test if all obsolete folders are removed after removal of objects
    final String objectPrefix = "/a/b/c/";
    final String objectId1 = objectPrefix + "test1";
    final String objectId2 = objectPrefix + "d/e/test2";

    // check if all subfolders have been removed, remove sub object first
    _service.putObject(STORE_NAME, objectId1, getStringAsBytes(TEST_DATA));
    _service.putObject(STORE_NAME, objectId2, getStringAsBytes(TEST_DATA));
    File prefixFolder = new File(_visibleStorePath, STORE_NAME + objectPrefix);
    String[] subFolders = prefixFolder.list();
    // should be folder d and object test2
    assertEquals(2, subFolders.length);
    _service.removeObject(STORE_NAME, objectId2);
    subFolders = prefixFolder.list();
    // should be just object test2
    assertEquals(1, subFolders.length);
    _service.removeObject(STORE_NAME, objectId1);
    prefixFolder = new File(_visibleStorePath, STORE_NAME);
    subFolders = prefixFolder.list();
    // should be empty
    assertEquals(0, subFolders.length);

    // check if all subfolders have been removed, remove super object first
    _service.putObject(STORE_NAME, objectId1, getStringAsBytes(TEST_DATA));
    _service.putObject(STORE_NAME, objectId2, getStringAsBytes(TEST_DATA));
    prefixFolder = new File(_visibleStorePath, STORE_NAME + objectPrefix);
    subFolders = prefixFolder.list();
    // should be folder d and object test2
    assertEquals(2, subFolders.length);
    _service.removeObject(STORE_NAME, objectId1);
    subFolders = prefixFolder.list();
    // should be folder d
    assertEquals(1, subFolders.length);
    _service.removeObject(STORE_NAME, objectId2);
    prefixFolder = new File(_visibleStorePath, STORE_NAME);
    subFolders = prefixFolder.list();
    // should be empty
    assertEquals(0, subFolders.length);
    _service.removeStore(STORE_NAME);
  }

  /** test removeObjects method. */
  public void testRemoveObjectsByPrefix() throws Exception {
    _service.removeStore(STORE_NAME);
    _service.ensureStore(STORE_NAME);
    _service.putObject(STORE_NAME, "aa", getStringAsBytes(TEST_DATA));
    _service.putObject(STORE_NAME, "ab", getStringAsBytes(TEST_DATA));
    _service.putObject(STORE_NAME, "ba", getStringAsBytes(TEST_DATA));
    _service.putObject(STORE_NAME, "bb", getStringAsBytes(TEST_DATA));
    _service.putObject(STORE_NAME, "cc", getStringAsBytes(TEST_DATA));
    _service.removeObjects(STORE_NAME, "b");
    final Collection<StoreObject> infos = _service.getStoreObjectInfos(STORE_NAME);
    assertEquals(3, infos.size());
    final Collection<StoreObject> prefixInfos = _service.getStoreObjectInfos(STORE_NAME, "b");
    assertEquals(0, prefixInfos.size());
  }

  /** test removeObjects method, remove complete directory. */
  public void testRemoveObjectsByPrefixSubDir() throws Exception {
    _service.removeStore(STORE_NAME);
    _service.ensureStore(STORE_NAME);
    _service.putObject(STORE_NAME, "a/a", getStringAsBytes(TEST_DATA));
    _service.putObject(STORE_NAME, "a/b", getStringAsBytes(TEST_DATA));
    _service.putObject(STORE_NAME, "b/a", getStringAsBytes(TEST_DATA));
    _service.putObject(STORE_NAME, "b/b", getStringAsBytes(TEST_DATA));
    _service.putObject(STORE_NAME, "c/c", getStringAsBytes(TEST_DATA));
    _service.removeObjects(STORE_NAME, "b/");
    final Collection<StoreObject> infos = _service.getStoreObjectInfos(STORE_NAME);
    assertEquals(3, infos.size());
    final Collection<StoreObject> prefixInfos = _service.getStoreObjectInfos(STORE_NAME, "b");
    assertEquals(0, prefixInfos.size());
  }

  /** test removeObjects method, remove file and directory. */
  public void testRemoveObjectsByPrefixDirAndFiles() throws Exception {
    _service.removeStore(STORE_NAME);
    _service.ensureStore(STORE_NAME);
    _service.putObject(STORE_NAME, "aa", getStringAsBytes(TEST_DATA));
    _service.putObject(STORE_NAME, "ba", getStringAsBytes(TEST_DATA));
    _service.putObject(STORE_NAME, "b/a", getStringAsBytes(TEST_DATA));
    _service.putObject(STORE_NAME, "b/b", getStringAsBytes(TEST_DATA));
    _service.putObject(STORE_NAME, "c/c", getStringAsBytes(TEST_DATA));
    _service.removeObjects(STORE_NAME, "b");
    final Collection<StoreObject> infos = _service.getStoreObjectInfos(STORE_NAME);
    assertEquals(2, infos.size());
    final Collection<StoreObject> prefixInfos = _service.getStoreObjectInfos(STORE_NAME, "b");
    assertEquals(0, prefixInfos.size());
  }

  /** test removeObjects method, no matchesy. */
  public void testRemoveObjectsByPrefixNoMatch() throws Exception {
    _service.removeStore(STORE_NAME);
    _service.ensureStore(STORE_NAME);
    _service.putObject(STORE_NAME, "aa", getStringAsBytes(TEST_DATA));
    _service.putObject(STORE_NAME, "ba", getStringAsBytes(TEST_DATA));
    _service.putObject(STORE_NAME, "b/a", getStringAsBytes(TEST_DATA));
    _service.putObject(STORE_NAME, "b/b", getStringAsBytes(TEST_DATA));
    _service.putObject(STORE_NAME, "c/c", getStringAsBytes(TEST_DATA));
    _service.removeObjects(STORE_NAME, "d");
    final Collection<StoreObject> infos = _service.getStoreObjectInfos(STORE_NAME);
    assertEquals(5, infos.size());
  }

  /** test removeObjects method, clear store. */
  public void testRemoveObjectsByPrefixEmptyPrefix() throws Exception {
    _service.removeStore(STORE_NAME);
    _service.ensureStore(STORE_NAME);
    _service.putObject(STORE_NAME, "aa", getStringAsBytes(TEST_DATA));
    _service.putObject(STORE_NAME, "ba", getStringAsBytes(TEST_DATA));
    _service.putObject(STORE_NAME, "b/a", getStringAsBytes(TEST_DATA));
    _service.putObject(STORE_NAME, "b/b", getStringAsBytes(TEST_DATA));
    _service.putObject(STORE_NAME, "c/c", getStringAsBytes(TEST_DATA));
    _service.removeObjects(STORE_NAME, "");
    final Collection<StoreObject> infos = _service.getStoreObjectInfos(STORE_NAME);
    assertEquals(0, infos.size());
  }

  /** test removeObjects method in some more complicated structure. */
  public void testRemoveObjectsByPrefixInBiggerStructure() throws Exception {
    _service.removeStore(STORE_NAME);
    _service.ensureStore(STORE_NAME);
    _service.putObject(STORE_NAME, "a/a/a", getStringAsBytes(TEST_DATA));
    _service.putObject(STORE_NAME, "b/aa/a", getStringAsBytes(TEST_DATA));
    _service.putObject(STORE_NAME, "b/ba/a", getStringAsBytes(TEST_DATA));
    _service.putObject(STORE_NAME, "b/bb/b", getStringAsBytes(TEST_DATA));
    _service.putObject(STORE_NAME, "b/ca/a", getStringAsBytes(TEST_DATA));
    _service.putObject(STORE_NAME, "c/c/c", getStringAsBytes(TEST_DATA));
    _service.removeObjects(STORE_NAME, "b/b");
    Collection<StoreObject> infos = _service.getStoreObjectInfos(STORE_NAME);
    assertEquals(4, infos.size());
    infos = _service.getStoreObjectInfos(STORE_NAME, "b/");
    assertEquals(2, infos.size());
    infos = _service.getStoreObjectInfos(STORE_NAME, "b/b");
    assertEquals(0, infos.size());
  }

  /** return input stream as string. */
  protected String getStringFromInputStream(final InputStream inputStream) {
    try {
      final StringWriter writer = new StringWriter();
      IOUtils.copy(inputStream, writer, "UTF-8");
      return writer.toString();
    } catch (final Exception e) {
      return null;
    } finally {
      IOUtils.closeQuietly(inputStream);
    }
  }

  /** return string as UTF-8 bytes. */
  protected byte[] getStringAsBytes(final String data) throws Exception {
    final ByteArrayOutputStream ao = new ByteArrayOutputStream();
    final OutputStreamWriter os = new OutputStreamWriter(ao, "UTF-8");
    os.append(data);
    os.close();
    return ao.toByteArray();
  }
}
