/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Juergen Schumacher (Attensity Europe GmbH) - initial implementation
 **********************************************************************************************************************/
package org.eclipse.smila.objectstore.test;

import org.apache.commons.httpclient.HttpStatus;
import org.apache.commons.httpclient.methods.DeleteMethod;
import org.apache.commons.httpclient.methods.GetMethod;
import org.apache.commons.httpclient.methods.PutMethod;
import org.eclipse.smila.datamodel.Any;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.AnySeq;
import org.eclipse.smila.http.server.util.RequestHandler;
import org.eclipse.smila.objectstore.httphandler.StoreAdminHandler;

/**
 * Test class for {@link StoreAdminHandler}.
 */
public class TestStoreAdminHandler extends AStoreHandlerTest {

  /**
   * test correct handler implementation class. 
   */
  public void testHandlerClass() throws Exception {
    super.setUp();
    final RequestHandler requestHandler = getService(RequestHandler.class, "(uriPattern=/store/\\([^/]+\\)/?$)");
    assertNotNull(requestHandler);
    assertTrue(requestHandler instanceof StoreAdminHandler);
  }

  /** test store creation via HTTP. */
  public void testCreateStore() throws Exception {
    final PutMethod method = new PutMethod(BASE_URL + "new-store");
    final int code = _client.executeMethod(method);
    assertEquals(HttpStatus.SC_CREATED, code);
    assertTrue(_store.existsStore("new-store"));
  }

  /** test status for trying to create an existing store. */
  public void testCreateExistingStore() throws Exception {
    _store.ensureStore("existing-store");
    final PutMethod method = new PutMethod(BASE_URL + "existing-store");
    final int code = _client.executeMethod(method);
    assertEquals(HttpStatus.SC_BAD_REQUEST, code);
    assertTrue(_store.existsStore("existing-store"));
  }

  /** test store deletion via HTTP. */
  public void testDeleteStore() throws Exception {
    _store.ensureStore("store-to-delete");
    final DeleteMethod method = new DeleteMethod(BASE_URL + "store-to-delete");
    final int code = _client.executeMethod(method);
    assertEquals(HttpStatus.SC_OK, code);
    assertFalse(_store.existsStore("store-to-delete"));
  }

  /** test that no error occurs when deleting a missing store. */
  public void testDeleteMissingStore() throws Exception {
    assertFalse(_store.existsStore("store-not-existing"));
    final DeleteMethod method = new DeleteMethod(BASE_URL + "store-not-existing");
    final int code = _client.executeMethod(method);
    assertEquals(HttpStatus.SC_OK, code);
    assertFalse(_store.existsStore("store-to-delete"));
  }

  /** test store info including the object list. */
  public void testStoreInfoComplete() throws Exception {
    _store.ensureStore("store");
    _store.putObject("store", "obj1", "content1".getBytes());
    _store.putObject("store", "obj2", "content2".getBytes());
    _store.putObject("store", "obj3", "content3".getBytes());
    final AnyMap storeInfo = getUrlContent(BASE_URL + "store");
    assertEquals("store", storeInfo.getStringValue("storeName"));
    assertTrue(storeInfo.containsKey("storeProperties"));
    assertEquals(3, storeInfo.getLongValue("objectCount").intValue());
    assertEquals(24, storeInfo.getLongValue("size").intValue());
    assertTrue(storeInfo.containsKey("objects"));
    final AnySeq objectList = storeInfo.getSeq("objects");
    assertEquals(3, objectList.size());
    for (final Any object : objectList) {
      assertTrue(object.isMap());
      final AnyMap objectMap = (AnyMap) object;
      final String name = objectMap.getStringValue("id");
      assertTrue(name.startsWith("obj"));
      assertEquals(8, objectMap.getLongValue("size").intValue());
      assertTrue(objectMap.containsKey("timestamp"));
    }
  }

  /** test store info with suppressed object list. */
  public void testStoreInfoNoObjects() throws Exception {
    _store.ensureStore("store");
    _store.putObject("store", "obj1", "content1".getBytes());
    _store.putObject("store", "obj2", "content2".getBytes());
    _store.putObject("store", "obj3", "content3".getBytes());
    final AnyMap storeInfo = getUrlContent(BASE_URL + "store?returnObjects=false");
    assertEquals("store", storeInfo.getStringValue("storeName"));
    assertTrue(storeInfo.containsKey("storeProperties"));
    assertEquals(3, storeInfo.getLongValue("objectCount").intValue());
    assertEquals(24, storeInfo.getLongValue("size").intValue());
    assertFalse(storeInfo.containsKey("objects"));
  }

  /** test error status for reading store info of a missing store. */
  public void testGetStoreInfoNoStore() throws Exception {
    assertFalse(_store.existsStore("not-existing-store"));
    final GetMethod method = new GetMethod(BASE_URL + "not-existing-store");
    final int code = _client.executeMethod(method);
    assertEquals(HttpStatus.SC_NOT_FOUND, code);
  }

  /** test error status for trying to create a store with an invalid name. */
  public void testInvalidStoreName() throws Exception {
    final PutMethod method = new PutMethod(BASE_URL + "invalid@store");
    final int code = _client.executeMethod(method);
    assertEquals(HttpStatus.SC_BAD_REQUEST, code);
  }

}
