/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Juergen Schumacher (Attensity Europe GmbH) - initial implementation
 **********************************************************************************************************************/
package org.eclipse.smila.objectstore.test;

import java.io.BufferedReader;
import java.io.InputStreamReader;
import java.net.URL;
import java.net.URLConnection;

import org.apache.commons.httpclient.HttpStatus;
import org.apache.commons.httpclient.methods.DeleteMethod;
import org.apache.commons.httpclient.methods.GetMethod;
import org.apache.commons.httpclient.methods.PutMethod;
import org.apache.commons.httpclient.methods.StringRequestEntity;
import org.eclipse.smila.datamodel.Any;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.http.server.util.RequestHandler;
import org.eclipse.smila.objectstore.httphandler.StoreObjectHandler;

/**
 * Test for {@link StoreObjectHandler}.
 */
public class TestStoreObjectHandler extends AStoreHandlerTest {

  /** test store name. */
  protected static final String STORE_NAME = "teststore";

  protected static final String TEST_URL = BASE_URL + STORE_NAME + "/";

  /** {@inheritDoc} */
  @Override
  protected void setUp() throws Exception {
    super.setUp();
    _store.ensureStore(STORE_NAME);
    _store.clearStore(STORE_NAME);
  }

  /** check handler implementation class. */
  public void testHandlerClass() throws Exception {
    super.setUp();
    final RequestHandler requestHandler =
      getService(RequestHandler.class, "(uriPattern=/store/\\([^/]+\\)/\\(.+\\)$)");
    assertNotNull(requestHandler);
    assertTrue(requestHandler instanceof StoreObjectHandler);
  }

  /** test standard behavior. */
  public void testBonObject() throws Exception {
    final String objectId = "4711";
    final AnyMap object = createObject();
    _store.putObject(STORE_NAME, objectId, _anyWriter.writeBinaryObject(object));
    final AnyMap response = getUrlContent(TEST_URL + objectId);
    assertEquals(object, response);
  }

  /** test fallback to JSON parsing. */
  public void testJsonObject() throws Exception {
    final String objectId = "4711.json";
    final AnyMap object = createObject();
    _store.putObject(STORE_NAME, objectId, _anyWriter.writeJsonObject(object).getBytes());
    final AnyMap response = getUrlContent(TEST_URL + objectId);
    assertEquals(object, response);
  }

  /** test record bulk streaming. */
  public void testBonBulk() throws Exception {
    final String objectId = "4711.app";
    final AnyMap object = createObject();
    final int noOfRecords = 10;
    for (int i = 0; i < noOfRecords; i++) {
      object.put("number", i);
      _store.appendToObject(STORE_NAME, objectId, _anyWriter.writeBinaryObject(object));
    }
    final URL url = new URL(TEST_URL + objectId);
    final URLConnection conn = url.openConnection();
    final BufferedReader reader = new BufferedReader(new InputStreamReader(conn.getInputStream(), "utf-8"));
    String jsonLine = null;
    int count = 0;
    while ((jsonLine = reader.readLine()) != null) {
      object.put("number", count);
      final Record readRecord = _recordReader.readJsonObject(jsonLine);
      assertEquals(object, readRecord.getMetadata());
      ++count;
    }
    assertEquals(noOfRecords, count);
  }

  /** test error on empty file. */
  public void testEmptyObject() throws Exception {
    final String objectId = "4712";
    final byte[] binaryObject = new byte[0];
    _store.putObject(STORE_NAME, objectId, binaryObject);
    final GetMethod method = new GetMethod(TEST_URL + objectId);
    final int code = _client.executeMethod(method);
    assertEquals(HttpStatus.SC_NO_CONTENT, code);
  }

  /** test error on object with no BON content. */
  public void testErrorNoBon() throws Exception {
    final String objectId = "4713";
    _store.putObject(STORE_NAME, objectId, "no-bon".getBytes());
    final GetMethod method = new GetMethod(TEST_URL + objectId);
    final int code = _client.executeMethod(method);
    assertEquals(HttpStatus.SC_BAD_REQUEST, code);
  }

  /** test error not-existing object. */
  public void testErrorNoObject() throws Exception {
    final String objectId = "no-such-object";
    final GetMethod method = new GetMethod(TEST_URL + objectId);
    final int code = _client.executeMethod(method);
    assertEquals(HttpStatus.SC_NOT_FOUND, code);
  }

  /** test error on object with no BON content. */
  public void testWriteBon() throws Exception {
    final String objectId = "4714";
    final AnyMap object = createObject();
    final String jsonObject = _anyWriter.writeJsonObject(object);
    final PutMethod method = new PutMethod(TEST_URL + objectId);
    method.setRequestEntity(new StringRequestEntity(jsonObject, "application/json", "utf-8"));
    final int code = _client.executeMethod(method);
    assertEquals(HttpStatus.SC_OK, code);
    assertTrue(_store.existsObject(STORE_NAME, objectId));
    final byte[] content = _store.getObject(STORE_NAME, objectId);
    final Any readObject = _anyReader.readBinaryObject(content);
    assertEquals(object, readObject);
  }

  /** test for deleting one existing object. */
  public void testRemoveObject() throws Exception {
    final String objectId = "0815";
    _store.putObject(STORE_NAME, objectId, objectId.getBytes());
    final DeleteMethod method = new DeleteMethod(TEST_URL + objectId);
    final int code = _client.executeMethod(method);
    assertEquals(HttpStatus.SC_OK, code);
  }

  /** test for deleting non existing object. */
  public void testRemoveNonExistingObject() throws Exception {
    final String objectId = "no-such-object";
    final DeleteMethod method = new DeleteMethod(TEST_URL + objectId);
    final int code = _client.executeMethod(method);
    assertEquals(HttpStatus.SC_OK, code);
  }

  /** create a simple test object. */
  private AnyMap createObject() {
    final AnyMap object = DataFactory.DEFAULT.createAnyMap();
    object.put("foo", "bar");
    object.put("SMILA", "i like");
    return object;
  }
}
