/*******************************************************************************
 * Copyright (c) 2008 empolis GmbH and brox IT Solutions GmbH.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Juergen Schumacher (empolis GmbH) - initial API and implementation
 *******************************************************************************/

package org.eclipse.smila.processing.bpel.test;

import org.eclipse.smila.blackboard.BlackboardAccessException;

/**
 * Test the EchoPipeline. This pipeline just returns the given records.
 * 
 * @author jschumacher
 * 
 */
public class TestLoopPipeline extends AWorkflowProcessorTest {
  /**
   * name of pipeline to test.
   */
  public static final String PIPELINE_NAME = "LoopPipeline";

  /**
   * {@inheritDoc}
   * 
   * @see org.eclipse.smila.processing.bpel.test.AWorkflowProcessorTest#getPipelineName()
   */
  @Override
  protected String getPipelineName() {
    return PIPELINE_NAME;
  }

  /**
   * test single record loop with if-branch.
   * 
   * @throws Exception
   *           test fails
   */
  public void testOneRecordIfBranch() throws Exception {
    final String request = createBlackboardRecord("source", "key");
    getBlackboard().getMetadata(request).put("workflow-attribute", true);
    final String[] result = getProcessor().process(PIPELINE_NAME, getBlackboard(), new String[] { request });
    assertEquals(1, result.length);
    assertEquals(request, result[0]);
    assertAttributesSet(result[0]);
  }

  /**
   * test single record loop with else-branch.
   * 
   * @throws Exception
   *           test fails
   */
  public void testOneRecordElseBranch() throws Exception {
    final String request = createBlackboardRecord("source", "key");
    getBlackboard().getMetadata(request).put("workflow-attribute", false);
    final String[] result = getProcessor().process(PIPELINE_NAME, getBlackboard(), new String[] { request });
    assertEquals(1, result.length);
    assertEquals(request, result[0]);
    assertAttributesSet(result[0]);
  }

  /**
   * test multiple records in one call.
   * 
   * @throws Exception
   *           test fails
   */
  public void testManyRecords() throws Exception {
    final int numberOfRecords = 10;
    final String[] request = new String[numberOfRecords];
    for (int i = 0; i < numberOfRecords; i++) {
      final String id = "key" + i;
      request[i] = createBlackboardRecord("source", id);
      if (i % 2 == 0) {
        getBlackboard().getMetadata(request[i]).put("workflow-attribute", true);
      }
    }
    final String[] result = getProcessor().process(PIPELINE_NAME, getBlackboard(), request);
    assertEquals(numberOfRecords, result.length);
    for (int i = 0; i < numberOfRecords; i++) {
      assertEquals(request[i], result[i]);
      assertAttributesSet(result[i]);
    }
  }

  /**
   * test single record loop with else-branch.
   * 
   * @throws Exception
   *           test fails
   */
  public void testNoRecord() throws Exception {
    final String[] result = getProcessor().process(PIPELINE_NAME, getBlackboard(), new String[0]);
    assertEquals(0, result.length);
  }

  /** check if result attribute are set as exected. */
  private void assertAttributesSet(final String recordId) throws BlackboardAccessException {
    final boolean assertIfBranch = getBlackboard().getMetadata(recordId).containsKey("workflow-attribute");
    assertEquals(assertIfBranch, getBlackboard().getMetadata(recordId).containsKey("if1"));
    assertEquals(assertIfBranch, getBlackboard().getMetadata(recordId).containsKey("if2"));
    assertEquals(!assertIfBranch, getBlackboard().getMetadata(recordId).containsKey("else"));

  }
}
