/***********************************************************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. 
 * This program and the accompanying materials are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Daniel Stucky (empolis GmbH) - initial API and implementation
 * Drazen Cindric (Attensity Europe GmbH) - data model improvements
 **********************************************************************************************************************/

package org.eclipse.smila.processing.pipelets.test;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;

import org.eclipse.smila.blackboard.Blackboard;
import org.eclipse.smila.blackboard.Blackboard.Get;
import org.eclipse.smila.blackboard.BlackboardAccessException;
import org.eclipse.smila.blackboard.BlackboardFactory;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.processing.Pipelet;
import org.eclipse.smila.processing.ProcessingException;
import org.eclipse.smila.processing.util.ProcessingConstants;
import org.eclipse.smila.test.DeclarativeServiceTestCase;

/**
 * Base class for Pipelet tests.
 */
public abstract class APipeletTest extends DeclarativeServiceTestCase {

  /**
   * Blackboard service to use in test.
   */
  protected Blackboard _blackboard;

  /**
   * Check if BlackboardService service is active. Wait up to 30 seconds for start. Fail, if no service is starting.
   * {@inheritDoc}
   * 
   * @see junit.framework.TestCase#setUp()
   */
  @Override
  protected void setUp() throws Exception {
    super.setUp();
    final BlackboardFactory factory = getService(BlackboardFactory.class);
    assertNotNull("no BlackboardFactory service found.", factory);
    _blackboard = factory.createPersistingBlackboard();
    assertNotNull("no Blackboard created", _blackboard);
  }

  /**
   * @return the blackboard
   */
  public Blackboard getBlackboard() {
    return _blackboard;
  }

  /**
   * create a new record on the blackboard.
   * 
   * @param source
   *          source value of ID
   * @param key
   *          key value of ID
   * @return id of created record.
   * @throws BlackboardAccessException
   */
  protected String createBlackboardRecord(final String source, final String key) throws BlackboardAccessException {
    final String id = source + ":" + key;
    _blackboard.unloadRecord(id);
    _blackboard.getRecord(id, Get.NEW);
    return id;
  }

  /**
   * Removes the BOM from a UTF-8 String.
   * 
   * @param stringWithBom
   *          the String with BOM
   * @return a String without the BOM
   */
  protected String removeBOM(final String stringWithBom) {
    return stringWithBom.substring(1);
  }

  /**
   * tests pipelet error handling for given input records where some records are expected to fail.
   */
  protected void doRobustnessTest(final Pipelet pipelet, final AnyMap config, final Collection<String> recordIds,
    final int noOfSuccessfulRecords, final boolean failOnError) throws Exception {

    if (failOnError) {
      config.put(ProcessingConstants.KEY_FAIL_ON_ERROR, "true");
    } else {
      config.put(ProcessingConstants.KEY_FAIL_ON_ERROR, "false");
    }

    // execute
    if (failOnError) {
      try {
        pipelet.process(getBlackboard(), recordIds.toArray(new String[recordIds.size()]));
        fail("failOnError=true -> exception should be thrown");
      } catch (final ProcessingException e) {
        ; // expected
      }
    } else {
      final String[] result = pipelet.process(getBlackboard(), recordIds.toArray(new String[recordIds.size()]));
      final List<String> resultIds = Arrays.asList(result);
      assertEquals(noOfSuccessfulRecords, resultIds.size());
    }
  }

}
