/***********************************************************************************************************************
 * Copyright (c) 2008 empolis GmbH and brox IT Solutions GmbH. All rights reserved. This program and the accompanying
 * materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this distribution,
 * and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Peter Wissel (brox IT Solutions GmbH) - initial API and implementation
 **********************************************************************************************************************/

package org.eclipse.smila.solr.search;

import static java.lang.String.format;

import java.text.MessageFormat;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.solr.client.solrj.response.FacetField;
import org.apache.solr.client.solrj.response.FacetField.Count;
import org.apache.solr.client.solrj.response.Group;
import org.apache.solr.client.solrj.response.GroupCommand;
import org.apache.solr.client.solrj.response.GroupResponse;
import org.apache.solr.client.solrj.response.QueryResponse;
import org.apache.solr.client.solrj.response.SpellCheckResponse;
import org.apache.solr.client.solrj.response.SpellCheckResponse.Suggestion;
import org.apache.solr.common.SolrDocument;
import org.apache.solr.common.SolrDocumentList;
import org.apache.solr.common.util.NamedList;
import org.apache.solr.common.util.SimpleOrderedMap;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.AnySeq;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.datamodel.Value;
import org.eclipse.smila.search.api.SearchResultConstants;
import org.eclipse.smila.search.api.helper.ResultBuilder;
import org.eclipse.smila.solr.SolrConstants;

/**
 * builds the SMILA result record from the solr result objects.
 * 
 * @author pwissel
 * 
 */
public class SolrResultBuilder extends ResultBuilder {

  /**
   * Default workflow name.
   */
  private static final String DEFAULT_WORKFLOW = "SolrSearchDefaultWorkflow";

  /**
   * The log.
   */
  protected final Log _log = LogFactory.getLog(this.getClass());

  /**
   * The QueryResponse from solr server.
   */
  private final QueryResponse _response;

  /**
   * Constructor.
   * 
   * @param record
   *          the query record.
   * @param response
   *          the solr query response.
   */
  public SolrResultBuilder(Record record, QueryResponse response) {
    super(DEFAULT_WORKFLOW, record);
    _response = response;
  }

  /**
   * Constructor.
   * 
   * @param workflowName
   *          the workflow name.
   * @param record
   *          the query record.
   * @param response
   *          the solr query response.
   */
  public SolrResultBuilder(String workflowName, Record record, QueryResponse response) {
    super(workflowName, record);
    _response = response;
  }

  /**
   * Process the response result and add to query record.
   * 
   * @return
   */
  public Record processResponse() {
    // set runtime
    setRuntime(new Long(_response.getQTime()));

    processQueryResponse();
    processFacetResponse();
    processTermsResponse();
    processSpellcheckResponse();
    processMoreLikeThis();
    processGroupResponse();

    return getResult();
  }

  /**
   * 
   */
  private void processMoreLikeThis() {
    final Object mltResults = _response.getResponse().get(SolrConstants.MORE_LIKE_THIS);
    if (mltResults != null) {

    }
  }

  /**
   * Process query response.
   */
  private void processQueryResponse() {
    final SolrDocumentList results = _response.getResults();
    if (results == null) {
      return;
    }
    SimpleOrderedMap<SolrDocumentList> moreLikeThisLists = getMoreLikeThisLists();
    final Map<String, Map<String, List<String>>> highlighting = _response.getHighlighting();
    // set count and max score
    setCount(results.getNumFound());
    setMaxScore(results.getMaxScore());

    int index = 0;
    // iterate over result documents
    for (SolrDocument document : results) {
      final AnyMap item = convertSolrDocToSmilaResultItemStub(document);
      getResultRecords().add(item);
      addResultFieldsToItem(item, document);
      processHighlightingResponse(highlighting, item);
      if (moreLikeThisLists != null) {
        // for each normal result item there exists at the same index a List of relatedDocs (might be empty though)
        final SolrDocumentList relatedDocs = moreLikeThisLists.getVal(index);
        addMoreLikeThisToItem(item, relatedDocs);
      }
      index++;
    }
  }

  /**
   * @return get MoreLikeThis lists from response, if exists and has the current type, otherwise it returns null.
   */
  @SuppressWarnings("unchecked")
  private SimpleOrderedMap<SolrDocumentList> getMoreLikeThisLists() {
    final Object mltObject = _response.getResponse().get(SolrConstants.MORE_LIKE_THIS);
    if (mltObject != null) {
      if (mltObject instanceof SimpleOrderedMap) {
        return (SimpleOrderedMap<SolrDocumentList>) mltObject;
      } else if (_log.isDebugEnabled()) {
        _log.debug(format("object %s is not a SimpleOrderedMap but of type %s", SolrConstants.MORE_LIKE_THIS,
          mltObject.getClass()));
      }
    } else if (_log.isDebugEnabled()) {
      _log.debug(format("object %s does not exist in response", SolrConstants.MORE_LIKE_THIS));

    }
    return null;
  }

  /**
   * @param document
   * @return
   */
  private AnyMap convertSolrDocToSmilaResultItemStub(SolrDocument document) {
    // get document (record) id
    final String id = (String) document.getFieldValue(SolrConstants.CORE_FIELD_ID);

    // get score
    Double score = -1.0;
    final Object scoreObj = document.getFieldValue(SolrConstants.CORE_FIELD_SCORE);
    try {
      final Number scoreNumber = (Number) scoreObj;
      score = scoreNumber.doubleValue();
    } catch (Exception e) {
      if (_log.isWarnEnabled()) {
        final String msg;
        if (e instanceof NullPointerException) {
          msg = "No score value in the solr result for id: " + id;
        } else if (e instanceof ClassCastException) {
          msg =
            MessageFormat.format(
              "The score value returned from solr is not a Number for id: {0} -> score as string: {1}", id,
              scoreObj);
        } else {
          msg = "Cannot get score for id: " + id;
        }
        _log.warn(msg + ". Setting score to -1. Check your config to fix this.");
      }
    }

    // create the bare result item
    final AnyMap resultItem = getResultRecords().getFactory().createAnyMap();
    resultItem.put(Record.RECORD_ID, id);
    resultItem.put(SearchResultConstants.WEIGHT, score);
    return resultItem;
  }

  /**
   * 
   * adds the relatedDocs as nested Seq/Maps*
   * 
   * @param item
   * @param relatedDocs
   */
  private void addMoreLikeThisToItem(AnyMap item, SolrDocumentList relatedDocs) {

    // always add the numners for the MLT results
    final AnyMap map = item.getMap(SolrConstants.MLT_RESULT_META, true);
    map.put(SearchResultConstants.START, relatedDocs.getStart());
    map.put(SearchResultConstants.COUNT, relatedDocs.getNumFound());
    map.put(SolrConstants.MAX_SCORE, relatedDocs.getMaxScore());

    // always add this seq, even if it is empty!
    final AnySeq mltSeq = item.getSeq(SolrConstants.MLT_RESULT_ITEMS, true);
    for (SolrDocument solrMltDoc : relatedDocs) {
      final AnyMap mltSmilaItem = convertSolrDocToSmilaResultItemStub(solrMltDoc);
      addResultFieldsToItem(mltSmilaItem, solrMltDoc);
      mltSeq.add(mltSmilaItem);
    }

  }

  /**
   * Add a result field (fl) to the resilt item.
   * 
   * @param item
   *          the result item.
   * @param document
   *          the solr document.
   */
  private void addResultFieldsToItem(AnyMap item, final SolrDocument document) {
    for (final Entry<String, Object> entry : document.entrySet()) {
      try {
        final String key = entry.getKey();
        if (key.equals(SolrConstants.CORE_FIELD_ID) || key.equals(SolrConstants.CORE_FIELD_SCORE)) {
          continue;
        }

        final Object value = entry.getValue();
        if (value != null) {

          if (value instanceof Collection) {
            addMultiKeyValuePairToItem(key, value, item);
          } else {
            addSingleKeyValuePairToItem(key, value, item);
          }

        }

      } catch (Exception exception) {
        if (_log.isWarnEnabled()) {
          _log.warn("Error adding solr result to record item for field name: " + entry.getKey()
            + ". this field has been skipped.", exception);
        }
      }
    }
  }

  /**
   * Add a single key value pair to result item.
   * 
   * @param key
   *          the key.
   * @param value
   *          the value object.
   * @param item
   *          the result item.
   */
  private void addSingleKeyValuePairToItem(String key, Object value, AnyMap item) {
    final DataFactory factory = item.getFactory();
    final Value autoConvertValue = factory.autoConvertValue(value);
    item.put(key, autoConvertValue);
  }

  /**
   * Add a multi value key value pair to result item as a sequence.
   * 
   * @param key
   *          the key.
   * @param value
   *          the values as list.
   * @param item
   *          the result item.
   */
  private void addMultiKeyValuePairToItem(String key, Object value, AnyMap item) {
    final DataFactory factory = item.getFactory();
    final Collection<?> multiValues = (Collection<?>) value;
    final AnySeq seq = factory.createAnySeq();
    item.put(key, seq);

    for (final Object multiValue : multiValues) {
      final Value autoConvertValue = factory.autoConvertValue(multiValue);
      seq.add(autoConvertValue);
    }
  }

  /**
   * Process facet response.
   */
  private void processFacetResponse() {
    if (_response.getFacetFields() != null) {
      for (FacetField facet : _response.getFacetFields()) {
        addFacetFieldToRecord(facet);
      }
    }
    if (_response.getFacetDates() != null) {
      for (FacetField facet : _response.getFacetDates()) {
        addFacetFieldToRecord(facet);
      }
    }
    if (_response.getFacetQuery() != null) {
      final Map<String, AnySeq> facets = new HashMap<String, AnySeq>();
      for (Entry<String, Integer> entry : _response.getFacetQuery().entrySet()) {
        final String facetName = StringUtils.substringBeforeLast(entry.getKey(), "_");
        AnySeq facet = facets.get(facetName);
        if (facet == null) {
          facet = addFacet(facetName);
        }
        addFacetValue(facet, entry.getKey(), entry.getValue().longValue());
      }
    }
  }

  /**
   * Add a facet field to query record.
   * 
   * @param facet
   *          the facet field.
   */
  private void addFacetFieldToRecord(FacetField facet) {
    final List<Count> values = facet.getValues();
    if (values == null || values.size() < 1) {
      return;
    }
    final AnySeq facetSeq = addFacet(facet.getName());
    for (Count count : values) {
      addFacetValue(facetSeq, count.getName(), count.getCount());

      /* PERF: return the facet query only when when turned by a parameter| TM @ Jan 19, 2012 */
      // final String filterQuery = count.getAsFilterQuery();
      // final Value value = factory.autoConvertValue(filterQuery);
      // map.put(SolrConstants.FILTER_QUERY, value);
    }
    facetSeq.add(facet.getValueCount());
  }

  /**
   * Process terms response.
   */
  private void processTermsResponse() {
    final NamedList<?> terms = (NamedList<?>) _response.getResponse().get(SolrConstants.TERMS);
    if (terms == null) {
      return;
    }

    final AnyMap map = getTermsMap(true);
    final DataFactory factory = map.getFactory();

    for (int i = 0; i < terms.size(); i++) {
      final NamedList<?> field = (NamedList<?>) terms.getVal(i);

      for (int z = 0; z < field.size(); z++) {
        final String name = field.getName(z);
        final Object value = field.getVal(z);
        final Value autoConvertValue = factory.autoConvertValue(value);
        map.put(name, autoConvertValue);
      }
    }
  }

  /**
   * Process spellcheck result.
   */
  private void processSpellcheckResponse() {
    final SpellCheckResponse spellcheck = _response.getSpellCheckResponse();
    if (spellcheck == null) {
      return;
    }

    final Map<String, Suggestion> suggestions = spellcheck.getSuggestionMap();
    if (suggestions != null && suggestions.size() > 0) {
      final AnyMap smilaSpellResults = getSpellcheckMap(true);

      // add suggestions
      final AnyMap smilaMispellingsMap = smilaSpellResults.getMap(SpellcheckSolrConstants.SUGGESTIONS, true);
      for (final Entry<String, Suggestion> entry : suggestions.entrySet()) {
        final Suggestion suggestion = entry.getValue();
        if (suggestion != null) {
          final AnyMap smilaSpellResult = smilaMispellingsMap.getMap(entry.getKey(), true);
          smilaSpellResult.put(SpellcheckSolrConstants.NUM_FOUND, suggestion.getNumFound());
          smilaSpellResult.put(SpellcheckSolrConstants.START_OFFSET, suggestion.getStartOffset());
          smilaSpellResult.put(SpellcheckSolrConstants.END_OFFSET, suggestion.getEndOffset());
          smilaSpellResult.put(SpellcheckSolrConstants.ORIG_FREQ, suggestion.getOriginalFrequency());
          final AnyMap resultSuggestions = smilaSpellResult.getMap(SpellcheckSolrConstants.SUGGESTIONS, true);

          final List<String> alternatives = suggestion.getAlternatives();
          final List<Integer> alternativeFrequencies = suggestion.getAlternativeFrequencies();
          for (int i = 0; i < suggestion.getNumFound(); i++) {
            final AnyMap resultSuggestion = resultSuggestions.getMap(alternatives.get(i), true);
            if (alternativeFrequencies != null) {
              resultSuggestion.put(SpellcheckSolrConstants.FREQ, alternativeFrequencies.get(i));
            }
          }
        }
      }

      // add collation, if exists.
      final String collation = spellcheck.getCollatedResult();
      if (collation != null) {
        smilaSpellResults.put(SpellcheckSolrConstants.COLLATION, collation);
      }

    }

  }

  /**
   * Get the solr result map (_solr.result).
   * 
   * @return the solr result map.
   */
  AnyMap getSolrResultMap(Boolean create) {
    return getResult().getMetadata().getMap(SolrConstants.RESULT_MAP, create);
  }

  /**
   * Get the terms map.
   * 
   * @param create
   *          true to create maps, false otherwise.
   * @return the terms map.
   */
  private AnyMap getTermsMap(Boolean create) {
    return getSolrResultMap(create).getMap(SolrConstants.TERMS, create);
  }

  /**
   * Get the spellcheck map.
   * 
   * @param create
   *          true to create maps, false otherwise.
   * @return the spellcheckmap.
   */
  private AnyMap getSpellcheckMap(Boolean create) {
    return getSolrResultMap(create).getMap(SpellcheckSolrConstants.SPELLCHECK, create);
  }

  /**
   * Set max score.
   * 
   * @param max
   *          score.
   */
  private void setMaxScore(Number maxScore) {
    if (maxScore != null) {
      getSolrResultMap(true).put(SolrConstants.MAX_SCORE, maxScore);
    }
  }

  /**
   * Process Solrs group response.
   */
  private void processGroupResponse() {
    // get group response from Solr
    final GroupResponse groupResponse = _response.getGroupResponse();
    if (groupResponse == null) {
      return;
    }
    // Read the group results per command
    for (GroupCommand command : groupResponse.getValues()) {
      final String name = command.getName();
      // Create and add AnySeq to 'groups' within result record
      final AnySeq groupSeq = addGroup(name);
      final DataFactory factory = groupSeq.getFactory();
      // Iterate over all groups within command
      for (Group group : command.getValues()) {
        String value = group.getGroupValue();
        if (value == null) {
          // if a group has no name set to default one
          value = SolrConstants.GROUPBY_DEFAULT_IF_NULL;
        }
        final SolrDocumentList docList = group.getResult();
        final AnySeq results = factory.createAnySeq();
        // Iterate over all documents returned and add them to 'results' AnySeq
        for (SolrDocument doc : docList) {
          final AnyMap result = convertSolrDocToSmilaResultItemStub(doc);
          addResultFieldsToItem(result, doc);
          results.add(result);
        }
        final Long count = docList.getNumFound();
        addGroupResults(groupSeq, factory.autoConvertValue(value), count, results);
      }
    }
  }

  private void processHighlightingResponse(final Map<String, Map<String, List<String>>> highlighting,
    AnyMap resultItem) {
    if (highlighting == null) {
      return;
    }
    final String recordid = resultItem.getStringValue(Record.RECORD_ID);
    for (Entry<String, List<String>> entry : highlighting.get(recordid).entrySet()) {
      final String attribute = entry.getKey();
      final List<String> text = entry.getValue();
      if (text.size() > 1) {
        addHighlightText(resultItem, attribute, text);
      } else {
        addHighlightText(resultItem, attribute, text.get(0));
      }
    }
  }

}
