/*********************************************************************************************************************
 * Copyright (c) 2008, 2012 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
**********************************************************************************************************************/
package org.eclipse.smila.taskworker.output;

import java.io.IOException;
import java.util.Map;

import org.eclipse.smila.datamodel.Any;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.datamodel.ipc.IpcAnyWriter;
import org.eclipse.smila.datamodel.ipc.IpcRecordWriter;
import org.eclipse.smila.objectstore.ObjectStoreException;
import org.eclipse.smila.objectstore.ObjectStoreService;
import org.eclipse.smila.taskmanager.BulkInfo;
import org.eclipse.smila.taskworker.util.Counters;

/**
 * Similar to {@link org.eclipse.smila.taskworker.output.StreamOutput} and
 * {@link org.eclipse.smila.taskworker.output.RecordOutput} but uses the Append API to create the result object. Works
 * only if the output bulk info contains a valid Appendable ID.
 * 
 * @author cind01
 */
public class AppendableOutput extends Output {
  /** for writing Any objects to BON. */
  private final IpcAnyWriter _ipcAnyWriter = new IpcAnyWriter();

  /** for writing records to BON. */
  private final IpcRecordWriter _ipcRecordWriter = new IpcRecordWriter();

  /** size of values written. */
  private long _bytesWritten;

  /** number of records appended. */
  private long _recordCount;

  /**
   * @param dataObject
   *          The data object
   * @param objectStore
   *          The reference to the object store service
   */
  public AppendableOutput(final BulkInfo dataObject, final ObjectStoreService objectStore) {
    super(dataObject, objectStore);
  }

  /** append bytes. */
  public void append(final byte[] data) throws ObjectStoreException {
    append(data, null);
  }

  /** append bytes. */
  public void append(final byte[] data, final Integer numberOfRecords) throws ObjectStoreException {
    final long start = startTime();
    _objectStore.appendToObject(getStoreName(), getObjectName(), data);
    if (numberOfRecords != null) {
      _recordCount += numberOfRecords;
    }
    _bytesWritten += data.length;
    timePerform(start);
  }

  /** append a {@link Record} converted to BON to the data object. */
  public void appendRecord(final Record record) throws ObjectStoreException, IOException {
    append(_ipcRecordWriter.writeBinaryObject(record), 1);
  }

  /** append an {@link Any} record converted to BON to the data object. */
  public void appendAny(final Any object) throws ObjectStoreException, IOException {
    append(_ipcAnyWriter.writeBinaryObject(object), 1);
  }

  /** finalize the Appendable object. Further appending will not be possible afterwards. */
  @Override
  public void commit() throws ObjectStoreException {
    _objectStore.finishObject(getStoreName(), getObjectName());
  }

  /** {@inheritDoc} */
  @Override
  public void abort() throws ObjectStoreException {
    _objectStore.finishObject(getStoreName(), getObjectName());
  }

  /** @return size of data written. */
  public long getBytesWritten() {
    return _bytesWritten;
  }

  /** number of Records (or Anys) appended. */
  public long getRecordCount() {
    return _recordCount;
  }

  /** @return counter map of superclass extended by the number of records written to the stream. */
  @Override
  public Map<String, Number> getCounter() {
    final Map<String, Number> counter = super.getCounter();
    Counters.add(counter, "size", _bytesWritten);
    if (_recordCount > 0) {
      Counters.add(counter, "recordCount", _recordCount);
    }
    return counter;
  }
}
