package org.eclipse.smila.importing.crawler.web.utils;

import java.util.UUID;

import org.eclipse.smila.datamodel.Any;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.importing.ImportingConstants;
import org.eclipse.smila.importing.crawler.web.WebCrawlerConstants;
import org.eclipse.smila.utils.digest.DigestHelper;

/** helper class for calculating the delta hash attribute for records produced by the web-crawler. */
public final class DeltaHash {
  /** prevent instance creation. */
  private DeltaHash() {
  }

  /**
   * determine delta hash: if content is attached, calculate a digest. Else use the lastModified date reported by the
   * webserver. If none is present, use size (and content-type) reported by server. Finally, if size is not set,
   * generate a UUID as deltahash to force updating.
   * 
   * <p>
   * <b>Please note that the record's attributes must not have been mapped before calculating the hash!</b>
   * </p>
   */
  public static void calculate(final Record record) {
    final AnyMap metadata = record.getMetadata();
    String deltaHash = null;
    if (record.hasAttachment(WebCrawlerConstants.ATTACHMENT_CONTENT)) {
      deltaHash = DigestHelper.calculateDigest(record.getAttachmentAsBytes(WebCrawlerConstants.ATTACHMENT_CONTENT));
    } else {
      Any lastModified = null;
      lastModified = metadata.get(WebCrawlerConstants.ATTRIBUTE_LASTMODIFIED);
      if (lastModified != null) {
        deltaHash = lastModified.toString();
      } else {
        Any contentLength = null;
        contentLength = metadata.get(WebCrawlerConstants.ATTRIBUTE_SIZE);
        if (contentLength != null) {
          deltaHash = contentLength.toString();
          deltaHash += "-" + metadata.get(WebCrawlerConstants.ATTRIBUTE_CONTENTTYPE);
        } else {
          // no information suitable for calculating delta found -> force update.
          deltaHash = UUID.randomUUID().toString();
        }
      }
    }
    if (deltaHash != null) {
      metadata.put(ImportingConstants.ATTRIBUTE_DELTA_HASH, deltaHash);
    }
  }

}
