/*******************************************************************************
 * Copyright (c) 2008, 2012 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Juergen Schumacher (Attensity Europe GmbH) - initial API and implementation
 *******************************************************************************/
package org.eclipse.smila.http.client.impl.base;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.util.Iterator;
import java.util.NoSuchElementException;

import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.ipc.IpcAnyReader;
import org.eclipse.smila.http.client.BulkResponse;

/**
 * default implementation of {@link BulkResponse}.
 */
public class BulkResponseImpl implements BulkResponse {

  /** for reading the input stream line by line. */
  private BufferedReader _lineReader;

  /** current line for parsing. */
  private String _jsonLine;

  /** JSON-to-any converter. */
  private final IpcAnyReader _jsonReader = new IpcAnyReader();

  /** create instance. */
  public BulkResponseImpl(final InputStream responseStream) throws IOException {
    _lineReader = new BufferedReader(new InputStreamReader(responseStream, HttpRequestFactory.ENCODING));
  }

  @Override
  public Iterator<AnyMap> iterator() {
    return this;
  }

  /**
   * Checks if there are more objects to read from the bulk. Can be called repeatedly.
   * 
   * @throws RuntimeException
   *           If reading the input stream fails. The IOException from the reader will be attached as the cause.
   */
  @Override
  public boolean hasNext() {
    if (_jsonLine == null && _lineReader != null) {
      try {
        _jsonLine = _lineReader.readLine();
      } catch (final IOException ex) {
        throw new RuntimeException("Failed to read another JSON line from bulk", ex);
      } finally {
        if (_jsonLine == null) {
          close();
        }
      }
    }
    return _jsonLine != null;
  }

  /**
   * Parses and returns the next object.
   * 
   * @throws NoSuchElementException
   *           If no objects are available anymore.
   * @throws RuntimeException
   *           If reading the input stream or parsing the JSON line fails. The exception from the reader or parser will
   *           be attached as the cause.
   */
  @Override
  public AnyMap next() {
    if (hasNext()) {
      try {
        final AnyMap nextObject = _jsonReader.readJsonObject(_jsonLine).asMap();
        _jsonLine = null;
        return nextObject;
      } catch (final Exception ex) {
        close();
        throw new RuntimeException("Failed to parse JSON line from bulk", ex);
      }
    }
    throw new NoSuchElementException("No more objects available in JSON bulk.");
  }

  /**
   * Currently not supported by this implementation.
   * 
   * @throws UnsupportedOperationException
   *           always.
   */
  @Override
  public void remove() {
    throw new UnsupportedOperationException();
  }

  /**
   * Closes used resources. Further usage is not possible afterwards.
   */
  @Override
  public void close() {
    if (_lineReader != null) {
      try {
        _lineReader.close();
      } catch (final Exception ex) {
        ; // ignore.
      }
      _lineReader = null;
      _jsonLine = null;
    }
  }

  @Override
  protected void finalize() throws Throwable {
    close();
    super.finalize();
  }

}
