/*******************************************************************************
 * Copyright (c) 2008, 2012 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Juergen Schumacher, Andreas Schank (both Attensity Europe GmbH) - initial API and implementation
 *******************************************************************************/
package org.eclipse.smila.http.client.impl.failover;

import java.util.Arrays;
import java.util.List;

import org.apache.http.conn.ClientConnectionManager;
import org.eclipse.smila.http.client.impl.DefaultHttpRequestFactory;
import org.eclipse.smila.http.client.impl.DefaultHttpResultHandler;
import org.eclipse.smila.http.client.impl.base.RestClientBase;
import org.eclipse.smila.http.client.util.HttpClientUtil;

/**
 * Alternative {@link org.eclipse.smila.http.client.RestClient} implementation that supports talking to a cluster of
 * SMILA services and can do failover to another node if the current node is not reachable anymore.
 * 
 * Create the client with a list of "host:port" strings. The client will talk to the first of those nodes as long as it
 * is reachable. Then it will failover to the second node. It will not switch back to the first node if it becomes
 * available again. It will also not distribute request to available nodes to provide load-balancing.
 * 
 * Restriction: if the request contains an InputStream to provide request content, then the request cannot be retried on
 * another node if the InputStream has been already read during the last failed attempt, though there may still be other
 * nodes available for retry. So if possible don't use stream requests if you require that all nodes are tried.
 */
public class FailoverRestClient extends RestClientBase {
  /**
   * Talk to the given SMILA servers.
   * 
   * @param hostsAndPorts
   *          list of hosts and ports of SMILA services, in format "host:port". Don't include a trailing slash ("/").
   */
  public FailoverRestClient(final String... hostsAndPorts) {
    this(Arrays.asList(hostsAndPorts));
  }

  /**
   * Talk to the given SMILA servers.
   * 
   * @param hostsAndPorts
   *          list of hosts and ports of SMILA services, in format "host:port". Don't include a trailing slash ("/").
   */
  public FailoverRestClient(final List<String> hostsAndPorts) {
    this(hostsAndPorts, HttpClientUtil.createThreadSafeConnectionManager());
  }

  /**
   * Talk to the given SMILA server, using a non-default connection configuration.
   * 
   * @param hostsAndPorts
   *          list of hosts and ports of SMILA services, in format "host:port". Don't include a trailing slash ("/").
   * @param maxTotalConnections
   *          maximum number of total HTTP connections
   * @param maxConnectionsPerHost
   *          maximum number of connections per host
   */
  public FailoverRestClient(final List<String> hostsAndPorts, final int maxTotalConnections,
    final int maxConnectionsPerHost) {
    this(hostsAndPorts, HttpClientUtil
      .createThreadSafeConnectionManager(maxTotalConnections, maxConnectionsPerHost));
  }

  /**
   * Talk to given SMILA server, using a given connection manager.
   * 
   * @param hostsAndPorts
   *          list of hosts and ports of SMILA services, in format "host:port". Don't include a trailing slash ("/").
   * @param connectionManager
   *          an initialized Apache HttpClient connection manager.
   */
  public FailoverRestClient(final List<String> hostsAndPorts, final ClientConnectionManager connectionManager) {
    super(connectionManager, new DefaultHttpRequestFactory(), new FailoverHttpRequestExecutor(hostsAndPorts),
      new DefaultHttpResultHandler());
  }
}
