/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Juergen Schumacher, Andreas Weber, Drazen Cindric, Andreas Schank (all Attensity Europe GmbH) - initial
 * implementation
 **********************************************************************************************************************/

package org.eclipse.smila.jobmanager.internal;

import java.util.Collection;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.jobmanager.JobRunDataProvider;
import org.eclipse.smila.jobmanager.JobRunInfo;
import org.eclipse.smila.jobmanager.definitions.BucketDefinition;
import org.eclipse.smila.jobmanager.definitions.DefinitionPersistence;
import org.eclipse.smila.jobmanager.definitions.JobDefinition;
import org.eclipse.smila.jobmanager.definitions.JobManagerConstants;
import org.eclipse.smila.jobmanager.definitions.JobRunDefinitions;
import org.eclipse.smila.jobmanager.definitions.WorkerDefinition;
import org.eclipse.smila.jobmanager.definitions.WorkflowDefinition;
import org.eclipse.smila.jobmanager.exceptions.ConfigNotFoundException;
import org.eclipse.smila.jobmanager.exceptions.JobManagerException;
import org.eclipse.smila.jobmanager.persistence.PermanentStorage;
import org.eclipse.smila.jobmanager.persistence.RunStorage;
import org.eclipse.smila.jobmanager.persistence.RunStorageException;

/**
 * Implements {@link JobRunDataProvider}.
 */
public class JobRunDataProviderImpl implements JobRunDataProvider {

  /** local logger. */
  private final Log _log = LogFactory.getLog(getClass());

  /** access to definitions in config area. */
  private DefinitionPersistence _defPersistence;

  /** service reference to permanent storage. */
  private PermanentStorage _permStorage;

  /** service reference to run storage. */
  private RunStorage _runStorage;

  /** {@inheritDoc} */
  @Override
  public AnyMap getJobRunData(final String jobName, final String jobId) throws JobManagerException {
    return getJobRunData(jobName, jobId, false);
  }

  /** {@inheritDoc} */
  @Override
  public AnyMap getJobRunData(final String jobName, final String jobId, final boolean returnDetails)
    throws JobManagerException {
    try {
      final String jobRunId = _runStorage.getJobRunId(jobName);
      if (jobId.equals(jobRunId)) {
        return _runStorage.getJobRunData(jobName, returnDetails);
      }
    } catch (final RunStorageException ex) {
      _log.warn("Error determining live job run data for job '" + jobName + "'. Trying to read from history.", ex);
    }
    if (_permStorage.containsJobRun(jobName, jobId)) {
      final AnyMap jobRunData = _permStorage.getJobRunData(jobName, jobId);
      if (!returnDetails && jobRunData != null) {
        jobRunData.remove(JobManagerConstants.DATA_JOB_RUN_WORKFLOW_DEF);
        jobRunData.remove(JobManagerConstants.DATA_JOB_RUN_BUCKET_DEFS);
      }
      return jobRunData;
    }
    throw new ConfigNotFoundException("No job run data available for run '" + jobId + "' of job '" + jobName + "'.");
  }

  @Override
  public AnyMap getWorkflowRunData(final String jobName, final String jobId, final String workflowRunId)
    throws JobManagerException {
    final String jobRunId = _runStorage.getJobRunId(jobName);
    if (jobId.equals(jobRunId)) {
      return _runStorage.getWorkflowRunData(jobName, workflowRunId);
    }
    throw new ConfigNotFoundException("No workflow run data available for run '" + jobId + "' of job '" + jobName
      + "'.");
  }

  @Override
  public JobRunInfo getJobRunInfo(String jobName) throws JobManagerException {
    return _runStorage.getJobRunInfo(jobName);
  }

  @Override
  public Collection<String> getCompletedJobRunIds(final String jobName) throws JobManagerException {
    final String currentjobRunId = _runStorage.getJobRunId(jobName);
    final Collection<String> completedJobRuns = _permStorage.getJobRunIds(jobName);
    if (currentjobRunId != null) {
      // there is a small timespan where job run is written to persistent storage but not yet deleted from RunStorage
      completedJobRuns.remove(currentjobRunId);
    }
    return completedJobRuns;
  }

  /**
   * Returns a job from the persistence definition.
   */
  private JobDefinition getJob(final String name) throws JobManagerException {
    final JobDefinition job = _defPersistence.getJob(name);
    if (job == null) {
      throw new ConfigNotFoundException("Job '" + name + "' not found.");
    }
    return job;
  }

  /**
   * Returns a workflow from the persistence definition.
   * 
   * @param name
   *          the name of the workflow
   * @return the workflow with the given name
   * @throws JobManagerException
   *           the workflow does not exist or an error occurred while accessing workflow definitions.
   */
  private WorkflowDefinition getWorkflow(final String name) throws JobManagerException {
    final WorkflowDefinition workflow = _defPersistence.getWorkflow(name);
    if (workflow == null) {
      throw new ConfigNotFoundException("Workflow '" + name + "' not found.");
    }
    return workflow;
  }

  /**
   * Returns a worker from the persistence definition.
   * 
   * @param name
   *          the name of the worker
   * @return the worker with the given name
   * @throws ConfigNotFoundException
   *           the worker does not exist.
   */
  public WorkerDefinition getWorker(final String name) throws ConfigNotFoundException {
    final WorkerDefinition worker = _defPersistence.getWorker(name);
    if (worker == null) {
      throw new ConfigNotFoundException("Worker '" + name + "' not found.");
    }
    return worker;
  }

  @Override
  public JobRunDefinitions getJobRunDefinitions(final String jobName) throws JobManagerException {
    final JobRunDefinitions runDefs = new JobRunDefinitions();
    final JobDefinition jobDef = getJob(jobName);
    runDefs.setJobDefinition(jobDef);
    final WorkflowDefinition workflowDef = getWorkflow(jobDef.getWorkflow());
    runDefs.setWorkflowDefinition(workflowDef);
    final Collection<String> workflowBuckets = workflowDef.getReferencedBuckets();
    final Collection<String> persistentBuckets = _defPersistence.getBuckets();
    persistentBuckets.retainAll(workflowBuckets); // intersection
    for (final String bucket : persistentBuckets) {
      final BucketDefinition bucketDef = _defPersistence.getBucket(bucket);
      runDefs.addBucketDefinition(bucketDef);
    }
    return runDefs;
  }

  /** set OSGI service. */
  public void setDefinitionPersistence(final DefinitionPersistence defPersistence) {
    _defPersistence = defPersistence;
  }

  /** unset OSGI service. */
  public void unsetDefinitionPersistence(final DefinitionPersistence defPersistence) {
    if (_defPersistence == defPersistence) {
      _defPersistence = null;
    }
  }

  /**
   * @param permStorage
   *          PermanentStorage reference.
   */
  public void setPermanentStorage(final PermanentStorage permStorage) {
    _permStorage = permStorage;
  }

  /**
   * @param permStorage
   *          PermanentStorage reference.
   */
  public void unsetPermanentStorage(final PermanentStorage permStorage) {
    if (_permStorage == permStorage) {
      _permStorage = null;
    }
  }

  /**
   * @param runStorage
   *          RunStorage reference.
   */
  public void setRunStorage(final RunStorage runStorage) {
    _runStorage = runStorage;
  }

  /**
   * @param runStorage
   *          RunStorage reference.
   */
  public void unsetRunStorage(final RunStorage runStorage) {
    if (_runStorage == runStorage) {
      _runStorage = null;
    }
  }

}
