/*******************************************************************************
 * Copyright (c) 2008, 2012 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Juergen Schumacher (empolis GmbH) - initial API and implementation Andreas Weber (Attensity Europe
 * GmbH) - removed processing services as BPEL pipeline extensions
 *******************************************************************************/

package org.eclipse.smila.processing.bpel.activities;

import java.util.HashMap;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.ode.bpel.common.FaultException;
import org.apache.ode.bpel.evar.ExternalVariableModuleException;
import org.apache.ode.bpel.rtrep.common.extension.AbstractExtensionBundle;
import org.apache.ode.bpel.rtrep.common.extension.ExtensionContext;
import org.apache.ode.utils.DOMUtils;
import org.eclipse.smila.blackboard.Blackboard;
import org.eclipse.smila.datamodel.xml.XmlSerializationUtils;
import org.eclipse.smila.ode.ODEServer;
import org.eclipse.smila.processing.Pipelet;
import org.eclipse.smila.processing.PipeletTrackerListener;
import org.eclipse.smila.processing.ProcessingException;
import org.eclipse.smila.processing.bpel.RequestTable;
import org.eclipse.smila.processing.bpel.counter.ProcessingPerformanceCounter;
import org.eclipse.smila.processing.bpel.internal.ExtensionBundleProvider;
import org.eclipse.smila.processing.bpel.util.MessageHelper;
import org.w3c.dom.Element;
import org.w3c.dom.Node;

/** Pipelet Manager and Invoker. */
public final class PipeletManager implements PipeletTrackerListener, ExtensionBundleProvider {

  /** Map of class names to currently active simple pipelet classes. */
  private final Map<String, Class<? extends Pipelet>> _activePipeletClasses =
    new HashMap<String, Class<? extends Pipelet>>();

  /** mapping pipelet holder keys to pipelet holders for lookup at invocation. */
  private final Map<String, PipeletHolder> _pipeletHolderMap = new HashMap<String, PipeletHolder>();

  /** table of currently executed requests. */
  private RequestTable _requests;

  /** local logger. */
  private final Log _log = LogFactory.getLog(getClass());

  @Override
  public AbstractExtensionBundle getExtensionBundle() {
    final PipeletExtensionBundle bundle = PipeletExtensionBundle.getInstance();
    bundle.setPipeletManager(this);
    return bundle;
  }

  /** invoke a pipelet. */
  public void invokeActivity(final String activityKey, final ExtensionContext context, final Element element) {
    final long startTime = System.nanoTime();
    ProcessingPerformanceCounter counter = null;
    boolean success = false;
    // this allows pipelets/services to call external webservices using the included org.apache.axis2 bundle.
    final ClassLoader tcclBackup = Thread.currentThread().getContextClassLoader();
    Thread.currentThread().setContextClassLoader(ODEServer.class.getClassLoader());
    try {
      final PipeletHolder pipeletHolder = _pipeletHolderMap.get(activityKey);
      if (pipeletHolder == null) {
        throw new ProcessingException("no registration found for key: " + activityKey);
      }
      counter = pipeletHolder.getCounter();
      invokePipelet(context, activityKey, pipeletHolder);
      context.complete();
      success = true;
    } catch (final Exception ex) {
      if (counter != null) {
        counter.addError(ex, false);
      }
      context.completeWithFault(ex);
    } finally {
      if (counter != null) {
        counter.countInvocationNanos(System.nanoTime() - startTime, success, 0, 0);
      }
      Thread.currentThread().setContextClassLoader(tcclBackup);
    }
  }

  /**
   * invoke pipelet holder to execute a pipelet.
   * 
   * @param context
   *          extension context
   * @param key
   *          activity key
   * @param pipeletHolder
   *          registered pipeletHolder
   * @param processor
   *          processor owning the process which contains the activity
   * @throws ProcessingException
   *           invocation failed.
   */
  private void invokePipelet(final ExtensionContext context, final String key, final PipeletHolder pipeletHolder)
    throws ProcessingException {
    String requestId = null;
    try {
      final String inputVariableName = pipeletHolder.getInputVariable();
      final String indexVariableName = pipeletHolder.getIndexVariable();
      String outputVariableName = pipeletHolder.getOutputVariable();
      if (outputVariableName == null && indexVariableName == null) {
        // use input variable automatically only if processing the complete input.
        outputVariableName = inputVariableName;
      }
      if (_log.isDebugEnabled()) {
        _log.debug(key + ": invoking " + pipeletHolder.getPrintName() + ", processing " + inputVariableName
          + " -> " + outputVariableName);
      }
      final Element inputVariable = (Element) context.readVariable(inputVariableName);
      final MessageHelper messageHelper = _requests.getMessageHelper();
      requestId = messageHelper.parseRequestId(inputVariable);
      Node indexVariable = null;
      if (indexVariableName != null) {
        indexVariable = context.readVariable(indexVariableName);
      }
      if (_log.isDebugEnabled()) {
        logVariable(key, "input", inputVariable);
        logVariable(key, "index", indexVariable);
      }
      checkAvailability(pipeletHolder);
      final Blackboard blackboard = _requests.getBlackboard(requestId);
      final String[] request = messageHelper.parseMessage(blackboard, inputVariable, indexVariable);
      final String[] result = doInvoke(pipeletHolder, blackboard, request);
      if (outputVariableName != null) {
        final Element outputVariable = messageHelper.createMessage(blackboard, result, requestId);
        if (_log.isDebugEnabled()) {
          logVariable(key, "output", outputVariable);
        }
        context.writeVariable(outputVariableName, outputVariable);
      }
    } catch (final Exception ex) {
      throw newProcessingException(ex, key, requestId);
    }
  }

  private void checkAvailability(final PipeletHolder pipeletHolder) throws ProcessingException {
    final Pipelet pipelet = pipeletHolder.getPipelet();
    if (pipelet == null) {
      throw new ProcessingException("Pipelet of class " + pipeletHolder.getClassName() + " for activity "
        + pipeletHolder.getKey() + " is not yet instantiated.");
    }
  }

  /** log variable value to debug log, if not null. */
  private void logVariable(final String key, final String name, final Node variable) {
    if (variable != null) {
      final String variableString = DOMUtils.domToString(variable);
      _log.debug(key + ": " + name + " = " + variableString);
    }
  }

  /**
   * actually invoke the pipelet.
   * 
   * @return Ids of result records
   * @throws ProcessingException
   *           error while processing.
   */
  private String[] doInvoke(final PipeletHolder pipeletHolder, final Blackboard blackboard, final String[] request)
    throws ProcessingException {
    int incomingIds = 0;
    int outgoingIds = 0;
    try {
      final Pipelet pipelet = pipeletHolder.getPipelet();
      incomingIds = getRecordCount(request);
      final String[] result = pipelet.process(blackboard, request);
      outgoingIds = getRecordCount(result);

      if (_log.isTraceEnabled()) {
        if (result == null) {
          _log.trace("piplet result records: none");
        } else {

          for (String id : result) {
            try {
              _log.trace("piplet result record: " + id
                + XmlSerializationUtils.serialize2string(blackboard.getRecord(id)));

            } catch (Exception e) {
              _log.trace("piplet result record: " + id + "error while serializing record");
            }
          } // for
        }
      } // if

      return result;
    } finally {
      if (pipeletHolder.getCounter() != null) {
        pipeletHolder.getCounter().countIds(incomingIds, outgoingIds);
      }
    }
  }

  /**
   * Counts the number of recordIds in a given String[], recordIds may be null.
   * 
   * @param recordIds
   *          the Id[]
   * @return the count
   */
  private int getRecordCount(final String[] recordIds) {
    if (recordIds != null) {
      return recordIds.length;
    }
    return 0;
  }

  /**
   * create ProcessingException from an exception thrown in pipeline element invocation. The exception is also stored in
   * the associated processor for later returning to the client, if possible.
   * 
   * @param cause
   *          exception thrown in the invocation
   * @param key
   *          pipeline element key.
   * @param requestId
   *          request id (may be null, if error occurs before it could be determined).
   * @param requests
   *          associated request manager
   * @return processing exception to throw
   */
  private ProcessingException newProcessingException(final Exception cause, final String key, final String requestId) {
    final String message;
    final Throwable causeToReport;
    if (cause instanceof ProcessingException) {
      message = "Invocation of pipeline element " + key + " failed due to pipelet error: " + cause.getMessage();
      causeToReport = cause.getCause() == null ? cause : cause.getCause();
    } else {
      if (cause instanceof FaultException || cause instanceof ExternalVariableModuleException) {
        message = "Invocation of pipeline element " + key + " failed due to BPEL variable access error.";
      } else {
        message = "Invocation of pipeline element " + key + " failed due to runtime error.";
      }
      causeToReport = cause;
    }
    final ProcessingException procEx = new ProcessingException(message, causeToReport);
    if (requestId == null) {
      // we cannot report this exception at the end of the pipeline invocation to the caller, so log it here.
      _log.warn(message, causeToReport);
    } else {
      _requests.setPipeletException(requestId, procEx);
    }
    return procEx;
  }

  /** register a pipelet invocation. */
  public void registerActivity(final PipeletHolder pipelet) throws ProcessingException {
    final String activityKey = pipelet.getKey();
    if (_log.isInfoEnabled()) {
      _log.info(activityKey + ": found " + pipelet.getPrintName() + ", processing " + pipelet.getInputVariable()
        + " -> " + pipelet.getOutputVariable()
        + (pipelet.getIndexVariable() == null ? "" : ", using loop variable " + pipelet.getIndexVariable()));
      if (pipelet.getConfiguration() == null) {
        _log.info(activityKey + ": no pipelet configuration found.");
      } else {
        _log.info(activityKey + ": pipelet configuration parsed.");
      }
    }
    initPipeletInstance(pipelet);
    _pipeletHolderMap.put(activityKey, pipelet);
  }

  /**
   * initialize a pipelet instance.
   * 
   * @param instance
   *          instance.
   * @throws ProcessingException
   *           error during initialization.
   */
  private void initPipeletInstance(final PipeletHolder instance) throws ProcessingException {
    if (instance.getPipelet() == null) {
      final String className = instance.getClassName();
      if (_activePipeletClasses.containsKey(className)) {
        final Class<? extends Pipelet> pipeletClass = _activePipeletClasses.get(className);
        try {
          final Pipelet pipelet = pipeletClass.newInstance();
          if (instance.getConfiguration() != null) {
            // added by jschumacher, 2009-03-23
            // this allows pipelets/services to init clients to external webservices using the
            // included org.apache.axis2 bundle.
            final ClassLoader tcclBackup = Thread.currentThread().getContextClassLoader();
            Thread.currentThread().setContextClassLoader(ODEServer.class.getClassLoader());
            try {
              pipelet.configure(instance.getConfiguration());
            } finally {
              Thread.currentThread().setContextClassLoader(tcclBackup);
            }
          }
          instance.setPipelet(pipelet);
        } catch (final InstantiationException e) {
          throw new ProcessingException("error instantiating pipelet class " + className, e);
        } catch (final IllegalAccessException e) {
          throw new ProcessingException("error instantiating pipelet class " + className, e);
        }
      }
    }
  }

  /**
   * learn about new pipelet classes and instantiate pipelets waiting for their classes.
   * 
   * {@inheritDoc}
   * 
   * @see org.eclipse.smila.processing.PipeletTrackerListener#pipeletsAdded(java.util.Map)
   */
  @Override
  public void pipeletsAdded(final Map<String, Class<? extends Pipelet>> pipeletClasses) {
    _log.info("Pipelets have been added: " + pipeletClasses.keySet());
    _activePipeletClasses.putAll(pipeletClasses);
    for (final PipeletHolder holder : _pipeletHolderMap.values()) {
      try {
        initPipeletInstance(holder);
      } catch (final ProcessingException ex) {
        _log.error("error when initializing pending pipelet", ex);
      }
    }
  }

  /**
   * forget pipelet classes and remove instances.
   * 
   * {@inheritDoc}
   * 
   * @see org.eclipse.smila.processing.PipeletTrackerListener#pipeletsRemoved(java.util.Map)
   */
  @Override
  public void pipeletsRemoved(final Map<String, Class<? extends Pipelet>> pipeletClasses) {
    _log.info("Pipelets have been removed: " + pipeletClasses.keySet());
    for (final PipeletHolder holder : _pipeletHolderMap.values()) {
      if (pipeletClasses.containsKey(holder.getClassName())) {
        holder.setPipelet(null);
      }
    }
    for (final String className : pipeletClasses.keySet()) {
      _activePipeletClasses.remove(className);
    }
  }

  /** method for DS to set a service reference. */
  public void setRequestTable(final RequestTable requests) {
    _requests = requests;
  }

  /** method for DS to unset a service reference. */
  public void unsetRequestTable(final RequestTable requests) {
    if (_requests == requests) {
      _requests = null;
    }
  }
}
