/***********************************************************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Juergen Schumacher (Attensity Europe GmbH) - implementation
 **********************************************************************************************************************/
package org.eclipse.smila.processing.bpel.internal;

import java.io.File;
import java.io.FileFilter;
import java.io.IOException;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import javax.xml.namespace.QName;
import javax.xml.stream.XMLStreamException;

import org.apache.commons.io.FileUtils;
import org.apache.commons.io.filefilter.NotFileFilter;
import org.apache.commons.io.filefilter.WildcardFileFilter;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.ode.bpel.compiler.BpelC;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.ode.ODEServer;
import org.eclipse.smila.processing.ProcessingException;
import org.eclipse.smila.processing.WorkflowProcessor;
import org.eclipse.smila.processing.bpel.util.BpelConstants;
import org.eclipse.smila.processing.bpel.util.ConfigurationHelper;
import org.eclipse.smila.utils.config.ConfigUtils;
import org.eclipse.smila.utils.workspace.WorkspaceHelper;

/**
 * manages deployment directories for SMILA BPEL workflows and control deployment processes in {@link ODEServer}.
 */
public class DeploymentManager {

  /** name of deployment descriptor file. */
  private static final String DEPLOY_XML = "deploy.xml";

  /** name of SMILA processing WSDL file. */
  private static final String PROCESSOR_WSDL = "processor.wsdl";

  /** name of SMILA record XML schema. */
  private static final String RECORD_XSD = "record.xsd";

  /** placeholder in deploy.xml template for the workflow name. */
  private static final String DEPLOY_NAME_PLACEHOLDER = "__NAME__";

  /** string to replace in deployment descriptor by an invoke element. */
  private static final String ENDTAG_PROCESS = "  </process>";

  /** format string to create an invoke element for the deployment descriptor. */
  private static final String FORMAT_DEPLOY_INVOKE = "    <invoke partnerLink='%1$s\'>\n" //
    + "      <service name='proc:%1$s' port='ProcessorPort' />\n" //
    + "    </invoke>\n";

  /** file name extension for BPEL files. */
  private static final String BPEL_EXTENSION = ".bpel";

  /** local logger. */
  private final Log _log = LogFactory.getLog(getClass());

  /** map associating predefined workflow names to the diretories containing their deployment files. */
  private final Map<String, File> _predefinedWorkflowDirectories = new HashMap<String, File>();

  /** map associating custom workflow names to the diretories containing their deployment files. */
  private final Map<String, File> _customWorkflowDirectories = new HashMap<String, File>();

  /** for creating prefixes for custom workflow deployment directories in workspace. */
  private long _deployCount;

  /** BPEL compiler used for validating BPEL workflows. */
  private final BpelC _bpelCompiler = BpelC.newBpelCompiler();

  /** file name filter for BPEL files. */
  private final FileFilter _bpelFilter = new FileFilter() {
    @Override
    public boolean accept(final File path) {
      return path.getName().endsWith(BPEL_EXTENSION) && path.isFile();
    }
  };

  /** create instance for given ODEServer. */
  public DeploymentManager() {
    _bpelCompiler.setDryRun(true); // do not generate compiled output during validation
  }

  /**
   * deploy a set of predefined workflows from a directory in the configuration area.
   * 
   * @throws IOException
   *           error copying the file.
   * @throws ProcessingException
   *           failed to deploy the directory in the server, probably because some definitions were not valid.
   */
  public synchronized Collection<QName> deployPredefinedWorkflows(final String configurationDirectoryName,
    final ODEServer server) throws ProcessingException, IOException {
    final File configurationDirectory =
      ConfigUtils.getConfigFile(ConfigurationHelper.BUNDLE_NAME, configurationDirectoryName);
    final File deploymentDirectory = copyToWorkspace(configurationDirectory);
    final Collection<QName> processes;
    try {
      processes = server.deploy(deploymentDirectory);
    } catch (final RuntimeException ex) {
      throw new ProcessingException("Predefined workflow directory '" + configurationDirectoryName
        + "' contains invalid definitions, no workflow from this directory will be available.", ex);
    }
    for (final QName processName : processes) {
      _predefinedWorkflowDirectories.put(processName.getLocalPart(), deploymentDirectory);
    }
    return processes;
  }

  /**
   * @return target workspace directory.
   * @throws IOException
   */
  private File copyToWorkspace(final File configurationDirectory) throws IOException {
    final String directoryName = configurationDirectory.getName();
    final File workspaceDirectory =
      WorkspaceHelper.createWorkingDir(ConfigurationHelper.BUNDLE_NAME, directoryName);
    _log.info("Predefined workflow deploy directory is " + workspaceDirectory.getAbsolutePath());
    FileUtils.cleanDirectory(workspaceDirectory);
    FileUtils.copyDirectory(configurationDirectory, workspaceDirectory, new NotFileFilter(new WildcardFileFilter(
      ".*")));
    _log.info("Pipeline configuration directory has been copied to workspace successfully.");
    return workspaceDirectory;
  }

  /**
   * @return deploy directory created for validated workflow.
   * @throws ProcessingException
   *           if workflow definition is not valid.
   */
  public synchronized File validateWorkflow(final String workflowName, final AnyMap workflowDefinition)
    throws ProcessingException {
    boolean isWorkflowValid = false;
    final String bpelDefinition = workflowDefinition.getStringValue(WorkflowProcessor.WORKFLOW_DEFINITION);
    if (StringUtils.isEmpty(bpelDefinition)) {
      throw new ProcessingException("Workflow '" + workflowName + "' can not be deployed, "
        + WorkflowProcessor.WORKFLOW_DEFINITION + " is empty");
    }
    final File deploymentDirectory = createDeploymentDirectory(workflowName, bpelDefinition);
    try {
      final File bpelFile = deploymentDirectory.listFiles(_bpelFilter)[0];
      _bpelCompiler.compile(bpelFile);
      isWorkflowValid = true;
      return deploymentDirectory;
    } catch (final Exception e) {
      throw new ProcessingException("Error while validating workflow '" + workflowName + "'", e);
    } finally {
      if (!isWorkflowValid) {
        deleteDeploymentDirectory(workflowName, deploymentDirectory);
      }
    }
  }

  /**
   * deploy a single custom workflow. The map must contain the workflow name as a single string value for key "name" and
   * BPEL XML definition as a single string value for key "definition". If a custom workflow with the same name exists
   * already, the old version is undeployed after successful deployment, and the old deployment directory in workspace
   * is removed.
   */
  public synchronized QName deployWorkflow(final String workflowName, final AnyMap workflowDefinition,
    final ODEServer server) throws ProcessingException {
    final String bpelDefinition = workflowDefinition.getStringValue(WorkflowProcessor.WORKFLOW_DEFINITION);
    final File deploymentDirectory = createDeploymentDirectory(workflowName, bpelDefinition);
    return deployWorkflowDir(workflowName, deploymentDirectory, server);
  }

  /**
   * deploy a single custom workflow located in the given deployment directory.
   */
  public synchronized QName deployWorkflowDir(final String workflowName, final File deploymentDirectory,
    final ODEServer server) throws ProcessingException {
    Collection<QName> processes = null;
    Iterator<QName> processIter = null;
    if (isCustomWorkflow(workflowName)) {
      // workflow already defined -> update
      final File undeploymentDirectory = _customWorkflowDirectories.get(workflowName);
      try {
        processes = server.redeploy(deploymentDirectory, undeploymentDirectory);
      } catch (final RuntimeException ex) {
        throw new ProcessingException("Custom workflow '" + workflowName + "' could not be updated.", ex);
      }
      deleteDeploymentDirectory(workflowName, undeploymentDirectory);
    } else {
      // workflow not defined yet -> add
      try {
        processes = server.deploy(deploymentDirectory);
      } catch (final RuntimeException ex) {
        throw new ProcessingException("Custom workflow '" + workflowName
          + "' is not valid and could not be deployed .", ex);
      }
    }
    processIter = processes.iterator();
    if (!processIter.hasNext()) {
      throw new ProcessingException("No process was deployed. Check log for reasons.");
    }
    _customWorkflowDirectories.put(workflowName, deploymentDirectory);
    return processIter.next();
  }

  /**
   * create a new directory in workspace to deploy the given workflow definition from.
   * 
   * @throws ProcessingException
   *           error writing directory.
   */
  private File createDeploymentDirectory(final String workflowName, final String bpelDefinition)
    throws ProcessingException {
    boolean directoryCompleted = false;
    File deploymentDirectory = null;
    try {
      deploymentDirectory =
        WorkspaceHelper.createWorkingDir(ConfigurationHelper.BUNDLE_NAME, workflowName + "-" + ++_deployCount);
      writeFile(deploymentDirectory, workflowName + BPEL_EXTENSION, bpelDefinition);
      copyFileToWorkspace(deploymentDirectory, RECORD_XSD);
      copyFileToWorkspace(deploymentDirectory, PROCESSOR_WSDL);
      final String deployDescriptor = createDeployDescriptor(workflowName, bpelDefinition);
      writeFile(deploymentDirectory, DEPLOY_XML, deployDescriptor);
      directoryCompleted = true;
      return deploymentDirectory;
    } catch (final ProcessingException ex) {
      throw ex;
    } catch (final Exception ex) {
      throw new ProcessingException("Failed to create a workspace directory to deploy pipeline '" + workflowName
        + "'", ex);
    } finally {
      if (!directoryCompleted) {
        deleteDeploymentDirectory(workflowName, deploymentDirectory);
      }
    }
  }

  /** try to delete directory. */
  private void deleteDeploymentDirectory(final String workflowName, final File deploymentDirectory) {
    if (deploymentDirectory != null) {
      try {
        FileUtils.deleteDirectory(deploymentDirectory);
      } catch (final IOException ex) {
        _log.warn("Failed to delete directory '" + deploymentDirectory + "' for workflow '" + workflowName
          + "', ignoring.", ex);
      }
    }
  }

  /**
   * create deployment descriptor: read template, insert workflow name, scan BPEL definition for sub-pipeline calls and
   * add corresponding <code>&lt;invoke&gt;</code> elements to deployment descriptor.
   */
  private String createDeployDescriptor(final String workflowName, final String bpelDefinition) throws Exception {
    final String deployDescriptorTemplate = readConfigFile(DEPLOY_XML);
    String deployDescriptor = deployDescriptorTemplate.replaceAll(DEPLOY_NAME_PLACEHOLDER, workflowName);
    final BpelScanner scanner = new BpelScanner(bpelDefinition);
    try {
      while (scanner.findNextInvoke()) {
        if (BpelConstants.VALUE_OPERATION_PIPELINECALL.equals(scanner.getCurrentOperation())
          && scanner.getCurrentPortType().endsWith(BpelConstants.VALUE_PORTTYPE_PIPELINECALL)) {
          final String partnerLink = scanner.getCurrentPartnerLink();
          final String deployInvoke = String.format(FORMAT_DEPLOY_INVOKE, partnerLink);
          deployDescriptor = deployDescriptor.replace(ENDTAG_PROCESS, deployInvoke + ENDTAG_PROCESS);
        }
      }
    } catch (final XMLStreamException ex) {
      throw new ProcessingException("Definition of pipeline '" + workflowName + "' is not well-formed XML", ex);
    }
    final String processName = scanner.getProcessName();
    if (processName == null) {
      throw new ProcessingException("Name of BPEL process definition not found, this pipeline cannot be deployed.");
    }
    if (!workflowName.equals(processName)) {
      throw new ProcessingException("Name of BPEL process definition is '" + processName + "', not '"
        + workflowName + "' as expected.");
    }
    if (_log.isDebugEnabled()) {
      _log.debug("Deployment descriptor for '" + workflowName + "': " + deployDescriptor);
    }
    return deployDescriptor;
  }

  /** copy file from bundle's "xml" directory to a directory. */
  private void copyFileToWorkspace(final File directory, final String filename) throws IOException {
    final String fileContent = readConfigFile(filename);
    writeFile(directory, filename, fileContent);
  }

  /** read content of file in bundle's "xml" directory. */
  private String readConfigFile(final String filename) throws IOException {
    return ConfigUtils.getConfigContent(ConfigurationHelper.BUNDLE_NAME, "xml/" + filename);
  }

  /** write file in directory using UTF-8 as the string encoding. */
  private void writeFile(final File directory, final String filename, final String fileContent) throws IOException {
    final File file = new File(directory, filename);
    FileUtils.writeStringToFile(file, fileContent, "utf-8");
  }

  /**
   * undeploy custom workflow and remove deployment directory in workspace.
   * 
   * @throws ProcessingException
   *           trying to delete a predefined workflow or error undeploying from BPEL server.
   */
  public synchronized QName undeployWorkflow(final String workflowName, final ODEServer server)
    throws ProcessingException {
    if (workflowName == null) {
      throw new ProcessingException("Workflow name must not be null.");
    }
    if (isPredefinedWorkflow(workflowName)) {
      throw new ProcessingException("Cannot undeploy predefined workflow '" + workflowName + "'");
    }
    final File deploymentDirectory = _customWorkflowDirectories.get(workflowName);
    if (deploymentDirectory != null) {
      final Collection<QName> processNames;
      try {
        processNames = server.undeploy(deploymentDirectory);
      } catch (final RuntimeException ex) {
        throw new ProcessingException("Failed to undeploy workflow " + workflowName + " deployed from "
          + deploymentDirectory, ex);
      }
      try {
        FileUtils.deleteDirectory(deploymentDirectory);
      } catch (final IOException ex) {
        _log.warn("Failed to delete directory '" + deploymentDirectory + "' for undeployed workflow '"
          + workflowName + "', ignoring.", ex);
      }
      _customWorkflowDirectories.remove(workflowName);
      if (processNames != null && !processNames.isEmpty()) {
        return processNames.iterator().next();
      }
    }
    return null;
  }

  /**
   * @return true if the workflow is deployed, false if not.
   */
  public boolean isDeployedWorkflow(final String workflowName) {
    return isPredefinedWorkflow(workflowName) || isCustomWorkflow(workflowName);
  }

  /**
   * @return true if the workflow is a predefined workflow, false if it is a custom workflow or not deployed.
   */
  public boolean isPredefinedWorkflow(final String workflowName) {
    return _predefinedWorkflowDirectories.containsKey(workflowName);
  }

  /**
   * @return true if the workflow is a custom workflow, false if it is a predefined workflow or not deployed.
   */
  public boolean isCustomWorkflow(final String workflowName) {
    return _customWorkflowDirectories.containsKey(workflowName);
  }
}
