/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Daniel Stucky (empolis GmbH) - initial API and implementation Drazen Cindric (Attensity Europe GmbH) -
 * data model improvements
 *******************************************************************************/
package org.eclipse.smila.processing.pipelets;

import java.util.regex.PatternSyntaxException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.smila.blackboard.Blackboard;
import org.eclipse.smila.blackboard.BlackboardAccessException;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.InvalidValueTypeException;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.processing.Pipelet;
import org.eclipse.smila.processing.ProcessingException;
import org.eclipse.smila.processing.parameters.MissingParameterException;
import org.eclipse.smila.processing.parameters.ParameterAccessor;
import org.eclipse.smila.processing.util.ResultCollector;

/**
 * <p>
 * Pipelet to filter records according to some expression.
 * </p>
 * <p>
 * If the configured expression <b>filterExpression</b> matches the value (simple String) in the configure metadata
 * attribute <b>filterAttribute</b> the record is written to the output. Otherwise the record is ignored. The
 * configuration has to be part of the "_parameters" attribute in each record or is contained in the pipeline
 * definition.
 * </p>
 * <p>
 * E.g.
 * 
 * <pre>
 *  "_parameters": {
 *         "filterAttribute": "_recordid",
 *         "filterExpression": "[0-5]{1}"
 *     }
 * </pre>
 * 
 * matches only records with the record id between 0 and 5.
 * </p>
 * 
 */
public class FilterPipelet implements Pipelet {
  /** filter attribute parameter. */
  private static final String FILTER_ATTRIBUTE_PARAMETER = "filterAttribute";

  /** filter expression parameter. */
  private static final String FILTER_EXPRESSION_PARAMETER = "filterExpression";

  /** local logger. */
  private final Log _log = LogFactory.getLog(getClass());

  /** static pipelet configuration from pipeline. */
  private AnyMap _configuration;

  /** {@inheritDoc} */
  @Override
  public void configure(final AnyMap configuration) throws ProcessingException {
    _configuration = configuration;
  }

  /** {@inheritDoc} */
  @Override
  public String[] process(final Blackboard blackboard, final String[] recordIds) throws ProcessingException {
    if (recordIds != null && recordIds.length > 0) {
      final ParameterAccessor parameters = new ParameterAccessor(blackboard, _configuration);
      final ResultCollector resultCollector = new ResultCollector(parameters, _log, true);
      for (final String recordId : recordIds) {
        try {
          final Record record = blackboard.getRecord(recordId);
          if (record != null) {
            parameters.setCurrentRecord(recordId);
            final String filterAttribute = parameters.getRequiredParameter(FILTER_ATTRIBUTE_PARAMETER);
            final String filterExpression = parameters.getRequiredParameter(FILTER_EXPRESSION_PARAMETER);
            if (recordMatches(record, filterAttribute, filterExpression)) {
              resultCollector.addResult(recordId);
            }
          } else if (_log.isDebugEnabled()) {
            _log.debug("Record " + recordId + " does not exist on blackboard.");
          }
        } catch (final BlackboardAccessException ex) {
          _log.warn("Record " + recordId + " not accessible", ex);
          resultCollector.addFailedResult(recordId, ex);
        } catch (final MissingParameterException ex) {
          _log.warn("Missing parameter for record " + recordId + ": " + ex);
          resultCollector.addFailedResult(recordId, ex);
        }
      }
      final String[] matchingRecords = resultCollector.getResultIds();
      final int matchingRecordsCount = matchingRecords.length;
      final int droppedRecordsCount = recordIds.length - matchingRecordsCount;
      if (_log.isDebugEnabled()) {
        _log.debug("Matched " + matchingRecordsCount + " records, dropped " + droppedRecordsCount + " records.");
      }
      return matchingRecords;
    }
    return recordIds;
  }

  /**
   * @return true if filterAttribute of record matches filterExpression, else false.
   */
  private boolean recordMatches(final Record record, final String filterAttribute, final String filterExpression) {
    try {
      final String valueToMatch = record.getMetadata().getStringValue(filterAttribute);
      if (valueToMatch != null) {
        if (valueToMatch.matches(filterExpression)) {
          if (_log.isDebugEnabled()) {
            _log.debug("Record " + record.getId() + ": " + filterAttribute + "=" + valueToMatch + " matches "
              + filterExpression);
          }
          return true;
        } else if (_log.isDebugEnabled()) {
          _log.debug("Record " + record.getId() + ": " + filterAttribute + "=" + valueToMatch + " does not match "
            + filterExpression);
        }
      } else if (_log.isDebugEnabled()) {
        _log.debug("Record " + record.getId() + " does not have attribute " + filterAttribute + " set.");
      }
    } catch (final InvalidValueTypeException ex) {
      if (_log.isDebugEnabled()) {
        _log.warn("Invalid value type in filter attribute " + filterAttribute + ": " + ex);
      }
    } catch (final PatternSyntaxException ex) {
      _log.warn("Invalid pattern " + filterExpression + " for matching record " + record.getId() + ": " + ex);
    }
    return false;
  }
}
