/**********************************************************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Andreas Weber (Attensity Europe GmbH) - initial implementation
 **********************************************************************************************************************/

package org.eclipse.smila.taskworker.output;

import java.io.IOException;
import java.util.Map;

import org.eclipse.smila.datamodel.Any;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.datamodel.ipc.IpcAnyWriter;
import org.eclipse.smila.datamodel.ipc.IpcRecordWriter;
import org.eclipse.smila.ipc.IpcFactory;
import org.eclipse.smila.ipc.IpcStreamWriter;
import org.eclipse.smila.ipc.bon.BinaryFactory;
import org.eclipse.smila.objectstore.ObjectStoreException;
import org.eclipse.smila.objectstore.ObjectStoreService;
import org.eclipse.smila.taskmanager.BulkInfo;
import org.eclipse.smila.taskworker.util.Counters;

/**
 * Provides simplified access for creating {@link Record} bulks by writing one {@link Record} (or {@link Any}) at a
 * time. You can also directly access the underlying {@link IpcStreamWriter} BON writer, but you should not mix up
 * direct access to the BON writer with the {@link #writeRecord(Record)} and {@link #writeAny(Any)} methods.
 */
public class RecordOutput extends Output {

  /** BON IPC factory. */
  private static final IpcFactory IPCFACTORY = new BinaryFactory();

  /** used to access the objectstore output stream. */
  private final StreamOutput _streamOutput;

  /** for writing Any objects to BON. */
  private final IpcAnyWriter _ipcAnyWriter = new IpcAnyWriter();

  /** for writing records to BON. */
  private final IpcRecordWriter _ipcRecordWriter = new IpcRecordWriter();

  /** for writing records. */
  private IpcStreamWriter _writer;

  /** records written. */
  private long _recordCount;

  /**
   * @param dataObject
   *          The data object
   * @param objectStore
   *          The reference to the object store service
   */
  public RecordOutput(final BulkInfo dataObject, final ObjectStoreService objectStore) {
    super(dataObject, objectStore);
    _streamOutput = new StreamOutput(dataObject, objectStore);
  }

  /**
   * @param record
   *          The record to write
   */
  public void writeRecord(final Record record) throws ObjectStoreException, IOException {
    final IpcStreamWriter writer = getStreamWriter();
    final long start = startTime();
    _ipcRecordWriter.writeStream(record, writer);
    _streamOutput.getStream().flush();
    timePerform(start);
    _recordCount++;
  }

  /**
   * @param any
   *          The object to write
   */
  public void writeAny(final Any any) throws ObjectStoreException, IOException {
    final IpcStreamWriter writer = getStreamWriter();
    final long start = startTime();
    _ipcAnyWriter.writeStream(any, writer);
    _streamOutput.getStream().flush();
    timePerform(start);
    _recordCount++;
  }

  /**
   * Commit the data object to objectstore. Called by the WorkerManager after the task was finished.
   */
  @Override
  public void commit() throws ObjectStoreException, IOException {
    if (_writer != null) {
      _writer.closeWithoutStream();
    }
    _streamOutput.commit();
  }

  /** {@inheritDoc} */
  @Override
  public void abort() throws IOException {
    if (_writer != null) {
      _writer.closeWithoutStream();
    }
    _streamOutput.abort();
  }

  /**
   * @return the stream writer
   */
  public IpcStreamWriter getStreamWriter() throws ObjectStoreException, IOException {
    if (_writer == null) {
      final long start = startTime();
      _writer = IPCFACTORY.newStreamWriter(_streamOutput.getStream());
      timeOpen(start);
    }
    return _writer;
  }

  /** @return number of records written so far using {@link #writeRecord()} */
  public long getRecordCount() {
    return _recordCount;
  }

  /**
   * @see StreamOutput#getBytesWritten()
   */
  public long getBytesWritten() {
    return _streamOutput.getBytesWritten();
  }

  /** @return counter map of superclass extended by the number of records written to the stream. */
  @Override
  public Map<String, Number> getCounter() {
    final Map<String, Number> counter = _streamOutput.getCounter();
    Counters.add(counter, "recordCount", _recordCount);
    return counter;
  }

}
