/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Juergen Schumacher, Andreas Weber, Drazen Cindric, Andreas Schank (all Attensity Europe GmbH) - initial
 * implementation
 **********************************************************************************************************************/
package org.eclipse.smila.bulkbuilder.helper;

import java.util.Iterator;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.smila.bulkbuilder.BulkbuilderException;
import org.eclipse.smila.bulkbuilder.InvalidJobException;
import org.eclipse.smila.jobmanager.JobRunDataProvider;
import org.eclipse.smila.jobmanager.JobRunInfo;
import org.eclipse.smila.jobmanager.JobState;
import org.eclipse.smila.jobmanager.JobTaskProcessor;
import org.eclipse.smila.jobmanager.exceptions.JobManagerException;
import org.eclipse.smila.taskmanager.ResultDescription;
import org.eclipse.smila.taskmanager.Task;
import org.eclipse.smila.taskmanager.TaskManager;
import org.eclipse.smila.taskmanager.TaskmanagerException;
import org.eclipse.smila.workermanager.WorkerManager;

/**
 * Initial task handling. Gets, caches, keeps-alive and finishes (initial) tasks by communicating with
 * taskmanager/jobmanager/workermanager.
 */
public class BulkbuilderTaskProvider {

  /** local Logger. */
  private final Log _log = LogFactory.getLog(getClass());

  /** map with job name and task. */
  private final Map<String, Task> _jobsAndTasks = new ConcurrentHashMap<String, Task>();

  /** the task manager to use for creating tasks. */
  private final TaskManager _taskManager;

  /** the workermanager is used for task keep alive. */
  private final WorkerManager _workerManager;

  /** the job run data provider to use for checking job state. */
  private final JobRunDataProvider _jobRunDataProvider;

  /** the job run task processor to get the initial task. */
  private final JobTaskProcessor _jobTaskProcessor;

  /** name of worker to get tasks for. */
  private final String _workerName;

  /** */
  public BulkbuilderTaskProvider(final TaskManager taskManager, final WorkerManager workerManager,
    final JobRunDataProvider jobDataProvider, final JobTaskProcessor jobTaskProcessor, final String workerName) {
    _taskManager = taskManager;
    _workerManager = workerManager;
    _jobRunDataProvider = jobDataProvider;
    _jobTaskProcessor = jobTaskProcessor;
    _workerName = workerName;
  }

  /**
   * Returns the initial task for bulk building.
   * 
   * @param jobName
   *          The job name
   * @return The initial task
   * @throws BulkbuilderException
   *           An exception if something goes wrong
   */
  public Task getInitialTask(final String jobName) throws BulkbuilderException {
    try {
      Task task = _jobsAndTasks.get(jobName);
      if (task == null) {
        synchronized (this) {
          task = _jobsAndTasks.get(jobName);
          if (task == null) {
            task = _jobTaskProcessor.getInitialTask(_workerName, jobName);
            if (task != null) {
              if (_log.isDebugEnabled()) {
                _log.debug("got new initial task: " + task.getTaskId());
              }
              _jobsAndTasks.put(jobName, task);
              _workerManager.addKeepAliveTask(task);
            } else {
              throw new BulkbuilderException("Couldn't get initial task for job " + jobName);
            }
          }
        }
      }
      return task;
    } catch (final JobManagerException te) {
      throw new InvalidJobException("Error getting initial task for job '" + jobName + "'.", te);
    }
  }

  /**
   * Finishes a task.
   */
  public Task finishTask(final String jobName, final ResultDescription resultDescription)
    throws BulkbuilderException {
    final Task task = _jobsAndTasks.remove(jobName);
    if (_log.isDebugEnabled()) {
      _log.debug("finishing initial task: " + task);
    }
    if (task != null) {
      try {
        _workerManager.removeKeepAliveTask(task);
        _taskManager.finishTask(_workerName, task.getTaskId(), resultDescription);
        if (_log.isTraceEnabled()) {
          _log.trace("Finished task for job '" + jobName + "'.");
        }
        return task;
      } catch (final TaskmanagerException te) {
        throw new BulkbuilderException(
          "Error finishing task '" + task.getTaskId() + "' for job '" + jobName + "'.", te);
      }
    } else if (_log.isInfoEnabled()) {
      _log.info("Could not find task to be finished for job '" + jobName + "'.");
    }
    return null;
  }

  /**
   * Removes a task.
   */
  public void removeTask(final Task task) {
    final Iterator<Task> iter = _jobsAndTasks.values().iterator();
    while (iter.hasNext()) {
      final Task foundTask = iter.next();
      if (foundTask.getTaskId().equals(task.getTaskId())) {
        iter.remove();
        return;
      }
    }
  }

  /**
   * 
   * @param jobName
   *          name of job
   * @throws BulkbuilderException
   *           if job is not in state RUNNING or FINISHING
   */
  public void checkJobActive(final String jobName) throws BulkbuilderException {
    try {
      final JobRunInfo jobRunInfo = _jobRunDataProvider.getJobRunInfo(jobName);
      if (jobRunInfo == null) {
        throw new InvalidJobException("No job run info for job '" + jobName + "', job not defined or not active.");
      }
      final JobState jobState = jobRunInfo.getState();
      if (jobState != JobState.RUNNING && jobState != JobState.FINISHING) {
        throw new InvalidJobException("Cannot commit in job '" + jobName + "' because it is in state '" + jobState
          + "'");
      }
    } catch (final JobManagerException ex) {
      throw new BulkbuilderException("Error checking state of job '" + jobName + "'", ex);
    }
  }

}
