/***********************************************************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH.  
 * All rights reserved. This program and the accompanying materials are made available
 * under the terms of the Eclipse Public License v1.0 which accompanies this distribution, 
 * and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Juergen Schumacher (empolis GmbH) - initial API and implementation
 *               Andreas Weber (Attensity Europe GmbH) - data model simplification
 **********************************************************************************************************************/

package org.eclipse.smila.datamodel.impl;

import java.io.Serializable;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.Map;

import org.eclipse.smila.datamodel.Any.ValueType;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.Attachment;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.datamodel.InMemoryAttachment;
import org.eclipse.smila.datamodel.Record;

/**
 * Default implementation of SMILA Records.
 * 
 * @author jschumacher
 * 
 */
public class RecordImpl implements Record, Serializable {

  /**
   * serializable, of course.
   */
  private static final long serialVersionUID = 1L;

  /**
   * metadata of record.
   */
  private final AnyMap _metadata = new AnyMapImpl();

  /**
   * attachments of record (used linked map to preserve order).
   */
  private final Map<String, Attachment> _attachments = new LinkedHashMap<String, Attachment>();

  /** default constructor to avoid external instantiation. */
  RecordImpl() {
  }

  @Override
  public String getId() {
    return _metadata.getStringValue(RECORD_ID);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getSource() {
    return _metadata.getStringValue(SOURCE);
  }

  @Override
  public AnyMap getMetadata() {
    return _metadata;
  }

  @Override
  public int attachmentSize() {
    return _attachments.size();
  }

  @Override
  public Attachment getAttachment(final String name) {
    return _attachments.get(name);
  }

  @Override
  public byte[] getAttachmentAsBytes(final String name) {
    if (hasAttachment(name)) {
      return _attachments.get(name).getAsBytes();
    }
    return null;
  }

  @Override
  public Iterator<String> getAttachmentNames() {
    return _attachments.keySet().iterator();
  }

  @Override
  public boolean hasAttachments() {
    return !_attachments.isEmpty();
  }

  @Override
  public boolean hasAttachment(final String name) {
    return _attachments.containsKey(name);
  }

  @Override
  public void removeAttachment(final String name) {
    final Attachment removed = _attachments.remove(name);
    if (removed != null) {
      removed.dispose();
    }
  }

  @Override
  public void removeAttachments() {
    for (final Attachment attachment : _attachments.values()) {
      attachment.dispose();
    }
    _attachments.clear();
  }

  @Override
  public void setAttachment(final Attachment attachment) {
    final Attachment oldAttachment = _attachments.put(attachment.getName(), attachment);
    if (oldAttachment != null) {
      oldAttachment.dispose();
    }
  }

  @Override
  public void setAttachment(final String name, final byte[] attachment) {
    _attachments.put(name, new InMemoryAttachment(name, attachment));
  }

  @Override
  public void setId(final String id) {
    if (id != null) {
      _metadata.put(RECORD_ID, new ValueImpl(ValueType.STRING, id));
    }
  }

  @Override
  public void setSource(final String source) {
    if (source != null) {
      _metadata.put(SOURCE, new ValueImpl(ValueType.STRING, source));
    }
  }

  @Override
  public String toString() {
    final StringBuilder builder = new StringBuilder();
    builder.append(_metadata);
    if (hasAttachments()) {
      builder.append(" + ").append(_attachments.keySet());
    }
    return builder.toString();
  }

  @Override
  public DataFactory getFactory() {
    return DefaultDataFactoryImpl.INSTANCE;
  }
}
