/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Juergen Schumacher (Attensity Europe GmbH) - initial API and implementation. Drazen Cindric (Attensity
 * Europe GmbH) - data model improvements
 *******************************************************************************/

package org.eclipse.smila.datamodel.xml;

import java.util.ArrayList;
import java.util.List;

import javax.xml.stream.XMLStreamException;
import javax.xml.stream.XMLStreamReader;

import org.eclipse.smila.datamodel.Any;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.AnySeq;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.datamodel.StoredAttachment;
import org.eclipse.smila.datamodel.Value;

/**
 * StAX based Id reader. Should give better performance than the DOM based IdParser.
 * 
 * @author jschumacher
 * 
 */
public class StaxRecordReader {

  /**
   * my object factory.
   */
  private final DataFactory _dataFactory;

  /**
   * create default instance.
   */
  public StaxRecordReader() {
    this(DataFactory.DEFAULT);
  }

  /**
   * @param dataFactory
   *          data factory to use.
   */
  public StaxRecordReader(final DataFactory dataFactory) {
    _dataFactory = dataFactory;
  }

  /**
   * read Record list from the XML stream. The stream must be currently at the RecordList start tag.
   * 
   * @param staxReader
   *          source XML stream
   * @return Record list read from stream or an empty list, if stream is not currently at a RecordList start tag.
   * @throws XMLStreamException
   *           StAX error.
   */
  public List<Record> readRecords(final XMLStreamReader staxReader) throws XMLStreamException {
    final List<Record> records = new ArrayList<Record>();
    if (isStartTag(staxReader, XmlConstants.TAG_RECORDLIST)) {
      staxReader.nextTag();
      while (isStartTag(staxReader, XmlConstants.TAG_RECORD)) {
        records.add(readRecord(staxReader));
        staxReader.nextTag();
      }
    }
    return records;
  }

  /**
   * read Record from the XML stream. The stream must be currently at the Record start tag.
   * 
   * @param staxReader
   *          source XML stream
   * @return Record read from stream or null, if stream is not currently at a Record start tag.
   * @throws XMLStreamException
   *           StAX error.
   */
  public Record readRecord(final XMLStreamReader staxReader) throws XMLStreamException {
    Record record = null;
    if (isStartTag(staxReader, XmlConstants.TAG_RECORD)) {
      record = _dataFactory.createRecord();
      readElements(staxReader, record.getMetadata());
      readAttachments(staxReader, record);
    }
    return record;
  }

  /**
   * @param value
   *          The value
   * @param type
   *          The value type
   * @return Instance of {@link Value}
   * @throws XMLStreamException
   *           An exception if something went wrong
   */
  private Value readValue(final String value, final String type) throws XMLStreamException {
    return _dataFactory.parseFromString(value, type);
  }

  /**
   * Reads a {@link Value} and adds or puts it into the given {@link Any} object.
   * 
   * @param staxReader
   *          The reader
   * @param container
   *          The {@link Any} object to add the value to.
   * @throws XMLStreamException
   *           An exception if something went wrong
   */
  private void readValue(final XMLStreamReader staxReader, final Any container) throws XMLStreamException {
    final String key = staxReader.getAttributeValue(null, XmlConstants.ATTRIBUTE_KEY);
    final String valueType = staxReader.getAttributeValue(null, XmlConstants.ATTRIBUTE_TYPE);
    final String content = staxReader.getElementText();
    final Value value = readValue(content, valueType);
    addToContainer(container, key, value);
  }

  /**
   * Reads an {@link AnySeq} and adds or puts it into the given {@link Any} object.
   * 
   * @param staxReader
   *          The reader
   * @param container
   *          The {@link Any} object to add the sequence to.
   * @throws XMLStreamException
   *           An exception if something went wrong
   */
  private void readSeq(final XMLStreamReader staxReader, final Any container) throws XMLStreamException {
    final String key = staxReader.getAttributeValue(null, XmlConstants.ATTRIBUTE_KEY);
    final AnySeq seq = _dataFactory.createAnySeq();
    readElements(staxReader, seq);
    addToContainer(container, key, seq);
  }

  /**
   * Reads an {@link AnyMap} and adds or puts it into the given {@link Any} object.
   * 
   * @param staxReader
   *          The reader
   * @param container
   *          The {@link Any} object to add the value to.
   * @throws XMLStreamException
   *           An exception if something went wrong
   */
  private void readMap(final XMLStreamReader staxReader, final Any container) throws XMLStreamException {
    final String key = staxReader.getAttributeValue(null, XmlConstants.ATTRIBUTE_KEY);
    final AnyMap map = _dataFactory.createAnyMap();
    readElements(staxReader, map);
    addToContainer(container, key, map);
  }

  /**
   * parse elements of a sequence or a map.
   */
  private void readElements(final XMLStreamReader staxReader, final Any container) throws XMLStreamException {
    staxReader.nextTag();
    while (isStartTag(staxReader, XmlConstants.TAG_VAL) || isStartTag(staxReader, XmlConstants.TAG_SEQ)
      || isStartTag(staxReader, XmlConstants.TAG_MAP)) {
      if (isStartTag(staxReader, XmlConstants.TAG_VAL)) {
        readValue(staxReader, container);
      } else if (isStartTag(staxReader, XmlConstants.TAG_SEQ)) {
        readSeq(staxReader, container);
      } else if (isStartTag(staxReader, XmlConstants.TAG_MAP)) {
        readMap(staxReader, container);
      }
      staxReader.nextTag();
    }
  }

  /**
   * read attachment names from the XML stream.
   * 
   * @param staxReader
   *          source XML stream param record Record to add the attachments to.
   * @param record
   *          record to add attachments too.
   * @throws XMLStreamException
   *           StAX error.
   */
  private void readAttachments(final XMLStreamReader staxReader, final Record record) throws XMLStreamException {
    while (isStartTag(staxReader, XmlConstants.TAG_ATTACHMENT)) {
      final String attachmentName = staxReader.getElementText();
      if (attachmentName != null && attachmentName.length() > 0) {
        record.setAttachment(new StoredAttachment(attachmentName));
      }
      staxReader.nextTag();
    }
  }

  /**
   * 
   * @param staxReader
   *          source XML stream
   * @param tagName
   *          tag name
   * @return true if we are currently at a start tag with the specificied name
   */
  private boolean isStartTag(final XMLStreamReader staxReader, final String tagName) {
    return staxReader.isStartElement() && tagName.equals(staxReader.getLocalName());
  }

  /**
   * add element to container.
   */
  private void addToContainer(final Any container, final String key, final Any element) {
    if (container.isMap() && key != null) {
      ((AnyMap) container).put(key, element);
    } else if (container.isSeq()) {
      ((AnySeq) container).add(element);
    }
  }

}
