/**********************************************************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Andreas Schank (Attensity Europe GmbH) - initial implementation
 **********************************************************************************************************************/
package org.eclipse.smila.processing.worker;

import java.util.Collection;
import java.util.LinkedHashSet;
import java.util.Map;

import org.eclipse.smila.blackboard.Blackboard;
import org.eclipse.smila.datamodel.Any;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.processing.ProcessingException;
import org.eclipse.smila.processing.WorkflowProcessor;
import org.eclipse.smila.processing.parameters.ParameterAccessor;
import org.eclipse.smila.taskworker.TaskContext;
import org.eclipse.smila.taskworker.input.RecordInput;
import org.eclipse.smila.taskworker.output.RecordOutput;

/**
 * A worker that is able to execute a pipeline.
 */
public class PipelineProcessorWorker extends ProcessingWorker {

  /** worker's name. */
  public static final String WORKER_NAME = "pipelineProcessor";

  /** key for the pipeline's name. */
  public static final String KEY_PIPELINE_NAME = "pipelineName";

  /** key for the number of parallel records for one execution of the pipeline. */
  public static final String KEY_PIPELINERUN_BULKSIZE = "pipelineRunBulkSize";

  /** default number of parallel records to be processed in one go. */
  public static final int DEFAULT_PIPELINERUN_BULKSIZE = 1;

  /** associated workflow processor. Set by DS. */
  private WorkflowProcessor _processor;

  /** @return {@value #WORKER_NAME} */
  @Override
  public String getName() {
    return WORKER_NAME;
  }

  /** {@inheritDoc} */
  @Override
  public boolean perform(final AnyMap parameters, final RecordInput recordInput, final RecordOutput recordOutput,
    final TaskContext taskContext) throws Exception {
    final long pipelineRunBulkSize = getPipelineRunBulkSize(parameters);
    final String pipelineName = getPipelineName(parameters);
    Record record = null;
    boolean success = false; // to check if at least some records were processed successful
    do {
      // process #pipelineRunBulkSize records at one go
      final Blackboard blackboard = getBlackboard(taskContext);
      try {
        final Collection<String> recordIds = new LinkedHashSet<String>();
        for (int i = 0; i < pipelineRunBulkSize; i++) {
          record = recordInput.getRecord();
          if (record == null) {
            break; // for
          }
          setTaskParameters(record, parameters);
          blackboard.setRecord(record);
          recordIds.add(record.getId());
        }
        if (!recordIds.isEmpty() && !taskContext.isCanceled()) {
          success |= processRecords(blackboard, recordIds, pipelineName, recordOutput, taskContext);
        }
      } finally {
        cleanupBlackboard(blackboard, taskContext);
      }
    } while (record != null && !taskContext.isCanceled());
    return success;
  }

  /** write task parameters to record. */
  private void setTaskParameters(final Record record, final AnyMap parameters) {
    final AnyMap parameterMap = record.getMetadata().getMap(ParameterAccessor.DEFAULT_PARAMETERS_ATTRIBUTE, true);
    for (final Map.Entry<String, Any> parameter : parameters.entrySet()) {
      parameterMap.put(parameter.getKey(), parameter.getValue());
    }
  }

  /**
   * process records. If processing throws a recoverable exception it is passed through so that the task can be retried
   * and may succeed then. Non-recoverable exceptions are catched and logged as warnings to the task log.
   * 
   * @return true, if processing was successful, false if a non-recoverable exception occured.
   */
  private boolean processRecords(final Blackboard blackboard, final Collection<String> recordIds,
    final String pipelineName, final RecordOutput recordOutput, final TaskContext taskContext) throws Exception {
    try {
      final String inputIds[] = recordIds.toArray(new String[recordIds.size()]);
      final String[] resultIds = _processor.process(pipelineName, blackboard, inputIds);
      writeResultRecords(blackboard, resultIds, recordOutput, taskContext);
      return true;
    } catch (final ProcessingException ex) {
      if (ex.isRecoverable()) {
        throw ex;
      } else {
        taskContext.getLog().warn("Failed to process records " + recordIds + ", skipping them.", ex);
        return false;
      }
    }
  }

  /**
   * @return value of pipelineName parameter.
   * @throws IllegalArgumentException
   *           if parameter is not set.
   */
  private String getPipelineName(final AnyMap parameters) {
    final String pipelineName = parameters.getStringValue(KEY_PIPELINE_NAME);
    if (pipelineName == null || pipelineName.isEmpty()) {
      throw new IllegalArgumentException("Pipeline name parameter '" + KEY_PIPELINE_NAME + "' is not set.");
    }
    if (!_processor.getWorkflowNames().contains(pipelineName)) {
      throw new IllegalArgumentException("Configured pipeline '" + pipelineName + "' doesn't exist.");
    }
    return pipelineName;
  }

  /**
   * @return value of {@value #KEY_PIPELINERUN_BULKSIZE} parameter, if set and grater than 0. Else default value
   *         {@link #DEFAULT_PIPELINERUN_BULKSIZE}.
   */
  private long getPipelineRunBulkSize(final AnyMap parameters) {
    final Long pipelineRunBulkSize = parameters.getLongValue(KEY_PIPELINERUN_BULKSIZE);
    if (pipelineRunBulkSize != null && pipelineRunBulkSize > 0) {
      return pipelineRunBulkSize;
    }
    return DEFAULT_PIPELINERUN_BULKSIZE;
  }

  /** set workflow processor reference (used by DS). */
  public void setProcessor(final WorkflowProcessor processor) {
    _processor = processor;
  }

  /** remove workflow processor reference (used by DS). */
  public void unsetProcessor(final WorkflowProcessor processor) {
    if (_processor == processor) {
      _processor = null;
    }
  }

}
