/*******************************************************************************
 * Copyright (c) 2008, 2012 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Juergen Schumacher, Andreas Schank (both Attensity Europe GmbH) - initial API and implementation
 *******************************************************************************/
package org.eclipse.smila.http.client.impl.failover;

import java.io.IOException;
import java.net.ConnectException;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.http.HttpEntityEnclosingRequest;
import org.apache.http.HttpHost;
import org.apache.http.HttpResponse;
import org.apache.http.NoHttpResponseException;
import org.apache.http.ProtocolException;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.client.utils.URIUtils;
import org.apache.http.impl.client.EntityEnclosingRequestWrapper;
import org.apache.http.impl.client.RequestWrapper;
import org.eclipse.smila.http.client.impl.base.HttpRequestExecutor;

/** request executor for the {@link FailoverRestClient}. */
public class FailoverHttpRequestExecutor implements HttpRequestExecutor {
  private final List<HttpHost> _httpHosts;

  private int _currentHostIndex;

  private final Log _log = LogFactory.getLog(getClass());

  /** create instance for given list of host-and-port strings. */
  public FailoverHttpRequestExecutor(final List<String> hostsAndPorts) {
    _httpHosts = convertHostAndPorts(hostsAndPorts);
  }

  /** convert host-and-port strings to instances of {@link HttpHost}. */
  private List<HttpHost> convertHostAndPorts(final List<String> hostsAndPorts) {
    final List<HttpHost> httpHosts = new ArrayList<HttpHost>(hostsAndPorts.size());
    for (String hostAndPort : hostsAndPorts) {
      if (hostAndPort.toLowerCase().startsWith("http://")) {
        hostAndPort = hostAndPort.substring("http://".length());
      }
      final String[] hostPortSplit = hostAndPort.split(":");
      final int port = Integer.parseInt(hostPortSplit[1]);
      httpHosts.add(new HttpHost(hostPortSplit[0], port));
    }
    return httpHosts;
  }

  @Override
  public String getHostAndPort() {
    final HttpHost currentHost = _httpHosts.get(_currentHostIndex);
    return currentHost.toString();
  }

  @Override
  public HttpResponse execute(final HttpClient client, final HttpUriRequest request) throws IOException {
    final RequestWrapper wrappedRequest = wrapRequest(request);
    int usedHostIndex = 0;
    while (wrappedRequest.getExecCount() < _httpHosts.size()) {
      try {
        usedHostIndex = setActualUri(wrappedRequest);
        wrappedRequest.incrementExecCount();
        return client.execute(wrappedRequest);
      } catch (final IOException ex) {
        handleIOException(wrappedRequest, usedHostIndex, ex);
      }
    }
    throw new NoHttpResponseException("No response received from any HTTP server for " + wrappedRequest.getURI());
  }

  /**
   * log exception, check if failover is possible and prepare it.
   * 
   * @throws ConnectException
   *           if request cannot be repeated.
   */
  private void handleIOException(final RequestWrapper wrappedRequest, final int usedHostIndex, final IOException ex)
    throws ConnectException {
    if (_log.isWarnEnabled()) {
      _log.warn("Request to " + wrappedRequest.getURI() + " failed", ex);
    }
    failoverToNextHost(usedHostIndex);
    if (!wrappedRequest.isRepeatable()) {
      throw new ConnectException("Request to " + wrappedRequest.getURI() + " is not repeatable anymore after "
        + ex.toString());
    }
  }

  /** wrap original request in a request wrapper to improve repeatability of stream requests. */
  private RequestWrapper wrapRequest(final HttpUriRequest request) {
    try {
      if (request instanceof HttpEntityEnclosingRequest) {
        return new EntityEnclosingRequestWrapper((HttpEntityEnclosingRequest) request);
      }
      return new RequestWrapper(request);
    } catch (final ProtocolException ex) {
      throw new IllegalArgumentException("Invalid request", ex);
    }
  }

  private int setActualUri(final RequestWrapper wrappedRequest) {
    final URI uri = wrappedRequest.getURI();
    try {
      final int actualHostIndex = _currentHostIndex;
      final URI actualUri = URIUtils.rewriteURI(uri, _httpHosts.get(actualHostIndex));
      wrappedRequest.setURI(actualUri);
      return actualHostIndex;
    } catch (final URISyntaxException ex) {
      throw new IllegalArgumentException("Cannot rewrite invalid URI", ex);
    }
  }

  /** change */
  private synchronized void failoverToNextHost(final int failedHostIndex) {
    if (failedHostIndex == _currentHostIndex) {
      if (++_currentHostIndex >= _httpHosts.size()) {
        _currentHostIndex = 0;
      }
      if (_log.isInfoEnabled()) {
        _log.info("Failover to " + getHostAndPort());
      }
    } else if (_log.isInfoEnabled()) {
      _log.info("Failover already done");
    }
  }

}
