package org.eclipse.smila.processing.pipelets;

import org.eclipse.smila.blackboard.Blackboard;
import org.eclipse.smila.blackboard.BlackboardAccessException;
import org.eclipse.smila.bulkbuilder.BulkbuilderService;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.processing.Pipelet;
import org.eclipse.smila.processing.ProcessingException;
import org.eclipse.smila.processing.parameters.ParameterAccessor;
import org.eclipse.smila.utils.service.ServiceUtils;

/**
 * A pipelet that pushs records to a job.
 * 
 * Copyright (c) 2012 Attensity Europe GmbH
 * 
 * @author Tobias Liefke
 */
public class PushRecordsPipelet implements Pipelet {

  /** Name of the property that contains the name of the target job. */
  public static final String JOB_NAME_PROPERTY = "job";

  /** The name of the target job. */
  private String _job;

  /** Reference to the BulkbuilderService. */
  private BulkbuilderService _bulkBuilder;

  /**
   * Initializes parameters.
   * 
   * @see Pipelet#configure(AnyMap)
   */
  @Override
  public void configure(AnyMap config) throws ProcessingException {
    final ParameterAccessor paramAccessor = new ParameterAccessor(config);
    _job = paramAccessor.getRequiredParameter(JOB_NAME_PROPERTY);
  }

  /**
   * @see Pipelet#process(Blackboard, String[])
   */
  @Override
  public String[] process(Blackboard blackboard, String[] recordIds) throws ProcessingException {
    // Load all records
    final Record[] records = new Record[recordIds.length];
    for (int i = 0; i < recordIds.length; i++) {
      try {
        records[i] = blackboard.getRecord(recordIds[i]);
      } catch (BlackboardAccessException e) {
        throw new ProcessingException(getClass() + ": Could not access record: " + recordIds[i], e);
      }
    }

    // Push all records
    try {
      for (final Record record : records) {
        getBulkbuilderService().addRecord(_job, record);
      }
    } catch (Exception e) {
      throw new ProcessingException(getClass() + ": Could not send all records to job: " + _job, e);
    }

    return recordIds;
  }

  /** Resolve the BulkbuilderService lazily. */
  private BulkbuilderService getBulkbuilderService() throws ProcessingException {
    if (_bulkBuilder == null) {
      try {
        _bulkBuilder = ServiceUtils.getService(BulkbuilderService.class);
      } catch (InterruptedException e) {
        throw new ProcessingException("Could not retrieve BulkbuilderService.", e);
      }
    }
    return _bulkBuilder;
  }

  /**
   * Sets the BulkbuilderService for testing purposes.
   * 
   * @param bulkBuilder
   *          The new BulkbuilderService to set.
   */
  public void setBulkbuilderService(BulkbuilderService bulkBuilder) {
    _bulkBuilder = bulkBuilder;
  }

}
