/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Andreas Weber (Attensity Europe GmbH) - initial implementation
 **********************************************************************************************************************/

package org.eclipse.smila.taskmanager;

import org.eclipse.smila.datamodel.Any;
import org.eclipse.smila.datamodel.AnyConvertException;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.DataFactory;


/**
 * Information about storage (store name and object id) for a bucket.
 */
public class BulkInfo {
  /** The key for the bucket. */
  public static final String KEY_BUCKET = "bucket";

  /** The key for the store. */
  public static final String KEY_STORE = "store";

  /** The key for the id. */
  public static final String KEY_ID = "id";

  /** The Bucket's name. */
  private final String _bucketName;

  /** The store name. */
  private final String _storeName;

  /** The object name. */
  private final String _objectName;

  /**
   * Constructs a new DataObject.
   * 
   * @param bucketName
   *          bucket name used in workflow definition
   * @param storeName
   *          name of bucket store
   * @param objectName
   *          name of object in store
   */
  public BulkInfo(final String bucketName, final String storeName, final String objectName) {
    super();
    _bucketName = bucketName;
    _storeName = storeName;
    _objectName = objectName;
  }

  /**
   * Converts an Any object to a BulkInfo object.
   * 
   * @param any
   *          the input as Any, expected to be an AnyMap
   * @return a BulkInfo constructed from bulkAny
   * @throws AnyConvertException
   *           an exception that occurred during parsing of the Any object.
   */
  public static BulkInfo fromAny(final Any any) throws AnyConvertException {
    if (!(any instanceof AnyMap)) {
      throw new AnyConvertException("Error parsing task from Any object that is no Map.");
    }
    try {
      final AnyMap bulkAny = (AnyMap) any;
      final String bucket = bulkAny.getStringValue(KEY_BUCKET);
      final String store = bulkAny.getStringValue(KEY_STORE);
      final String name = bulkAny.getStringValue(KEY_ID);
      return new BulkInfo(bucket, store, name);
    } catch (final Exception ex) {
      throw new AnyConvertException("Error parsing BulkInfo from Any object", ex);
    }
  }

  /**
   * @return the bucketDefinitionName
   */
  public String getBucketName() {
    return _bucketName;
  }

  /**
   * {@inheritDoc}
   */
  public String getStoreName() {
    return _storeName;
  }

  /**
   * {@inheritDoc}
   */
  public String getObjectName() {
    return _objectName;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String toString() {
    return "DataObject [_bucketName=" + _bucketName + ", _storeName=" + _storeName + ", _objectName=" + _objectName
      + "]";
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int hashCode() {
    final int prime = 31;
    int result = 1;
    if (_bucketName != null) {
      result = prime * result + _bucketName.hashCode();
    }
    if (_objectName != null) {
      result = prime * result + _objectName.hashCode();
    }
    if (_storeName != null) {
      result = prime * result + _storeName.hashCode();
    }
    return result;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean equals(final Object obj) {
    if (this == obj) {
      return true;
    }
    if (obj == null) {
      return false;
    }
    if (getClass() != obj.getClass()) {
      return false;
    }
    final BulkInfo other = (BulkInfo) obj;
    return equalsObjects(_bucketName, other._bucketName) && equalsObjects(_objectName, other._objectName)
      && equalsObjects(_storeName, other._storeName);
  }

  /**
   * Helper method for equals to reduce cyclomatic complexity.
   * 
   * @param one
   *          first object to compare
   * @param two
   *          second object to compare
   * @return 'true' if both objects are equal (or both are null), 'false' if not.
   */
  private static boolean equalsObjects(final Object one, final Object two) {
    if (one == null) {
      if (two != null) {
        return false;
      }
    } else if (!one.equals(two)) {
      return false;
    }
    return true;
  }

  /**
   * @return the BulkInfo as Any
   */
  public AnyMap toAny() {
    final AnyMap bulkInfoAny = DataFactory.DEFAULT.createAnyMap();
    bulkInfoAny.put(KEY_BUCKET, getBucketName());
    bulkInfoAny.put(KEY_STORE, getStoreName());
    bulkInfoAny.put(KEY_ID, getObjectName());
    return bulkInfoAny;
  }

}
