/***********************************************************************************************************************
 * Copyright (c) 2008 empolis GmbH and brox IT Solutions GmbH. All rights reserved. This program and the accompanying
 * materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this distribution,
 * and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Peter Wissel (brox IT Solutions GmbH) - initial API and implementation
 **********************************************************************************************************************/

package org.eclipse.smila.solr.search;

import org.apache.solr.common.params.CommonParams;
import org.eclipse.smila.blackboard.Blackboard;
import org.eclipse.smila.blackboard.BlackboardAccessException;
import org.eclipse.smila.datamodel.Any;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.AnySeq;
import org.eclipse.smila.datamodel.util.AnyUtil;
import org.eclipse.smila.search.api.helper.QueryParameterAccessor;
import org.eclipse.smila.solr.SolrConstants;

/**
 * The SolrQueryParameterAccessor.
 * 
 * @author pwissel
 * 
 */
public class SolrQueryParameterAccessor extends QueryParameterAccessor {

  /**
   * Constructor.
   * 
   * @param blackboard
   *          the blackboard.
   * @param queryRecordId
   *          the query record id.
   * @throws BlackboardAccessException
   */
  public SolrQueryParameterAccessor(final Blackboard blackboard, final String queryRecordId)
    throws BlackboardAccessException {
    super(blackboard, queryRecordId);
  }

  /**
   * Constructor.
   * 
   * @param blackboard
   *          the blackboard.
   * @param configuration
   *          the configuration.
   * @param queryRecordId
   *          the query record id.
   * @throws BlackboardAccessException
   */
  public SolrQueryParameterAccessor(final Blackboard blackboard, final AnyMap configuration,
    final String queryRecordId) throws BlackboardAccessException {
    super(blackboard, configuration, queryRecordId);
  }

  /**
   * Get request handler.
   * 
   * @return the request handler or null.
   */
  public String getRequestHandler() {
    return getSolrQueryParams().getStringValue(CommonParams.QT);
  }

  /**
   * Get terms.
   * 
   * @return the term map or null.
   */
  public AnyMap getTerms() {
    return getSolrQueryParams().getMap(SolrConstants.TERMS);
  }

  /**
   * Get filter query.
   * 
   * @return the filter query sequence or null.
   */
  public AnySeq getFilterQuery() {
    return getSolrQueryParams().getSeq(SolrConstants.FILTER_QUERY);
  }

  /**
   * Get shards.
   * 
   * @return the shards seq or null.
   */
  public AnySeq getShards() {
    return getSolrQueryParams().getSeq(SolrConstants.SHARDS);
  }

  /**
   * Get spellcheck.
   * 
   * @return the spellcheck map or null.
   */
  public AnyMap getSpellcheck() {
    return getSolrQueryParams().getMap(SpellcheckSolrConstants.SPELLCHECK);
  }

  public AnyMap getSolrQueryParams() {
    final Any _solrQueryParams = getParameterAny(SolrConstants.QUERY_MAP);
    return _solrQueryParams == null ? AnyUtil.EMPTY_MAP : _solrQueryParams.asMap();
  }

  /**
   * Get request handler.
   * 
   * @return the request handler or null.
   */
  public AnyMap getMoreLikeThis() {
    return getSolrQueryParams().getMap(SolrConstants.MORE_LIKE_THIS);
  }

  /**
   * returns the map containing the configs for the filter groups or null.
   */
  public AnyMap getFilterGroups() {
    return getSolrQueryParams().getMap(SolrConstants.FILTER_GROUPS);
  }

  /**
   * returns the localParams map from {@link SolrConstants#QUERY_MAP} for the requested filter group or null.
   */
  public AnyMap getFilterLocalParams() {
    final String filterGroup = SolrConstants.LOCAL_PARAMS;
    return getFilterLocalParams(filterGroup, false);
  }

  /**
   * returns the localParams map from {@link SolrConstants#QUERY_MAP} for the requested filter group or null.
   * 
   * @param filterGroup
   *          the filter group
   * @param create
   *          if true, then the intermediate maps are creared if missing.
   * @return the filter local params
   */
  public AnyMap getFilterLocalParams(final String filterGroup, boolean create) {
    final AnyMap filterGroups = getSolrQueryParams().getMap(SolrConstants.FILTER_GROUPS, create);
    if (filterGroups == null) {
      return null;
    }
    final AnyMap filterGroupConfig = filterGroups.getMap(filterGroup, create);
    if (filterGroupConfig == null) {
      return null;
    }
    return filterGroupConfig.getMap(SolrConstants.LOCAL_PARAMS, create);
  }

}
