/***********************************************************************************************************************
 * Copyright (c) 2008, 2012 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Juergen Schumacher (Attensity Europe GmbH) - implementation
 **********************************************************************************************************************/
package org.eclipse.smila.processing.bpel.internal;

import java.io.IOException;
import java.util.HashMap;
import java.util.Map;
import java.util.Properties;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicLong;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.smila.blackboard.Blackboard;
import org.eclipse.smila.processing.ProcessingException;
import org.eclipse.smila.processing.bpel.RequestTable;
import org.eclipse.smila.processing.bpel.util.ConfigurationHelper;
import org.eclipse.smila.processing.bpel.util.MessageHelper;

/** associates request IDs to blackboards and exceptions. provides MessageHelper. */
public class RequestTableImpl implements RequestTable {
  /** Request ID sequence. */
  private final AtomicLong _requestIdSequence = new AtomicLong(0);

  /** the blackboards currently in use associated to the request IDs. */
  private final Map<String, Blackboard> _blackboards = new ConcurrentHashMap<String, Blackboard>();

  /** last exception thrown in pipelet execution. To be reused in case the pipeline fails. */
  private final Map<String, Exception> _pipeletExceptions = new HashMap<String, Exception>();

  /** helper to create and parse XML DOM messages coming in and out of ODE engine. */
  private MessageHelper _messageHelper;

  /** local logger. */
  private final Log _log = LogFactory.getLog(getClass());

  protected void activate() {
    try {
      final Properties properties = ConfigurationHelper.readConfiguration();
      _messageHelper = new MessageHelper(properties);
    } catch (final IOException ex) {
      _log.error("Could not read configuration property file " + ConfigurationHelper.CONFIGURATION_FILE + ": "
        + ex.toString());
    }
  }

  @Override
  public String initRequest(final Blackboard blackboard) {
    final String requestId = Long.toString(_requestIdSequence.getAndIncrement());
    if (_log.isDebugEnabled()) {
      _log.debug("Starting to process request ID = " + requestId);
    }
    _blackboards.put(requestId, blackboard);
    return requestId;
  }

  @Override
  public Blackboard getBlackboard(final String id) throws ProcessingException {
    final Blackboard blackboard = _blackboards.get(id);
    if (blackboard == null) {
      throw new ProcessingException("Blackboard for request " + id + " is not registered anymore.");
    }
    return blackboard;
  }

  @Override
  public Exception getPipeletException(final String requestId) {
    return _pipeletExceptions.get(requestId);
  }

  @Override
  public void setPipeletException(final String requestId, final Exception ex) {
    _pipeletExceptions.put(requestId, ex);
  }

  @Override
  public void cleanupRequest(final String requestId) {
    if (_log.isDebugEnabled()) {
      _log.debug("Cleaning up request ID = " + requestId);
    }
    _blackboards.remove(requestId);
    _pipeletExceptions.remove(requestId);
  }

  @Override
  public MessageHelper getMessageHelper() {
    return _messageHelper;
  }
}
