package org.eclipse.smila.importing.util;

import java.util.ArrayList;
import java.util.List;
import java.util.regex.Pattern;

/**
 * Checks if a string input is a match according to the given in- and exlude patterns.
 * 
 * If include patterns are specified, at least one include pattern must match.
 * 
 * If exclude pattern are specified, no exclude pattern must match.
 * 
 * If no patterns are specified, every input string is a match.
 */
public class RegexPatternMatcher {

  /** the include patterns. */
  private final List<Pattern> _includePatterns = new ArrayList<Pattern>();

  /** the exclude patterns. */
  private final List<Pattern> _excludePatterns = new ArrayList<Pattern>();

  /** @return true, if given input matches according to given in- and exclude patterns. */
  public boolean matches(final String input) {
    // If at least one exclude pattern matches, than the input is no match.
    for (final Pattern exclude : _excludePatterns) {
      if (exclude.matcher(input).matches()) {
        return false;
      }
    }
    // exclude pattern have been survived before
    if (_includePatterns.isEmpty()) {
      return true;
    }
    // If include patterns are specified, at least one include pattern must match.
    for (final Pattern include : _includePatterns) {
      if (include.matcher(input).matches()) {
        return true;
      }
    }
    return false;
  }

  /**
   * @return true, if given input matches include patterns, without caring about exclude patterns. If no include pattern
   *         is specified, everything will be included.
   */
  public boolean isIncluded(final String input) {
    if (_includePatterns.isEmpty()) {
      return true;
    }
    // If include patterns are specified, at least one include pattern must match.
    for (final Pattern include : _includePatterns) {
      if (include.matcher(input).matches()) {
        return true;
      }
    }
    return false;
  }

  /**
   * @return true, if given input matches at least one exclude patterns, without caring about include patterns. If no
   *         exclude pattern is specified, nothing will be included.
   */
  public boolean isExcluded(final String input) {
    for (final Pattern exclude : _excludePatterns) {
      if (exclude.matcher(input).matches()) {
        return true;
      }
    }
    return false;
  }

  /** add an include pattern regex. */
  public void addIncludePattern(final String regex) {
    final Pattern p = Pattern.compile(regex);
    _includePatterns.add(p);
  }

  /** add an exclude pattern regex. */
  public void addExcludePattern(final String regex) {
    final Pattern p = Pattern.compile(regex);
    _excludePatterns.add(p);
  }

}
