/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Andreas Weber (Attensity Europe GmbH) - initial implementation
 **********************************************************************************************************************/

package org.eclipse.smila.workermanager.keepalive;

import java.util.Collection;
import java.util.Iterator;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.DelayQueue;
import java.util.concurrent.TimeUnit;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import org.eclipse.smila.taskmanager.BadParameterTaskmanagerException;
import org.eclipse.smila.taskmanager.BadParameterTaskmanagerException.Cause;
import org.eclipse.smila.taskmanager.Task;
import org.eclipse.smila.taskmanager.TaskManager;
import org.eclipse.smila.taskmanager.TaskmanagerException;
import org.eclipse.smila.workermanager.TaskKeepAliveListener;

/**
 * Helper class to store tasks and keep them alive.
 */
public class TaskKeepAlive implements Runnable {
  /** default schedule time in ms between checking of the delay queue. */
  public static final long DEFAULT_SCHEDULE_MILLIS = 2000;

  /** default timeout in s between keep alive calls. */
  public static final long TIME_OUT_DEFAULT_SECONDS = 10;

  /** The queue of the tasks to be kept alive. */
  private final DelayQueue<DelayedTask> _delayQueue = new DelayQueue<DelayedTask>();

  /** the reference of the task manager. */
  private final TaskManager _taskManager;

  /** local logger. */
  private final Log _log = LogFactory.getLog(getClass());

  /** timeout in s between keep alive calls. */
  private final long _timeout;

  /** listeners for keep alive failures. */
  private final CopyOnWriteArrayList<TaskKeepAliveListener> _keepAliveListeners =
    new CopyOnWriteArrayList<TaskKeepAliveListener>();

  /**
   * Constructor.
   * 
   * @param taskManager
   *          a reference to the task manager.
   */
  public TaskKeepAlive(final TaskManager taskManager) {
    _taskManager = taskManager;
    _timeout = TIME_OUT_DEFAULT_SECONDS;
  }

  /**
   * Constructor.
   * 
   * @param taskManager
   *          a reference to the task manager.
   * @param timeout
   *          the timespan after which a task should be kept alive
   */
  public TaskKeepAlive(final TaskManager taskManager, final long timeout) {
    _taskManager = taskManager;
    _timeout = timeout;
  }

  /**
   * Remove a task from the keep alive mechanism.
   * 
   * @param task
   *          the task to remove.
   */
  public synchronized void removeTask(final Task task) {
    // synchronize with keep-alive
    final Iterator<DelayedTask> iter = _delayQueue.iterator();
    while (iter.hasNext()) {
      final DelayedTask delayedTask = iter.next();
      if (delayedTask.getTask().getTaskId().equals(task.getTaskId())) {
        iter.remove();
        return;
      }
    }
  }

  /**
   * Adds a task to be regularly kept alive.
   * 
   * @param task
   *          the task to add.
   */
  public void addTask(final Task task) {
    _delayQueue.offer(new DelayedTask(task, _timeout, TimeUnit.SECONDS));
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void run() {
    // check for tasks that must be kept alive
    DelayedTask delayedTask;
    do {
      synchronized (this) {
        // keep synchronize block short and do not block while calling TaskManager...
        delayedTask = _delayQueue.poll();
        if (delayedTask != null) {
          // add the task again with new timeout.
          addTask(delayedTask.getTask());
        }
      }
      if (delayedTask != null) {
        try {
          keepAlive(delayedTask);
        } catch (final BadParameterTaskmanagerException ex) {
          if (ex.getCauseCode() == Cause.taskId || ex.getCauseCode() == Cause.workerName) {
            // the task or the tasks worker does not exist
            if (_log.isInfoEnabled()) {
              _log.info("Task '" + delayedTask.getTask().getTaskId()
                + "' seems to be gone, processing should be canceled.", ex);
            }
            // remove the task from the queue
            removeTask(delayedTask.getTask());
            for (final TaskKeepAliveListener listener : _keepAliveListeners) {
              try {
                listener.removedTask(delayedTask.getTask());
              } catch (final Throwable t) {
                _log.error("Cannot inform listener about removal of task.", t);
              }
            }
          } else {
            // just log the error...
            _log.error("Error while trying to keep alive task.", ex);
          }
        } catch (final TaskmanagerException ex) {
          // just log the error...
          _log.error("Error while trying to keep alive task.", ex);
        }
      }
    } while (delayedTask != null);
  }

  /**
   * keep alive a task.
   * 
   * @param delayedTask
   *          the task to be kept alive.
   * @throws TaskmanagerException
   *           an exception while keeping alive the task.
   */
  private void keepAlive(final DelayedTask delayedTask) throws TaskmanagerException {
    final Task taskToKeepAlive = delayedTask.getTask();
    if (_log.isTraceEnabled()) {
      _log.trace("Keeping task '" + taskToKeepAlive.getTaskId() + "' alive.");
    }
    _taskManager.keepAlive(taskToKeepAlive.getWorkerName(), taskToKeepAlive.getTaskId());
  }

  /**
   * Adds a {@link TaskKeepAliveListener} to this instance of {@link TaskKeepAlive}.
   * 
   * @param keepAliveListener
   *          the {@link TaskKeepAliveListener} to add.
   */
  public void addKeepAliveFailureListener(final TaskKeepAliveListener keepAliveListener) {
    _keepAliveListeners.add(keepAliveListener);
  }

  /**
   * Adds all given {@link TaskKeepAliveListener} to this instance of {@link TaskKeepAlive}.
   * 
   * @param keepAliveListeners
   *          collection of {@link TaskKeepAliveListener} to add.
   */
  public void addKeepAliveFailureListeners(final Collection<TaskKeepAliveListener> keepAliveListeners) {
    _keepAliveListeners.addAll(keepAliveListeners);
  }

  /**
   * Removes a {@link TaskKeepAliveListener} from this instance of {@link TaskKeepAlive}.
   * 
   * @param keepAliveListener
   *          the {@link TaskKeepAliveListener} to remove.
   */
  public void removeKeepAliveFailureListener(final TaskKeepAliveListener keepAliveListener) {
    _keepAliveListeners.remove(keepAliveListener);
  }

  /**
   * @return all currently registered keep alive listeners.
   */
  public Collection<TaskKeepAliveListener> getKeepAliveFailureListeners() {
    return _keepAliveListeners;
  }
}
