package org.eclipse.smila.datamodel.util;

import java.util.Iterator;
import java.util.Set;

import org.eclipse.smila.datamodel.Any;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.AnySeq;
import org.eclipse.smila.datamodel.Record;

/** Helper class for merging records. */
public final class RecordMerger {

  /**
   * merge metadata and attachments of record 'from' into record 'to'.
   * 
   * @param mergeExistingAttributes
   *          if 'true' the values of equal metadata attributes in record 'to' and 'from' will be merged (create AnySeq
   *          of values resp. merge existing AnySeq's). If 'false', equal metadata attributes or attachments already
   *          exist in record 'to' they will be overwritten, only attribute recordid will be kept if it was set before.
   */
  public static void mergeRecords(final Record to, final Record from, boolean mergeExistingAttributes) {
    final String toId = to.getId();
    if (mergeExistingAttributes) {
      mergeMetadata(to.getMetadata(), from.getMetadata());
    } else {
      to.getMetadata().putAll(from.getMetadata());
    }
    if (toId != null) {
      to.setId(toId); // if record id was set, it must be kept 
    }

    final Iterator<String> attachmentNames = from.getAttachmentNames();
    while (attachmentNames.hasNext()) {
      final String attachName = attachmentNames.next();
      to.setAttachment(attachName, from.getAttachmentAsBytes(attachName));
    }
  }

  /** helper method to merge metadata attributes. TODO do this recursive, not only on top level */
  private static void mergeMetadata(final AnyMap to, final AnyMap from) {
    final Set<String> fromKeys = from.keySet();
    for (final String fromKey : fromKeys) {
      if (to.containsKey(fromKey)) {
        final Any resultValue = to.get(fromKey);
        final Any valuesToMerge = from.get(fromKey);
        if (valuesToMerge.isValue() || valuesToMerge.isSeq()) {
          if (resultValue.isSeq()) {
            resultValue.asSeq().addAll(valuesToMerge.asSeq());
          } else if (resultValue.isValue()) {
            final AnySeq newSeq = resultValue.asSeq();
            newSeq.addAll(valuesToMerge.asSeq());
            to.put(fromKey, newSeq);
          }
        } else if (resultValue.isMap() && valuesToMerge.isMap()) {
          mergeMaps(resultValue.asMap(), valuesToMerge.asMap());
        }
      }
    }
  }

  /** helper method to merge metadata attributes. */
  private static void mergeMaps(final AnyMap to, final AnyMap from) {
    // TODO also merge map values?
    to.putAll(from);
  }
}
