/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Andreas Schank (Attensity Europe GmbH) - initial API and implementation
 *******************************************************************************/
package org.eclipse.smila.processing.httphandler;

import java.util.List;

import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.http.server.json.JsonRequestHandler;
import org.eclipse.smila.processing.PipeletTracker;
import org.eclipse.smila.utils.http.NotFoundHTTPResult;

/**
 * Implements the handling of HTTP requests for reading the description file of a pipelet.
 */
public class PipeletHandler extends JsonRequestHandler {

  /** Exception message if no description has been found. */
  public static final String EXCEPTION_MESSAGE = "There is no description for the given pipelet";

  /**
   * associated pipelet tracker.
   */
  private PipeletTracker _pipeletTracker;

  /**
   * {@inheritDoc}
   * 
   * Only GET is supported.
   */
  @Override
  protected boolean isValidMethod(final String method, final String requestUri) {
    return "GET".equals(method);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Object process(final String method, final String requestUri, final Record inputRecord) throws Exception {
    final String pipeletClassName = getPipeletClassName(requestUri);
    // only GET
    final AnyMap definitionAny = _pipeletTracker.getRegisteredPipeletDescriptions().get(pipeletClassName);
    if (definitionAny == null) {
      throw new PipeletDescriptionNotFoundException(EXCEPTION_MESSAGE + " '" + pipeletClassName + "'.");
    }
    return definitionAny;
  }

  /**
   * get pipeline class name from URI.
   * 
   * @param requestUri
   *          request uri
   * @return index name
   */
  private String getPipeletClassName(final String requestUri) {
    final List<String> uriParts = getDynamicUriParts(requestUri);
    if (uriParts.size() > 0) {
      return uriParts.get(0);
    }
    throw new IllegalArgumentException("Invalid request URI, does not match uri pattern " + getUriPattern());
  }

  /**
   * set PipeletTracker reference.
   * 
   * @param pipeletTracker
   *          PipeletTracker
   */
  public void setPipeletTracker(final PipeletTracker pipeletTracker) {
    _pipeletTracker = pipeletTracker;
  }

  /**
   * remove PipeletTracker reference.
   * 
   * @param pipeletTracker
   *          PipeletTracker
   */
  public void unsetPipeletTracker(final PipeletTracker pipeletTracker) {
    if (_pipeletTracker == pipeletTracker) {
      _pipeletTracker = null;
    }
  }

  /**
   * special exception to denote that for given pipelet class name no description could be found.
   * 
   */
  private static class PipeletDescriptionNotFoundException extends Exception implements NotFoundHTTPResult {
    /**
     * ... is serializable
     */
    private static final long serialVersionUID = 1L;

    /**
     * create exception.
     * 
     * @param message
     *          description.
     */
    public PipeletDescriptionNotFoundException(final String message) {
      super(message);
    }
  }
}
