/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Juergen Schumacher, Andreas Weber, Drazen Cindric, Andreas Schank (all Attensity Europe GmbH) - initial
 * implementation
 **********************************************************************************************************************/
package org.eclipse.smila.jobmanager.internal;

import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;

import org.apache.commons.io.IOUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.smila.common.definitions.DefinitionBase;
import org.eclipse.smila.common.exceptions.InvalidDefinitionException;
import org.eclipse.smila.datamodel.Any;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.ipc.IpcAnyReader;
import org.eclipse.smila.jobmanager.definitions.BucketDefinition;
import org.eclipse.smila.jobmanager.definitions.DataObjectTypeDefinition;
import org.eclipse.smila.jobmanager.definitions.DefinitionPersistence;
import org.eclipse.smila.jobmanager.definitions.JobDefinition;
import org.eclipse.smila.jobmanager.definitions.JobManagerConstants;
import org.eclipse.smila.jobmanager.definitions.WorkerDefinition;
import org.eclipse.smila.jobmanager.definitions.WorkflowDefinition;
import org.eclipse.smila.jobmanager.exceptions.IllegalJobStateException;
import org.eclipse.smila.jobmanager.exceptions.InvalidConfigException;
import org.eclipse.smila.jobmanager.exceptions.JobManagerException;
import org.eclipse.smila.jobmanager.exceptions.PersistenceException;
import org.eclipse.smila.jobmanager.persistence.PermanentStorage;
import org.eclipse.smila.jobmanager.persistence.RunStorage;
import org.eclipse.smila.jobmanager.taskgenerator.TaskGeneratorProvider;
import org.eclipse.smila.objectstore.ObjectStoreService;
import org.eclipse.smila.utils.config.ConfigUtils;
import org.eclipse.smila.utils.config.ConfigurationLoadException;
import org.osgi.service.component.ComponentContext;

/**
 * Handles read/write operations of jobmanager definitions.
 * 
 * Write operations are handled directly by writing to internal store after validating them. Read operations are
 * delegated to DefinitionAccessor.
 * 
 */
public class DefinitionPersistenceImpl implements DefinitionPersistence {

  /** Configuration file with predefined buckets. */
  private static final String FILENAME_PREDEFINED_BUCKETS = "buckets.json";

  /** Configuration file with predefined data object types. */
  private static final String FILENAME_PREDEFINED_DATA_OBJECT_TYPES = "dataObjectTypes.json";

  /** Configuration file with predefined workers. */
  private static final String FILENAME_PREDEFINED_WORKERS = "workers.json";

  /** Configuration file with predefined workflows. */
  private static final String FILENAME_PREDEFINED_WORKFLOWS = "workflows.json";

  /** Configuration file with predefined jobs. */
  private static final String FILENAME_PREDEFINED_JOBS = "jobs.json";

  /** local logger. */
  private final Log _log = LogFactory.getLog(getClass());

  /** for BON/JSON->Any deserialization. */
  private final IpcAnyReader _anyReader = new IpcAnyReader();

  /** for reading definitions from storage or configuration. */
  private DefinitionAccessor _definitionAccessor;

  /** for validating definitions before writing them to storage. */
  private DefinitionValidator _definitionValidator;

  /** for checking job state before manipulation. */
  private RunStorage _runStorage;

  /** permanent storage used for write operations. */
  private PermanentStorage _permStorage;

  /** references TaskGeneratorProvider service. */
  private TaskGeneratorProvider _taskGeneratorProvider;

  /** references objectstore service. */
  private ObjectStoreService _objectStore;

  /** flag used for lazy reading of configured job. */
  private boolean _hasReadConfiguredJobs;

  /**
   * OSGi Declarative Services service activation method. Create object with given underlying definition storage
   * service. The predefined data object type definitions, buckets and worker definitions are read from the
   * configuration area.
   * 
   * @param context
   *          OSGi service component context.
   */
  protected void activate(final ComponentContext context) {
    _definitionAccessor = new DefinitionAccessor();
    _definitionValidator = new DefinitionValidator(_definitionAccessor, _taskGeneratorProvider, _objectStore);
    readConfiguredDataObjectTypeDefinitions();
    readConfiguredBucketDefinitions();
    readConfiguredWorkerDefinitions();
    readConfiguredWorkflowDefinitions();
    _definitionAccessor.setStorage(_permStorage);
  }

  /** {@inheritDoc} */
  @Override
  public Collection<String> getDataObjectTypes() {
    return _definitionAccessor.getDataObjectTypes();
  }

  /** {@inheritDoc} */
  @Override
  public DataObjectTypeDefinition getDataObjectType(final String name) {
    return _definitionAccessor.getDataObjectType(name);
  }

  /** {@inheritDoc} */
  @Override
  public Collection<String> getWorkers() {
    return _definitionAccessor.getWorkers();
  }

  /** {@inheritDoc} */
  @Override
  public WorkerDefinition getWorker(final String name) {
    return _definitionAccessor.getWorker(name);
  }

  /** {@inheritDoc} */
  @Override
  public Collection<String> getBuckets() throws PersistenceException {
    return _definitionAccessor.getBuckets();
  }

  /** {@inheritDoc} */
  @Override
  public BucketDefinition getBucket(final String name) throws PersistenceException {
    return _definitionAccessor.getBucket(name);
  }

  /** {@inheritDoc} */
  @Override
  public void addBucket(final BucketDefinition bucketDefinition) throws PersistenceException,
    InvalidConfigException {
    _definitionValidator.validateBucket(bucketDefinition);
    _definitionValidator.validateBucketWorkflows(bucketDefinition);
    _permStorage.addBucket(bucketDefinition);
  }

  /** {@inheritDoc} */
  @Override
  public void removeBucket(final String name) throws PersistenceException, InvalidConfigException {
    _definitionValidator.checkRemoveBucket(name);
    _permStorage.removeBucket(name);
  }

  /** {@inheritDoc} */
  @Override
  public Collection<String> getJobs() throws PersistenceException {
    readConfiguredJobDefinitions();
    return _definitionAccessor.getJobs();
  }

  /** {@inheritDoc} */
  @Override
  public JobDefinition getJob(final String name) throws PersistenceException {
    readConfiguredJobDefinitions();
    return _definitionAccessor.getJob(name);
  }

  /** {@inheritDoc} */
  @Override
  public boolean hasJob(final String jobName) throws PersistenceException {
    return getJob(jobName) != null;
  }

  /** {@inheritDoc} */
  @Override
  public void removeJob(final String name) throws PersistenceException, IllegalJobStateException {
    try {
      if (_runStorage.getJobRunInfo(name) != null) {
        throw new IllegalJobStateException("Job '" + name + "' cannot be removed, it is still running.");
      }
      _definitionValidator.checkJobRemove(name);
    } catch (final IllegalJobStateException e) {
      throw e;
    } catch (final JobManagerException e) {
      throw new PersistenceException("Exception while checking job '" + name + "' for deletion.", e);
    }
    _permStorage.removeJob(name);
  }

  /** {@inheritDoc} */

  @Override
  public void addJob(final JobDefinition jobDefinition) throws PersistenceException, InvalidConfigException {
    _definitionValidator.validateJob(jobDefinition);
    _permStorage.addJob(jobDefinition);
  }

  /** {@inheritDoc} */

  @Override
  public Collection<String> getWorkflows() throws PersistenceException {
    return _definitionAccessor.getWorkflows();
  }

  /** {@inheritDoc} */

  @Override
  public WorkflowDefinition getWorkflow(final String name) throws PersistenceException {
    return _definitionAccessor.getWorkflow(name);
  }

  /** {@inheritDoc} */

  @Override
  public void removeWorkflow(final String name) throws PersistenceException, InvalidConfigException {
    _definitionValidator.checkRemoveWorkflow(name);
    _permStorage.removeWorkflow(name);
  }

  /** {@inheritDoc} */

  @Override
  public void addWorkflow(final WorkflowDefinition workflowDefinition) throws PersistenceException,
    InvalidConfigException {
    _definitionValidator.validateWorkflow(workflowDefinition);
    _permStorage.addWorkflow(workflowDefinition);
  }

  /**
   * read predefined buckets from configuration area. Tries to load as many definitions as possible, invalid definitions
   * are ignored, see log for details.
   */
  private void readConfiguredBucketDefinitions() {
    try {
      final InputStream bucketsJson =
        ConfigUtils.getConfigStream(JobManagerConstants.CONFIGURATION_BUNDLE, FILENAME_PREDEFINED_BUCKETS);
      try {
        final AnyMap bucketsAny = (AnyMap) _anyReader.readJsonStream(bucketsJson);
        // add a readonly flag for the preconfigured definitions.
        final Collection<BucketDefinition> buckets = parsePredefinedBuckets(bucketsAny);
        final Map<String, BucketDefinition> configuredBuckets = new HashMap<String, BucketDefinition>();
        for (final BucketDefinition bucket : buckets) {
          if (configuredBuckets.containsKey(bucket.getName())) {
            _log.warn("Configuration contains duplicate bucket definition '" + bucket.getName()
              + "', ignoring second one.");
          } else {
            try {
              _definitionValidator.validateBucket(bucket);
              configuredBuckets.put(bucket.getName(), bucket);
            } catch (final InvalidConfigException ex) {
              _log.warn("Configuration contains invalid bucket definition for bucket '" + bucket.getName() + "'",
                ex);
            }
          }
        }
        _definitionAccessor.setConfiguredBuckets(configuredBuckets);
      } catch (final IOException ex) {
        _log.warn("Error reading predefined buckets from configuration area, no predefined buckets available.", ex);
      } catch (final PersistenceException ex) {
        _log.warn("Error reading predefined buckets from configuration area, no predefined buckets available.", ex);
      } catch (final InvalidDefinitionException e) {
        _log.warn("Error parsing predefined bucket definitions from configuration area", e);
      } finally {
        IOUtils.closeQuietly(bucketsJson);
      }
    } catch (final ConfigurationLoadException ex) {
      _log.info("No predefined buckets found in configuration area.");
    }
  }

  /**
   * Reads the predefined data object type definitions from configuration area.
   */
  private void readConfiguredDataObjectTypeDefinitions() {
    try {
      final InputStream dotJson =
        ConfigUtils
          .getConfigStream(JobManagerConstants.CONFIGURATION_BUNDLE, FILENAME_PREDEFINED_DATA_OBJECT_TYPES);
      try {
        final AnyMap dotMap = (AnyMap) _anyReader.readJsonStream(dotJson);
        // add a readonly flag for the preconfigured definitions.
        final Collection<DataObjectTypeDefinition> dots = parsePredefinedDataObjectTypes(dotMap);
        final Map<String, DataObjectTypeDefinition> configuredDataObjectTypes =
          new HashMap<String, DataObjectTypeDefinition>();
        for (final DataObjectTypeDefinition dot : dots) {
          configuredDataObjectTypes.put(dot.getName(), dot);
        }
        _definitionAccessor.setConfiguredDataObjectTypes(configuredDataObjectTypes);
      } catch (final IOException ex) {
        _log.warn("Error reading predefined data object type definitions from configuration area, "
          + "no predefined data object type definitions available.", ex);
      } catch (final InvalidDefinitionException e) {
        _log.warn("Error parsing predefined data object type definitions from configuration area", e);
      } finally {
        IOUtils.closeQuietly(dotJson);
      }
    } catch (final ConfigurationLoadException cle) {
      _log.info("No predefined data object types found in configuration area.");
    }
  }

  /**
   * Reads the predefined worker definitions from configuration area.
   */
  private void readConfiguredWorkerDefinitions() {
    try {
      final InputStream workerJson =
        ConfigUtils.getConfigStream(JobManagerConstants.CONFIGURATION_BUNDLE, FILENAME_PREDEFINED_WORKERS);
      try {
        final AnyMap workerMap = (AnyMap) _anyReader.readJsonStream(workerJson);
        // add a readonly flag for the preconfigured definitions.
        final Collection<WorkerDefinition> workers = parsePredefinedWorkers(workerMap);
        final Map<String, WorkerDefinition> configuredWorkers = new HashMap<String, WorkerDefinition>();
        for (final WorkerDefinition worker : workers) {
          configuredWorkers.put(worker.getName(), worker);
        }
        _definitionAccessor.setConfiguredWorkers(configuredWorkers);
      } catch (final IOException ex) {
        _log.warn("Error reading predefined worker definitions from configuration area, "
          + "no predefined worker definitions available.", ex);
      } catch (final InvalidDefinitionException e) {
        _log.warn("Error parsing predefined worker definitions from configuration area", e);
      } finally {
        IOUtils.closeQuietly(workerJson);
      }
    } catch (final ConfigurationLoadException cle) {
      _log.info("No predefined workers found in configuration area.");
    }
  }

  /**
   * read predefined workflows from configuration area. Tries to load as many definitions as possible, invalid
   * definitions are ignored, see log for details.
   */
  private void readConfiguredWorkflowDefinitions() {
    try {
      final InputStream workflowsJson =
        ConfigUtils.getConfigStream(JobManagerConstants.CONFIGURATION_BUNDLE, FILENAME_PREDEFINED_WORKFLOWS);
      try {
        final AnyMap workflowsAnyMap = (AnyMap) _anyReader.readJsonStream(workflowsJson);
        // add a readonly flag for the preconfigured definitions.
        final Collection<WorkflowDefinition> workflows = parsePredefinedWorflows(workflowsAnyMap);
        final Map<String, WorkflowDefinition> configuredWorkflows = new HashMap<String, WorkflowDefinition>();
        for (final WorkflowDefinition workflow : workflows) {
          if (configuredWorkflows.containsKey(workflow.getName())) {
            _log.warn("Configuration contains duplicate workflow definition '" + workflow.getName()
              + "', ignoring second one.");
          } else {
            try {
              _definitionValidator.validateWorkflow(workflow);
              configuredWorkflows.put(workflow.getName(), workflow);
            } catch (final InvalidConfigException ex) {
              _log.warn("Configuration contains invalid workflow definition for workflow '" + workflow.getName()
                + "'.", ex);
            } catch (final PersistenceException e) {
              _log.warn("Error while trying to validate workflow '" + workflow.getName() + "'.", e);
            }
          }
        }
        _definitionAccessor.setConfiguredWorkflows(configuredWorkflows);
      } catch (final IOException ex) {
        _log.warn("Error reading predefined workflows from configuration area, no predefined workflows available.",
          ex);
      } catch (final InvalidDefinitionException e) {
        _log.warn("Error parsing predefined workflows from configuration area", e);
      } finally {
        IOUtils.closeQuietly(workflowsJson);
      }
    } catch (final ConfigurationLoadException ex) {
      _log.info("No predefined workflows found in configuration area.");
    }
  }

  /**
   * read predefined jobs from configuration area. Tries to load as many definitions as possible, invalid definitions
   * are ignored, see log for details.
   */
  private synchronized void readConfiguredJobDefinitions() {
    if (!_hasReadConfiguredJobs) {
      // we read them here (lazy) because job parameter validation uses TaskGenerators. And we can't be sure that all
      // TaskGenerators are already registered at TaskGeneratorProvider when our activate() method is running.
      try {
        final InputStream jobsJson =
          ConfigUtils.getConfigStream(JobManagerConstants.CONFIGURATION_BUNDLE, FILENAME_PREDEFINED_JOBS);
        try {
          final AnyMap jobsAny = (AnyMap) _anyReader.readJsonStream(jobsJson);
          final Collection<JobDefinition> jobs = parsePredefinedJobs(jobsAny);
          final Map<String, JobDefinition> configuredJobs = new HashMap<String, JobDefinition>();
          for (final JobDefinition job : jobs) {
            if (configuredJobs.containsKey(job.getName())) {
              _log.warn("Configuration contains duplicate job definition '" + job.getName()
                + "', ignoring second one.");
            } else {
              try {
                _definitionValidator.validateJob(job);
                configuredJobs.put(job.getName(), job);
              } catch (final InvalidConfigException ex) {
                _log.warn("Configuration contains invalid job definition for job '" + job.getName() + "'.", ex);
              } catch (final PersistenceException e) {
                _log.warn("Error while trying to validate job '" + job.getName() + "'.", e);
              }
            }
          }
          _definitionAccessor.setConfiguredJobs(configuredJobs);
        } catch (final IOException ex) {
          _log.warn("Error reading predefined jobs from configuration area, no predefined jobs available.", ex);
        } catch (final InvalidDefinitionException e) {
          _log.warn("Error parsing predefined jobs from configuration area", e);
        } finally {
          IOUtils.closeQuietly(jobsJson);
        }
      } catch (final ConfigurationLoadException ex) {
        _log.info("No predefined jobs found in configuration area.");
      }
      _hasReadConfiguredJobs = true;
    }
  }

  /**
   * Parse bucket list from an Any object containing a sequence of bucket descriptions:
   * 
   * <pre>
   * {
   *   "buckets": [
   *     {
   *       // see class comment for bucket format.
   *     }
   *   ]
   * }
   * </pre>
   * 
   * @param bucketsAny
   *          bucket sequence as Any.
   * @return list of parsed bucket definitions.
   * @throws InvalidDefinitionException
   *           error parsing Any.
   */
  private Collection<BucketDefinition> parsePredefinedBuckets(final AnyMap bucketsAny)
    throws InvalidDefinitionException {
    final Collection<BucketDefinition> buckets = new ArrayList<BucketDefinition>();
    try {
      if (bucketsAny.containsKey(BucketDefinition.KEY_BUCKETS)) {
        for (final Any bucketAny : bucketsAny.getSeq(BucketDefinition.KEY_BUCKETS)) {
          if (bucketAny.isMap()) {
            bucketAny.asMap().put(DefinitionBase.KEY_READ_ONLY, true);
            try {
              buckets.add(new BucketDefinition((AnyMap) bucketAny));
            } catch (final InvalidDefinitionException ex) {
              _log.warn("Configuration contains invalid bucket definition for bucket '"
                + bucketAny.asMap().getStringValue(DefinitionBase.KEY_NAME) + "'.", ex);
            }
          }
        }
      } else {
        throw new InvalidDefinitionException("Missing field " + BucketDefinition.KEY_BUCKETS);
      }
    } catch (final InvalidDefinitionException ex) {
      throw ex;
    } catch (final Exception ex) {
      throw new InvalidDefinitionException("Invalid any structure", ex);
    }
    return buckets;
  }

  /**
   * @param dotMap
   *          The any object with the desired values
   * @return the collection the data object type definitions
   * @throws InvalidDefinitionException
   *           exception if the any object is not filled with all desired values
   */
  private Collection<DataObjectTypeDefinition> parsePredefinedDataObjectTypes(final AnyMap dotMap)
    throws InvalidDefinitionException {
    try {
      final Collection<DataObjectTypeDefinition> dataObjectTypeDefinitions =
        new ArrayList<DataObjectTypeDefinition>();
      if (dotMap.containsKey(DataObjectTypeDefinition.KEY_DATA_OBJECT_TYPE_DEFINITIONS)) {
        for (final Any dataObjectTypeDefinitionAny : dotMap
          .getSeq(DataObjectTypeDefinition.KEY_DATA_OBJECT_TYPE_DEFINITIONS)) {
          if (dataObjectTypeDefinitionAny.isMap()) {
            dataObjectTypeDefinitionAny.asMap().put(DefinitionBase.KEY_READ_ONLY, true);
            try {
              dataObjectTypeDefinitions.add(new DataObjectTypeDefinition((AnyMap) dataObjectTypeDefinitionAny));
            } catch (final InvalidDefinitionException ex) {
              _log.warn("Configuration contains invalid data objetc type definition for data object type '"
                + dataObjectTypeDefinitionAny.asMap().getStringValue(DefinitionBase.KEY_NAME) + "'.", ex);
            }
          }
        }
      } else {
        throw new InvalidDefinitionException("Missing field: "
          + DataObjectTypeDefinition.KEY_DATA_OBJECT_TYPE_DEFINITIONS);
      }
      return dataObjectTypeDefinitions;
    } catch (final InvalidDefinitionException ice) {
      throw ice;
    } catch (final Exception e) {
      throw new InvalidDefinitionException("Error during reading dataObjectTypeDefinitionAny: "
        + e.getLocalizedMessage());
    }
  }

  /**
   * @param workerMap
   *          The any object with the desired values
   * @return the collection the data object type definitions
   * @throws InvalidDefinitionException
   *           exception if the any object is not filled with all desired values
   */
  private Collection<WorkerDefinition> parsePredefinedWorkers(final AnyMap workerMap)
    throws InvalidDefinitionException {
    try {
      final Collection<WorkerDefinition> workerDefinitions = new ArrayList<WorkerDefinition>();
      if (workerMap.containsKey(WorkerDefinition.KEY_WORKERS)) {
        for (final Any workerDefinitionAny : workerMap.getSeq(WorkerDefinition.KEY_WORKERS)) {
          if (workerDefinitionAny.isMap()) {
            workerDefinitionAny.asMap().put(DefinitionBase.KEY_READ_ONLY, true);
            try {
              workerDefinitions.add(new WorkerDefinition((AnyMap) workerDefinitionAny));
            } catch (final InvalidDefinitionException ex) {
              _log.warn("Configuration contains invalid worker definition for worker '"
                + workerDefinitionAny.asMap().getStringValue(DefinitionBase.KEY_NAME) + "'.", ex);
            }
          }
        }
      } else {
        throw new InvalidDefinitionException("Missing field: " + WorkerDefinition.KEY_WORKERS);
      }
      return workerDefinitions;
    } catch (final InvalidDefinitionException ice) {
      throw ice; // why the heck are these exceptions just rethrown??
    } catch (final Exception e) {
      throw new InvalidDefinitionException("Error during reading workerDefinitionsRecordAny ", e);
    }
  }

  /**
   * Parse workflow list from an Any object containing a sequence of workflow descriptions:
   * 
   * <pre>
   * {
   *   "workflows": [
   *     {
   *       // see class comment for job format.
   *     }
   *   ]
   * }
   * </pre>
   * 
   * @param workflowsAny
   *          workflow sequence as Any.
   * @return list of parsed workflow definitions.
   * @throws InvalidDefinitionException
   *           error parsing Any.
   */
  private Collection<WorkflowDefinition> parsePredefinedWorflows(final AnyMap workflowsAny)
    throws InvalidDefinitionException {
    final Collection<WorkflowDefinition> workflows = new ArrayList<WorkflowDefinition>();
    try {
      if (workflowsAny.containsKey(WorkflowDefinition.KEY_WORKFLOWS)) {
        for (final Any workflowAny : workflowsAny.getSeq(WorkflowDefinition.KEY_WORKFLOWS)) {
          if (workflowAny.isMap()) {
            workflowAny.asMap().put(DefinitionBase.KEY_READ_ONLY, true);
            try {
              workflows.add(new WorkflowDefinition((AnyMap) workflowAny));
            } catch (final InvalidDefinitionException ex) {
              _log.warn("Configuration contains invalid workflow definition for workflow '"
                + workflowAny.asMap().getStringValue(DefinitionBase.KEY_NAME) + "'.", ex);
            }
          }
        }
      } else {
        throw new InvalidConfigException("Missing field " + WorkflowDefinition.KEY_WORKFLOWS);
      }
    } catch (final InvalidDefinitionException ex) {
      throw ex;
    } catch (final Exception ex) {
      throw new InvalidDefinitionException("Invalid any structure", ex);
    }
    return workflows;
  }

  /**
   * Parse job list from an Any object containing a sequence of job descriptions:
   * 
   * <pre>
   * {
   *   "jobs": [
   *     {
   *       // see class comment for job format.
   *     }
   *   ]
   * }
   * </pre>
   * 
   * @param jobsAny
   *          job sequence as Any.
   * @return list of parsed job definitions.
   * @throws InvalidDefinitionException
   *           error parsing Any.
   */
  private Collection<JobDefinition> parsePredefinedJobs(final AnyMap jobsAny) throws InvalidDefinitionException {
    final Collection<JobDefinition> jobs = new ArrayList<JobDefinition>();
    if (jobsAny.containsKey(JobDefinition.KEY_JOBS)) {
      for (final Any jobAny : jobsAny.getSeq(JobDefinition.KEY_JOBS)) {
        if (jobAny.isMap()) {
          // add a readonly flag for the preconfigured definitions.
          jobAny.asMap().put(DefinitionBase.KEY_READ_ONLY, true);
          try {
            jobs.add(new JobDefinition(jobAny.asMap()));
          } catch (final InvalidDefinitionException ex) {
            _log.warn(
              "Configuration contains invalid job definition for job '"
                + jobAny.asMap().getStringValue(DefinitionBase.KEY_NAME) + "'.", ex);
          }
        } else {
          throw new InvalidDefinitionException("Invalid any structure.");
        }
      }
    } else {
      throw new InvalidDefinitionException("Missing field " + JobDefinition.KEY_JOBS);
    }
    return jobs;
  }

  /**
   * @param permStorage
   *          PermanentStorage reference.
   */
  public void setPermanentStorage(final PermanentStorage permStorage) {
    _permStorage = permStorage;
  }

  /**
   * @param permStorage
   *          PermanentStorage reference.
   */
  public void unsetPermanentStorage(final PermanentStorage permStorage) {
    if (_permStorage == permStorage) {
      _permStorage = null;
    }
  }

  /**
   * @param runStorage
   *          RunStorage reference.
   */
  public void setRunStorage(final RunStorage runStorage) {
    _runStorage = runStorage;
  }

  /**
   * @param runStorage
   *          RunStorage reference.
   */
  public void unsetRunStorage(final RunStorage runStorage) {
    if (_runStorage == runStorage) {
      _runStorage = null;
    }
  }

  /**
   * method for DS to set a service reference.
   * 
   * @param objectStore
   *          ObjectStoreService reference.
   */
  public void setObjectStoreService(final ObjectStoreService objectStore) {
    _objectStore = objectStore;
  }

  /**
   * method for DS to unset a service reference.
   * 
   * @param objectStore
   *          ObjectStoreService reference.
   */
  public void unsetObjectStoreService(final ObjectStoreService objectStore) {
    if (_objectStore == objectStore) {
      _objectStore = null;
    }
  }

  /**
   * method for DS to set a service reference.
   * 
   * @param taskGeneratorProvider
   *          TaskGeneratorProvider reference.
   */
  public void setTaskGeneratorProvider(final TaskGeneratorProvider taskGeneratorProvider) {
    _taskGeneratorProvider = taskGeneratorProvider;
  }

  /**
   * method for DS to unset a service reference.
   * 
   * @param taskGeneratorProvider
   *          TaskGeneratorProvider reference.
   */
  public void unsetTaskGeneratorProvider(final TaskGeneratorProvider taskGeneratorProvider) {
    if (_taskGeneratorProvider == taskGeneratorProvider) {
      _taskGeneratorProvider = null;
    }
  }

}
