/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Juergen Schumacher, Andreas Weber, Drazen Cindric, Andreas Schank (all Attensity Europe GmbH) - initial
 * implementation
 **********************************************************************************************************************/
package org.eclipse.smila.jobmanager.persistence;

import java.util.Collection;
import java.util.List;
import java.util.Map;

import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.jobmanager.JobRunInfo;
import org.eclipse.smila.jobmanager.JobState;
import org.eclipse.smila.jobmanager.definitions.JobRunDefinitions;
import org.eclipse.smila.jobmanager.definitions.JobRunMode;
import org.eclipse.smila.taskmanager.Task;

/**
 * Persistence component interface for run data.
 */
public interface RunStorage {

  /**
   * Remove all jobmanager run data. Can be called concurrently.
   * 
   * @throws RunStorageException
   *           error
   */
  void clear() throws RunStorageException;

  /**
   * Starts a job run. If another instance already started the job the method will throw an exception.
   * 
   * @param jobName
   *          The name of the job to start.
   * @param jobRunId
   *          The job run id for the started job run.
   * @param jobRunMode
   *          The job run mode for the started job run.
   * @param jobRunDefs
   *          The job run definitions that are stored for this job run.
   * @throws RunStorageException
   *           If there's an error while accessing the run storage.
   */
  void startJobRun(final String jobName, final String jobRunId, final JobRunMode jobRunMode,
    final JobRunDefinitions jobRunDefs) throws RunStorageException;

  /**
   * Called when a job run is finished. This does _not_ remove the job run data.
   * 
   * Since it is possible that more instances of a job manager try to finish a job run concurrently the one that
   * succeeded will get 'true' as return value, the others false but no exception will be thrown in that case, because
   * this is not an exceptional behavior.
   * 
   * @param jobName
   *          The job name
   * @param jobRunId
   *          The job run id for the finished job run.
   * @return true if the caller was the first to finish the job, false if another caller finished the job in the
   *         meantime.
   * @throws RunStorageException
   *           error
   */
  boolean finishJobRun(final String jobName, final String jobRunId) throws RunStorageException;

  /**
   * Removes the job run data of the given job run.
   * 
   * @param jobName
   *          the job
   * @param jobRunId
   *          the job run for which to remove the data
   * @throws RunStorageException
   *           error
   */
  void deleteJobRun(final String jobName, final String jobRunId) throws RunStorageException;

  /**
   * Cancels the given job run data -> active workflow runs and current tasks will be canceled. (Does not set job run
   * state!).
   * 
   * @param jobName
   *          the job
   * @param jobRunId
   *          the job run for which to remove the data
   * @return the ids of the workflow runs that were canceled.
   * @throws RunStorageException
   *           error
   */
  List<String> cancelJobRun(final String jobName, final String jobRunId) throws RunStorageException;

  /**
   * @return names of job that have currently run data stored in this RunStorage. Usually it means that they are
   *         {@link JobState#PREPARING},{@link JobState#RUNNING}, or {@link JobState#FINISHING}, but while cleaning up
   *         jobs in states {@link JobState#SUCCEEDED} and {@link JobState#FAILED}, they are still in RunStorage, too.
   * @throws RunStorageException
   *           error
   */
  Collection<String> getCurrentJobs() throws RunStorageException;

  /**
   * @param jobName
   *          the job
   * @return the id of the currently active job run, or 'null' if no job run is active.
   * @throws RunStorageException
   *           error
   */
  String getJobRunId(final String jobName) throws RunStorageException;

  /**
   * @param jobName
   *          job
   * @param returnDetails
   *          whether to return details (e.g. workflow/bucket definition)
   * @return the data for the currently active job run, or 'null' if no job run is active.
   * @throws RunStorageException
   *           error
   */
  AnyMap getJobRunData(final String jobName, final boolean returnDetails) throws RunStorageException;

  /**
   * get information about a workflow run. The result contains these keys with integer values:
   * <ul>
   * <li>{@link org.eclipse.smila.jobmanager.definitions.JobManagerConstants#DATA_WORKFLOW_RUN_NO_OF_ACTIVE_TASKS}
   * <li>{@link org.eclipse.smila.jobmanager.definitions.JobManagerConstants#DATA_WORKFLOW_RUN_NO_OF_TRANSIENT_BULKS}
   * </ul>
   * The result may contain additional keys.
   * 
   * @return workflow run information
   * 
   * @throws RunStorageException
   *           jobrun or workflowrun are not active anymore.
   */
  AnyMap getWorkflowRunData(String jobName, String workflowRunId) throws RunStorageException;

  /**
   * Sets the state for the given job run.
   * 
   * Do not invoke concurrently! This method does not check any preconditions and overwrites the state without further
   * notice. Please check that you are the only one to set the state for a job run before invoking this method!
   * 
   * @param jobName
   *          job
   * @param jobRunId
   *          job run
   * @param jobState
   *          the new state to set
   * @throws RunStorageException
   *           error
   */
  void setJobState(final String jobName, final String jobRunId, final JobState jobState) throws RunStorageException;

  /**
   * Sets the state for the given job run if the given expected state matches the current state.
   * 
   * @param jobName
   *          job
   * @param jobRunId
   *          job run
   * @param expectedState
   *          the old state that is expected to be the current state
   * @param newState
   *          the new state to set
   * @return 'true', if new state could be set, otherwise 'false'.
   * @throws RunStorageException
   *           error
   */
  boolean setJobState(final String jobName, final String jobRunId, final JobState expectedState,
    final JobState newState) throws RunStorageException;

  /**
   * @param jobName
   *          job
   * @return the state of the currently active job run for the given job, or 'null' if no job run is active.
   * @throws RunStorageException
   *           error
   */
  JobState getJobState(final String jobName) throws RunStorageException;

  /**
   * @param jobName
   *          job
   * @param jobRunId
   *          job run
   * @return the state of the given job run for the given job, or 'null' if this job run is not active
   * @throws RunStorageException
   *           error
   */
  JobState getJobState(final String jobName, final String jobRunId) throws RunStorageException;

  /**
   * @param jobName
   *          job
   * @return if job is defined and active: id and state of current run. Else null.
   * @throws RunStorageException
   *           error
   */
  JobRunInfo getJobRunInfo(final String jobName) throws RunStorageException;

  /**
   * Called when a new workflow run is started for the given job run.
   * 
   * @param jobName
   *          job
   * @param jobRunId
   *          job run
   * @return the workflow run id of the new workflow run.
   * @throws RunStorageException
   *           error
   */
  String startWorkflowRun(final String jobName, final String jobRunId) throws RunStorageException;

  /**
   * Called when a new completing workflow run is started for the given job run.
   * 
   * @param jobName
   *          job
   * @param jobRunId
   *          job run
   * @return the workflow run id of the new workflow run.
   * @throws RunStorageException
   *           error
   */
  String startCompletionWorkflowRun(String jobName, String jobRunId) throws RunStorageException;

  /**
   * Called before a workflow run is finished and deleted. Decides whether this thread will be allowed to finish the
   * workflow run.
   * 
   * Two flags are set here to avoid concurrent races:
   * 
   * 1.) a temporary job run data map entry 'workflow-run-finish-<workflow-run-id>' is set to a locally created id, to
   * make sure that only one thread is doing the finishing.
   * 
   * 2.) a temporary ephemeral 'finishing' flag is set to show that this workflow run is currently finished. This is
   * needed because server could be crashed in between. If entry above is set and flag is missing, than entry above can
   * be overwritten!
   * 
   * Both entries are removed in {@link RunStorage#deleteWorkflowRun(String, String, String)}.
   * 
   * @return 'true' if the calling thread is the one to finish the workflow, false if not.
   */
  boolean prepareToFinishWorkflowRun(final String jobName, final String jobRunId, final String workflowRunId)
    throws RunStorageException;

  /**
   * Called when a workflow run successfully finished. The workflow run data is merged into the job run data. Does not
   * delete the data of the workflow run.
   */
  void successfulWorkflowRun(final String jobName, final String jobRunId, final String workflowRunId)
    throws RunStorageException;

  /**
   * Called when a workflow run failed. The workflow run data is merged into the job run data. Does not delete the data
   * of the workflow run.
   */
  void failedWorkflowRun(final String jobName, final String jobRunId, final String workflowRunId)
    throws RunStorageException;

  /**
   * Deletes the data of the given workflow run.
   */
  void deleteWorkflowRunData(final String jobName, final String jobRunId, final String workflowRunId)
    throws RunStorageException;

  /**
   * Deletes the given workflow run.
   */
  void deleteWorkflowRun(final String jobName, final String jobRunId, final String workflowRunId)
    throws RunStorageException;

  /**
   * Checks if there are active workflow runs. Also checks if there are completed workflow runs that someone else
   * started finishing but couldn't succeed. Such completed workflow runs are cleaned up. (finished + removed)
   * 
   * @return 'true' if there are active workflow runs, otherwise 'false'.
   */
  boolean checkAndCleanupActiveWorkflowRuns(final String jobName, final String jobRunId) throws RunStorageException;

  /**
   * @param jobName
   *          job
   * @param jobRunId
   *          job run
   * @param workflowRunId
   *          workflow run
   * @return 'true' if the given workflow run exists.
   * @throws RunStorageException
   *           error
   */
  boolean hasWorkflowRun(final String jobName, final String jobRunId, final String workflowRunId)
    throws RunStorageException;

  /**
   * @param jobName
   *          job
   * @param jobRunId
   *          job run
   * @param workflowRunId
   *          workflow run
   * @return 'true' if there are active tasks in the given workflow run, otherwise 'false'
   * @throws RunStorageException
   *           error
   */
  boolean hasTasks(final String jobName, final String jobRunId, final String workflowRunId)
    throws RunStorageException;

  /**
   * Called when a new task is started for the given step in the given workflow of the given job run.
   * 
   * @param jobName
   *          job
   * @param jobRunId
   *          job run
   * @param workflowRunId
   *          workflow run
   * @param stepId
   *          step id
   * @param taskId
   *          task id
   * @throws RunStorageException
   *           if task already exists, or other error while storing task
   */
  void startTask(String jobName, String jobRunId, String workflowRunId, String stepId, String taskId)
    throws RunStorageException;

  /**
   * Called when a new task is started for the given step in the given workflow of the given job run. The
   * numberOfRetries parameter adds the possibility to store the number of times this task has already been restarted.
   * 
   * @param jobName
   *          job
   * @param jobRunId
   *          job run
   * @param workflowRunId
   *          workflow run
   * @param stepId
   *          step id
   * @param taskId
   *          task id
   * @param numberOfRetries
   *          current number of retries for this task
   * @throws RunStorageException
   *           if task already exists, or other error while storing task
   */
  void startTask(final String jobName, final String jobRunId, final String workflowRunId, final String stepId,
    final String taskId, final int numberOfRetries) throws RunStorageException;

  /**
   * Start multiple new tasks for a given job/workflow run at once.
   * 
   * @param jobName
   *          job
   * @param jobRunId
   *          job run
   * @param workflowRunId
   *          workflow run
   * @param tasks
   *          new tasks
   * @throws RunStorageException
   *           error while storing task
   */
  void startTasks(String jobName, String jobRunId, String workflowRunId, Collection<Task> tasks)
    throws RunStorageException;

  /**
   * get the number of retries for the task.
   * 
   * @param jobName
   *          job
   * @param jobRunId
   *          job run
   * @param workflowRunId
   *          workflow run
   * @param stepId
   *          step id
   * @param taskId
   *          task id
   * @return The number of retries for the task.
   * @throws RunStorageException
   *           if task already exists, or other error while storing task
   */
  int getTaskRetries(final String jobName, final String jobRunId, final String workflowRunId, final String stepId,
    final String taskId) throws RunStorageException;

  /**
   * Called when a task is successfully finished. Deletes the data of a given task for the given step in the given
   * workflow of the given job run.
   */
  void finishTask(final String jobName, final String jobRunId, final String workflowRunId, final String stepId,
    final String taskId, final Map<String, Number> workerCounter, final Map<String, String> properties)
    throws RunStorageException;

  /**
   * Called when a task finished as obsolete. Deletes the data of a given task for the given step in the given workflow
   * of the given job run.
   */
  void obsoleteTask(final String jobName, final String jobRunId, final String workflowRunId, final String stepId,
    final String taskId, final Map<String, String> properties) throws RunStorageException;

  /**
   * Called when a task is retried. Deletes the data of a given task for the given step in the given workflow of the
   * given job run.
   * 
   * @param retryByWorker
   *          whether the retry was triggered by a worker
   */
  void retriedTask(final String jobName, final String jobRunId, final String workflowRunId, final String stepId,
    final String taskId, final boolean retryByWorker, final Map<String, String> properties)
    throws RunStorageException;

  /**
   * Called when a task is failed. Deletes the data of a given task for the given step in the given workflow of the
   * given job run.
   */
  void failedTask(final String jobName, final String jobRunId, final String workflowRunId, final String stepId,
    final String taskId, final boolean failedAfterRetry, final Map<String, String> properties)
    throws RunStorageException;

  /**
   * Stores a transient bulk for the given workflow run.
   * 
   * @param jobName
   *          job
   * @param jobRunId
   *          job run
   * @param workflowRunId
   *          workflow run
   * @param bulk
   *          the bulk id
   * @throws RunStorageException
   *           error
   */
  void addTransientBulk(final String jobName, final String jobRunId, final String workflowRunId, final String bulk)
    throws RunStorageException;

  /**
   * @param jobName
   *          job
   * @param jobRunId
   *          job run
   * @param workflowRunId
   *          workflow run
   * @return the transient bulk ids for the given workflow run
   * @throws RunStorageException
   *           error
   */
  Collection<String> getTransientBulks(final String jobName, final String jobRunId, final String workflowRunId)
    throws RunStorageException;

  /**
   * @param bucketId
   *          a bucket ID
   * @return names of jobs triggered by this bucket
   * @throws RunStorageException
   *           error
   */
  Collection<String> getTriggeredJobs(String bucketId) throws RunStorageException;

  /**
   * @param bucketId
   *          a bucket ID
   * @param jobName
   *          job name
   * @throws RunStorageException
   *           error
   */
  void addJobTrigger(String bucketId, String jobName) throws RunStorageException;

  /**
   * @param bucketId
   *          a bucket ID
   * @param jobName
   *          job name
   * @throws RunStorageException
   *           error
   */
  void removeJobTrigger(String bucketId, String jobName) throws RunStorageException;

  /**
   * Check if the given task with the given task id exists for the job and workflowrun.
   * 
   * @param jobName
   *          the name of the job.
   * @param workflowRunId
   *          the id of the workflow run
   * @param taskId
   *          the id of the task
   * @return 'true' if such a task exists, 'false' if not.
   * @throws RunStorageException
   *           an exception occurred while checking for task existence.
   */
  boolean hasTask(String jobName, String workflowRunId, String taskId) throws RunStorageException;

  /**
   * @return string with current timestamp
   */
  String getCurrentTimestamp();

  /**
   * @param jobRunListener
   *          listener to be notified about job run events.
   * @param jobName
   *          job on which to listen
   */
  void registerJobRunListener(JobRunListener jobRunListener, String jobName);

}
