/*********************************************************************************************************************
 * Copyright (c) 2008, 2012 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Andreas Weber (Attensity Europe GmbH) - initial implementation
 **********************************************************************************************************************/
package org.eclipse.smila.jobmanager.taskgenerator;

import java.util.List;
import java.util.UUID;

import org.eclipse.smila.jobmanager.JobRun;
import org.eclipse.smila.jobmanager.definitions.JobDefinition;
import org.eclipse.smila.jobmanager.definitions.JobManagerConstants;
import org.eclipse.smila.jobmanager.definitions.WorkerDefinition;
import org.eclipse.smila.jobmanager.definitions.WorkerDefinition.Input;
import org.eclipse.smila.taskmanager.BulkInfo;
import org.eclipse.smila.taskmanager.Task;
import org.eclipse.smila.taskworker.input.InputMode;

/** Helper class used for task generation issues. */
public abstract class TaskGenerationUtil {

  /**
   * set task properties needed by JobManager for task-to-workflow association, and set qualifier, if the given worker
   * requires one.
   * 
   * @param tasks
   *          new tasks
   * @param jobRun
   *          the JobRun
   * @param workflowRunId
   *          id of workflow run in which these task was created.
   * @param worker
   *          the worker for which this task was created
   */
  public static void setAdditionalTaskProperties(final List<Task> tasks, final JobRun jobRun,
    final String workflowRunId, final WorkerDefinition worker) {
    String qualifierSlot = null;
    for (final Input input : worker.getInput()) {
      if (input.getModes().contains(InputMode.QUALIFIER)) {
        qualifierSlot = input.getName();
      }
    }
    for (final Task task : tasks) {
      task.getProperties().put(Task.PROPERTY_WORKFLOW_RUN_ID, workflowRunId);
      task.getProperties().put(Task.PROPERTY_JOB_RUN_ID, jobRun.getJobRunId());
      task.getProperties().put(Task.PROPERTY_JOB_NAME, jobRun.getJobName());
      if (qualifierSlot != null) {
        final List<BulkInfo> inputBulks = task.getInputBulks().get(qualifierSlot);
        if (inputBulks != null && !inputBulks.isEmpty()) {
          final BulkInfo firstBulk = inputBulks.get(0);
          task.setQualifier(firstBulk.getObjectName());
        }
      }
      if (!task.getParameters().containsKey(JobManagerConstants.SYSTEM_PARAMETER_SESSION_ID)) {
        task.getParameters().put(JobManagerConstants.SYSTEM_PARAMETER_SESSION_ID, UUID.randomUUID().toString());
      }
      final String deliverDely = getDeliverDely(jobRun, worker.getName());
      if (deliverDely != null) {
        task.getProperties().put(Task.PROPERTY_DELIVERY_DELAY, deliverDely);
      }
    }
  }

  private static String getDeliverDely(final JobRun jobRun, final String workerName) {
    String deliverDelay = null;
    final JobDefinition jobDefinition = jobRun.getJobDefinition();
    if (jobDefinition.getTaskDelayWorkers().contains(workerName)) {
      deliverDelay = Long.toString(jobDefinition.getTaskDelay());
    }
    return deliverDelay;
  }
}
