/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Andreas Schank (Attensity Europe GmbH) - initial implementation
 **********************************************************************************************************************/
package org.eclipse.smila.objectstore.filesystem;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.channels.FileLock;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Properties;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.regex.Pattern;

import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.AnySeq;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.objectstore.InvalidStoreNameException;
import org.eclipse.smila.objectstore.NoSuchObjectException;
import org.eclipse.smila.objectstore.NoSuchStoreException;
import org.eclipse.smila.objectstore.ObjectStoreException;
import org.eclipse.smila.objectstore.ObjectStoreService;
import org.eclipse.smila.objectstore.StoreExistsException;
import org.eclipse.smila.objectstore.StoreObject;
import org.eclipse.smila.objectstore.StoreOutputStream;
import org.eclipse.smila.utils.config.ConfigUtils;
import org.eclipse.smila.utils.config.ConfigurationLoadException;
import org.eclipse.smila.utils.workspace.WorkspaceHelper;
import org.osgi.service.component.ComponentContext;

/**
 * <p>
 * A file system based implementation of {@link ObjectStoreService}.
 * </p>
 * 
 * <p>
 * The following rules apply to this store implementation:
 * <ul>
 * <li>Store
 * <ul>
 * <li>A store is represented by a plain directory in the base store path.</li>
 * <li>A store can include plain files or a hierarchy of files that can be listed with
 * {@link #getStoreObjectInfos(String, String)}.</li>
 * <li>valid store names can currently contain up to 256 characters, including a-z and A-Z, as well as digits and the
 * hyphen ('-'), (as a regular expression: "[a-zA-Z0-9-]{0,256}").
 * </ul>
 * </li>
 * <li>Object
 * <ul>
 * <li>An object represents a file in a store.</li>
 * <li>An object will be <em>invisible</em> until the corresponding {@link SimpleStoreOutputStream} is closed or an
 * invocation of {@link #writeObject(String, String)} method succeeded.
 * <ul>
 * <li>exception: an invocation of the {@link #appendToObject(String, String, byte[])} method will append directly to
 * the visible object.</li>
 * </ul>
 * </ul>
 * </li>
 * </ul>
 */
public class SimpleObjectStoreService implements ObjectStoreService {

  /** the regular expression to determine whether store names are valid. */
  public static final String VALID_STORENAME_EXPRESSION = "[a-zA-Z0-9-]{0,256}";

  /** The Constant BUNDLE_ID. */
  public static final String BUNDLE_ID = "org.eclipse.smila.objectstore.filesystem";

  /** property for root of object store. */
  public static final String PROPERTY_ROOT_PATH = "root.path";

  /** property for file locking requested. */
  public static final String PROPERTY_FILE_LOCKING = "file.locking";

  /** concurrent hash map to lock for close() and append calls. */
  static final ConcurrentHashMap<File, AtomicInteger> LOCK_MAP = new ConcurrentHashMap<File, AtomicInteger>();

  /** object ids or store names can contain slashes to support hierarchies. */
  private static final String PATH_SEPERATOR = "/";

  /** The corresponding pattern to check for valid store names. */
  private static final Pattern VALID_STORENAME_PATTERN = Pattern.compile(VALID_STORENAME_EXPRESSION);

  /** the file name suffix for shadow files. */
  private static final String SHADOW_FILE_SUFFIX = ".~shadow-file~";

  /** the root of the object store. */
  private File _rootStorePath;

  /** the base path to the object store. */
  private File _visibleStorePath;

  /** is file locking active? */
  private boolean _fileLockingRequested;

  /** private log. */
  private final Log _log = LogFactory.getLog(getClass());

  /**
   * Cleans the temporary stores that were open when the system went down.
   * 
   * @throws IOException
   *           errors creating working dir.
   */
  protected void activate(final ComponentContext context) throws IOException {
    _rootStorePath = WorkspaceHelper.createWorkingDir(BUNDLE_ID);
    InputStream configFileStream = null;
    try {
      final File configFile = ConfigUtils.getConfigFile(BUNDLE_ID, "objectstoreservice.properties");
      if (configFile != null && configFile.exists()) {
        configFileStream = new BufferedInputStream(new FileInputStream(configFile));
        final Properties props = new Properties();
        props.load(configFileStream);
        final String storeRootPath = props.getProperty(PROPERTY_ROOT_PATH);
        if (storeRootPath != null && !"".equals(storeRootPath)) {
          _rootStorePath = new File(storeRootPath);
        }
        final String fileLockingRequest = props.getProperty(PROPERTY_FILE_LOCKING);
        if (fileLockingRequest != null) {
          setFileLockingRequested(Boolean.parseBoolean(fileLockingRequest));
        }
      } else {
        _log.info("No object store properties found.");
      }
    } catch (final IOException e) {
      _log.warn("Could not load object store properties. Defaulting store root to '"
        + _rootStorePath.getCanonicalPath() + "'.");
    } catch (final ConfigurationLoadException e) {
      _log.warn("Could not load object store properties. Defaulting store root to '"
        + _rootStorePath.getCanonicalPath() + "'.");
    } finally {
      if (configFileStream != null) {
        IOUtils.closeQuietly(configFileStream);
      }
    }

    if (_log.isInfoEnabled()) {
      _log.info("Setting objectstore root to '" + _rootStorePath.getCanonicalPath() + "'.");
    }
    _visibleStorePath = new File(_rootStorePath, "objectstore");
  }

  /** {@inheritDoc} */
  @Override
  public Collection<String> getStoreNames() throws ObjectStoreException {
    final Collection<String> storeNames = new ArrayList<String>();
    final File[] files = _visibleStorePath.listFiles();
    if (files != null) {
      for (final File file : files) {
        if (file.isDirectory()) {
          storeNames.add(file.getName());
        }
      }
    }
    return storeNames;
  }

  /** {@inheritDoc} */
  @Override
  public void ensureStore(final String storeName) throws ObjectStoreException {
    validateStoreName(storeName);
    final File store = new File(_visibleStorePath, storeName);
    if (store.isDirectory()) {
      return;
    } else {
      if (!store.mkdirs()) {
        throw new ObjectStoreException("Cannot create store '" + storeName + "'.");
      }
    }
  }

  /** {@inheritDoc} */
  @Override
  public synchronized boolean isValidStoreName(final String storeName) {
    return VALID_STORENAME_PATTERN.matcher(storeName).matches();
  }

  /** {@inheritDoc} */
  @Override
  public void createStore(final String storeName, final AnyMap storeProperties) throws ObjectStoreException {
    validateStoreName(storeName);
    final File store = new File(_visibleStorePath, storeName);
    if (store.exists()) {
      throw new StoreExistsException("Store with name '" + storeName + "' already exists.");
    }
    if (!store.mkdirs()) {
      throw new ObjectStoreException("Cannot create store with name '" + storeName + "'.");
    }
  }

  /** {@inheritDoc} */
  @Override
  public AnyMap getStoreProperties(final String storeName) throws ObjectStoreException {
    return DataFactory.DEFAULT.createAnyMap();
  }

  /** {@inheritDoc} */
  @Override
  public AnyMap getStoreInfo(final String storeName, final boolean includeObjectInfos) throws ObjectStoreException {
    validateStore(storeName);
    final AnyMap anyMap = DataFactory.DEFAULT.createAnyMap();
    final File store = new File(_visibleStorePath, storeName);
    anyMap.put(KEY_STORE_NAME, storeName);
    anyMap.put(KEY_STORE_PROPERTIES, getStoreProperties(storeName));
    final Collection<StoreObject> objectInfoList = listObjectInfos(store);
    anyMap.put(KEY_OBJECT_COUNT, objectInfoList.size());
    final AnySeq objects = DataFactory.DEFAULT.createAnySeq();
    int size = 0;
    for (final StoreObject info : objectInfoList) {
      objects.add(info.toAny());
      size += info.getSize();
    }
    anyMap.put(KEY_SIZE, size);
    if (includeObjectInfos) {
      anyMap.put(KEY_OBJECTS, objects);
    }
    return anyMap;
  }

  /** {@inheritDoc} */
  @Override
  public boolean existsStore(final String storeName) throws ObjectStoreException {
    final File store = new File(_visibleStorePath, storeName);
    return store.isDirectory();
  }

  /** {@inheritDoc} */
  @Override
  public void removeStore(final String storeName) throws ObjectStoreException {
    final File store = new File(_visibleStorePath, storeName);
    if (!store.exists()) {
      return;
    }
    try {
      FileUtils.deleteDirectory(store);
    } catch (final IOException e) {
      throw new ObjectStoreException("Cannot delete store '" + storeName + "'.", e);
    }
  }

  /** {@inheritDoc} */
  @Override
  public void removeAllStores() throws ObjectStoreException {
    try {
      FileUtils.deleteDirectory(_visibleStorePath);
    } catch (final IOException e) {
      throw new ObjectStoreException("Cannot remove stores.", e);
    }
  }

  /** {@inheritDoc} */
  @Override
  public void clearStore(final String storeName) throws ObjectStoreException {
    validateStore(storeName);
    final File store = new File(_visibleStorePath, storeName);
    try {
      FileUtils.cleanDirectory(store);
    } catch (final IOException e) {
      throw new ObjectStoreException("Cannot clear store '" + storeName + "'.", e);
    }
  }

  /** {@inheritDoc} */
  @Override
  public Collection<StoreObject> getStoreObjectInfos(final String storeName) throws ObjectStoreException {
    return getStoreObjectInfos(storeName, null);
  }

  /** {@inheritDoc} */
  @Override
  public Collection<StoreObject> getStoreObjectInfos(final String storeName, final String objectIdPrefix)
    throws ObjectStoreException {
    validateStore(storeName);
    final File store = new File(_visibleStorePath, storeName);
    return listObjectInfos(store, null, objectIdPrefix);
  }

  @Override
  public long countStoreObjects(final String storeName, final String objectIdPrefix) throws ObjectStoreException {
    validateStore(storeName);
    final File store = new File(_visibleStorePath, storeName);
    return countObjectInfos(store, null, objectIdPrefix);
  }

  /** {@inheritDoc} */
  @Override
  public Collection<String> getPrefixes(final String storeName, final String objectIdPrefix)
    throws ObjectStoreException {
    validateStore(storeName);
    File directory = new File(_visibleStorePath, storeName);
    final Collection<String> list = new ArrayList<String>();
    String dirPath = null;
    String filePrefix = null;

    if (objectIdPrefix != null) {
      final int index = objectIdPrefix.lastIndexOf(PATH_SEPERATOR);
      if (index > -1) {
        dirPath = objectIdPrefix.substring(0, index + 1);
        if (index + 1 < objectIdPrefix.length()) {
          filePrefix = objectIdPrefix.substring(index + 1);
        }
      } else {
        filePrefix = objectIdPrefix;
      }
    }

    if (dirPath != null) {
      directory = new File(directory, dirPath);
    } else {
      dirPath = "";
    }

    final File[] fileList = directory.listFiles();
    if (fileList != null) {
      for (final File file : fileList) {
        String fileId = file.getName();
        if (file.isDirectory()) {
          fileId += PATH_SEPERATOR;
        }
        final boolean isShadowFile = file.getName().endsWith(SHADOW_FILE_SUFFIX);
        if (!isShadowFile) {
          if (filePrefix == null || fileId.startsWith(filePrefix)) {
            list.add(dirPath + fileId);
          }
        }
      }
    }

    return list;
  }

  /** {@inheritDoc} */
  @Override
  public byte[] getObject(final String storeName, final String objectId) throws ObjectStoreException {
    final InputStream is = readObject(storeName, objectId);
    try {
      return IOUtils.toByteArray(is);
    } catch (final IOException e) {
      throw new ObjectStoreException("Could not read data from object '" + objectId + "' in store '" + storeName
        + "'.", e);
    } finally {
      IOUtils.closeQuietly(is);
    }
  }

  /** {@inheritDoc} */
  @Override
  public InputStream readObject(final String storeName, final String objectId) throws ObjectStoreException {
    validateStore(storeName);
    final File store = new File(_visibleStorePath, storeName);
    final File object = new File(store, objectId);
    if (!object.exists()) {
      throw new NoSuchObjectException("Object with id '" + objectId + "' does not exist in store '" + storeName
        + "'.");
    }
    try {
      return FileUtils.openInputStream(object);
    } catch (final IOException e) {
      throw new ObjectStoreException("Cannot read object with id '" + objectId + "' from store '" + storeName
        + "'.", e);
    }
  }

  /** {@inheritDoc} */
  @Override
  public void putObject(final String storeName, final String objectId, final byte[] data)
    throws ObjectStoreException {
    writeToObject(storeName, objectId, data);
  }

  /** {@inheritDoc} */
  @Override
  public void appendToObject(final String storeName, final String objectId, final byte[] data)
    throws ObjectStoreException {
    validateStore(storeName);
    FileOutputStream os = null;
    try {
      final File store = new File(_visibleStorePath, storeName);
      final File object = new File(store, objectId);
      if (!object.getParentFile().equals(store)) {
        object.getParentFile().mkdirs();
      }
      object.createNewFile();
      final AtomicInteger lock = new AtomicInteger(0);
      boolean written = false;
      do {
        AtomicInteger currentLock = SimpleObjectStoreService.LOCK_MAP.putIfAbsent(object, lock);
        if (currentLock == null) {
          currentLock = lock;
        }
        currentLock.incrementAndGet();
        // we synchronize here on an AtomicInteger, which may be reported in FindBugs,
        // the reason is: we want to synchronize access only to the same file, not the whole class
        synchronized (currentLock) {
          // check that no one deleted the lock...
          if (SimpleObjectStoreService.LOCK_MAP.get(object) == currentLock) {
            os = new FileOutputStream(object, true);
            final FileChannel fc = os.getChannel();
            FileLock fileLock = null;
            if (isFileLockingRequested()) {
              fileLock = fc.lock();
            }
            try {
              fc.write(ByteBuffer.wrap(data));
              fc.force(true);
              written = true;
            } finally {
              if (fileLock != null) {
                fileLock.release();
              }
              fc.close();
            }
            // clean up...
            if (currentLock.decrementAndGet() <= 0) {
              SimpleObjectStoreService.LOCK_MAP.remove(object);
            }
          }
        }
      } while (!written);
    } catch (final FileNotFoundException e) {
      throw new NoSuchObjectException("Object with id '" + objectId + "' does not exist in store '" + storeName
        + "'.", e);
    } catch (final IOException e) {
      throw new ObjectStoreException(
        "Cannot write object with id '" + objectId + "' in store '" + storeName + "'.", e);
    } finally {
      if (os != null) {
        IOUtils.closeQuietly(os);
      }
    }
  }

  /** {@inheritDoc} */
  @Override
  public boolean existsObject(final String storeName, final String objectId) throws ObjectStoreException {
    final File store = new File(_visibleStorePath, storeName);
    final File object = new File(store, objectId);
    return object.exists();
  }

  /** {@inheritDoc} */
  @Override
  public void removeObject(final String storeName, final String objectId) throws ObjectStoreException {
    final File store = new File(_visibleStorePath, storeName);
    final File object = new File(store, objectId);
    if (!object.exists()) {
      return;
    }
    if (!object.delete()) {
      throw new ObjectStoreException("Cannot remove object with id '" + objectId + "' from store '" + storeName
        + "'.");
    }
    removeEmptyParentFolders(object.getParentFile(), store);
  }

  @Override
  public void removeObjects(final String storeName, final String objectIdPrefix) throws ObjectStoreException {
    if (objectIdPrefix.isEmpty()) {
      clearStore(storeName);
      return;
    }
    final File store = new File(_visibleStorePath, storeName);
    final int lastSlash = objectIdPrefix.lastIndexOf(PATH_SEPERATOR);
    final File dirToDeleteIn = lastSlash > 0 ? new File(store, objectIdPrefix.substring(0, lastSlash)) : store;
    if (lastSlash == objectIdPrefix.length() - 1) {
      try {
        FileUtils.deleteDirectory(dirToDeleteIn);
        return;
      } catch (final IOException ex) {
        throw new ObjectStoreException("Could not remove objects with prefix " + objectIdPrefix, ex);
      }
    }
    final String filePrefix = lastSlash > 0 ? objectIdPrefix.substring(lastSlash + 1) : objectIdPrefix;
    final File[] files = dirToDeleteIn.listFiles();
    for (final File file : files) {
      if (file.getName().startsWith(filePrefix)) {
        if (!FileUtils.deleteQuietly(file)) {
          throw new ObjectStoreException("Could not remove objects with prefix " + objectIdPrefix);
        }
      }
    }
  }

  /** {@inheritDoc} */
  @Override
  public StoreObject getObjectInfo(final String storeName, final String objectId) throws ObjectStoreException {
    validateStore(storeName);
    final File store = new File(_visibleStorePath, storeName);
    final File object = new File(store, objectId);
    if (!object.exists()) {
      throw new NoSuchObjectException("Object with id '" + objectId + "' does not exist in store '" + storeName
        + "'.");
    }
    return new SimpleObjectInfo(object, objectId);
  }

  /**
   * {@inheritDoc}
   * 
   * This method is not supported in {@link SimpleObjectStoreService}.
   */
  @Override
  public void finishObject(final String storeName, final String objectId) throws ObjectStoreException {
    ; //
  }

  /**
   * Sets whether file locking is requested.
   * 
   * @param fileLocking
   *          'true' if file locking is requested, 'false' if not.
   */
  protected void setFileLockingRequested(final boolean fileLocking) {
    this._fileLockingRequested = fileLocking;
  }

  /**
   * @return 'true' if file locking is requested, 'false' if not.
   */
  public boolean isFileLockingRequested() {
    return _fileLockingRequested;
  }

  /**
   * Validates that the store name is valid and a store with the given name exists, throws an exception if it does not
   * exist.
   * 
   * @param storeName
   *          the name of the store to check.
   * @throws ObjectStoreException
   *           validation failed.
   */
  protected void validateStore(final String storeName) throws ObjectStoreException {
    validateStoreName(storeName);
    if (!existsStore(storeName)) {
      throw new NoSuchStoreException("Store with name '" + storeName + "' does not exist.");
    }
  }

  /**
   * Validates that the storeName is a valid one, throws an exception if it does not exist.
   * 
   * @param storeName
   *          the name to check.
   * @throws InvalidStoreNameException
   *           validation failed.
   */
  protected void validateStoreName(final String storeName) throws InvalidStoreNameException {
    if (!isValidStoreName(storeName)) {
      throw new InvalidStoreNameException("Store name '" + storeName
        + "' is invalid. Store names must satisfy the following regular expression: '" + VALID_STORENAME_EXPRESSION
        + "'.");
    }
  }

  /**
   * Puts data to an object in the store.
   * 
   * @param storeName
   *          the name of the store
   * @param objectId
   *          the object id
   * @param data
   *          the data to put (or append to) the object
   * @throws ObjectStoreException
   */
  protected void writeToObject(final String storeName, final String objectId, final byte[] data)
    throws ObjectStoreException {
    final StoreOutputStream sos = writeObject(storeName, objectId);
    final BufferedOutputStream bos = new BufferedOutputStream(sos);
    try {
      bos.write(data);
      bos.close();
    } catch (final IOException e) {
      sos.abort();
      throw new ObjectStoreException(
        "Cannot store object with id '" + objectId + "' in store '" + storeName + "'.", e);
    } finally {
      IOUtils.closeQuietly(bos);
    }
  }

  /** {@inheritDoc} */
  @Override
  public StoreOutputStream writeObject(final String storeName, final String objectId) throws ObjectStoreException {
    validateStore(storeName);
    try {
      final File store = new File(_visibleStorePath, storeName);
      final File object = new File(store, objectId);
      final File shadowFile = new File(store, objectId + UUID.randomUUID() + SHADOW_FILE_SUFFIX);
      createShadowFile(shadowFile);
      return new SimpleStoreOutputStream(shadowFile, object, isFileLockingRequested());
    } catch (final IOException e) {
      throw new ObjectStoreException(
        "Cannot store object with id '" + objectId + "' in store '" + storeName + "'.", e);
    }
  }

  /**
   * creates a shadow file and takes care that the file is created, repeats creation if someone in between deleted the
   * directory...
   */
  private void createShadowFile(final File shadowFile) throws IOException {
    final int retriesOnParallelDirectoryDeletion = 10;
    int retries = 0;
    for (;;) {
      try {
        shadowFile.getParentFile().mkdirs();
        shadowFile.createNewFile();
        return;
      } catch (final IOException e) {
        if (++retries > retriesOnParallelDirectoryDeletion) {
          throw e;
        }
      }
    }
  }

  /** lists all plain files (directories will be ignored) of a directory. */
  private Collection<StoreObject> listObjectInfos(final File directory) {
    return listObjectInfos(directory, null, null);
  }

  /** lists all plain files (directories will be ignored) of a directory. The ID is prefixed by the given path. */
  private Collection<StoreObject> listObjectInfos(final File directory, final String path, final String idPrefix) {
    final String dirPath = path == null ? "" : path + PATH_SEPERATOR;
    if (directory.isDirectory()) {
      final File[] fileList = directory.listFiles();
      if (fileList != null) {
        return listObjectInfos(dirPath, idPrefix, fileList);
      }
    }
    return Collections.emptyList();
  }

  /** convert files to StoreObjects and recurse into subdirectories. */
  private Collection<StoreObject> listObjectInfos(final String dirPath, final String idPrefix, final File[] fileList) {
    final Collection<StoreObject> list = new ArrayList<StoreObject>();
    for (final File file : fileList) {
      final String fileId = dirPath + file.getName();
      final boolean isShadowFile = file.getName().endsWith(SHADOW_FILE_SUFFIX);
      if (file.isFile() && !isShadowFile) {
        if (idPrefix == null || fileId.startsWith(idPrefix)) {
          list.add(new SimpleObjectInfo(file, fileId));
        }
      } else if (file.isDirectory()) {
        if (idPrefix == null || fileId.startsWith(idPrefix) || idPrefix.startsWith(fileId)) {
          list.addAll(listObjectInfos(file, fileId, idPrefix));
        }
      }
    }
    return list;
  }

  /** count all plain files (directories will be ignored) of a directory. The ID is prefixed by the given path. */
  private long countObjectInfos(final File directory, final String path, final String idPrefix) {
    final String dirPath = path == null ? "" : path + PATH_SEPERATOR;
    if (directory.isDirectory()) {
      final File[] fileList = directory.listFiles();
      if (fileList != null) {
        return countObjectInfos(dirPath, idPrefix, fileList);
      }
    }
    return 0;
  }

  /** count real files and recurse into subdirectories. */
  private long countObjectInfos(final String dirPath, final String idPrefix, final File[] fileList) {
    long count = 0;
    for (final File file : fileList) {
      final String fileId = dirPath + file.getName();
      final boolean isShadowFile = file.getName().endsWith(SHADOW_FILE_SUFFIX);
      if (file.isFile() && !isShadowFile) {
        if (idPrefix == null || fileId.startsWith(idPrefix)) {
          count++;
        }
      } else if (file.isDirectory()) {
        if (idPrefix == null || fileId.startsWith(idPrefix) || idPrefix.startsWith(fileId)) {
          count += countObjectInfos(file, fileId, idPrefix);
        }
      }
    }
    return count;
  }

  /**
   * Delete empty folders up to (but not including) the root folder.
   * 
   * @param folder
   *          the starting folder
   * @param rootFolder
   *          the root folder up to (but not including) which the empty directories will be deleted.
   */
  static void removeEmptyParentFolders(final File folder, final File rootFolder) {
    File actual = folder;
    while (actual != null && !actual.equals(rootFolder)) {
      if (!actual.delete()) {
        return;
      }
      actual = actual.getParentFile();
    }
  }
}
