/***********************************************************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Juergen Schumacher (Attensity Europe GmbH) - initial API and implementation 
 **********************************************************************************************************************/

package org.eclipse.smila.restapi;

import java.util.Collection;
import java.util.HashSet;

import org.eclipse.smila.clusterconfig.ClusterConfigService;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.AnySeq;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.http.server.json.JsonRequestHandler;
import org.eclipse.smila.workermanager.WorkerManager;
import org.eclipse.smila.zookeeper.ZooKeeperService;

/**
 * Implements the handling of HTTP state debug requests.
 */
public class DebugHandler extends JsonRequestHandler {
  /** WorkerManager reference. */
  private WorkerManager _workerManager;

  /** ClusterConfigService. */
  private ClusterConfigService _clusterService;

  /** zookeeper service. */
  private ZooKeeperService _zkService;

  /**
   * {@inheritDoc}
   */
  @Override
  public Object process(final String method, final String requestUri, final Record inputRecord) {
    final AnyMap result = FACTORY.createAnyMap();
    if (_workerManager != null) {
      result.put("workerManager", _workerManager.getInfo());
    }
    if (_zkService != null && _clusterService != null) {
      result.put("zookeeper", buildZookeeperSection());
    }
    return result;
  }

  /**
   * Build "zookeeper" section.
   * 
   * @return Any, representing zookeeper section
   */
  private AnyMap buildZookeeperSection() {
    try {
      return _zkService.getServerState();
    } catch (final Exception ex) {
      return exceptionToAny(new IllegalArgumentException("Error creating zookeeper section", ex));
    }
  }

  /**
   * @param wm
   *          new WorkerManager to set
   */
  public void setWorkerManager(final WorkerManager wm) {
    _workerManager = wm;
  }

  /**
   * @param wm
   *          new WorkerManager to remove
   */
  public void unsetWorkerManager(final WorkerManager wm) {
    if (_workerManager == wm) {
      _workerManager = null;
    }
  }

  /**
   * set new ClusterConfigService. To be called by DS runtime before activation.
   * 
   * @param clusterConfigService
   *          new ClusterConfigService
   */
  public void setClusterConfigService(final ClusterConfigService clusterConfigService) {
    _clusterService = clusterConfigService;
  }

  /**
   * remove a ClusterConfigService. To be called by DS runtime after deactivation.
   */
  public void unsetClusterConfigService(final ClusterConfigService clusterConfigService) {
    if (_clusterService == clusterConfigService) {
      _clusterService = null;
    }
  }

  /**
   * method for DS to set a service reference.
   * 
   * @param zkService
   *          ZooKeeperService reference.
   */
  public void setZooKeeperService(final ZooKeeperService zkService) {
    _zkService = zkService;
  }

  /**
   * method for DS to unset a service reference.
   * 
   * @param zkService
   *          ZooKeeperService reference.
   */
  public void unsetZooKeeperService(final ZooKeeperService zkService) {
    if (_zkService == zkService) {
      _zkService = null;
    }
  }

  /**
   * {@inheritDoc}
   * 
   * GET is currently the only valid method.
   */
  @Override
  protected boolean isValidMethod(final String method, final String requestUri) {
    return "GET".equals(method);
  }

  /**
   * convert an exception to an any object.
   * 
   * @param e
   *          exception to convert
   * @return any representation of exception
   */
  private AnyMap exceptionToAny(final Throwable e) {
    try {
      final AnyMap error = FACTORY.createAnyMap();
      error.put("error", exceptionToAny(e, new HashSet<String>()));
      return error;
    } catch (final Exception drecksEx) {
      throw new IllegalArgumentException("Error converting exception \"" + e.toString() + "\"", drecksEx);
    }
  }

  /**
   * convert an exception to an any object. stop in stacktrace printing when hitting known lines again.
   * 
   * @param e
   *          exception to convert
   * @param visitedLines
   *          lines that have been added to stacktraces before.
   * @return any representation of exception
   * @throws Exception
   *           creating the object.
   */
  private AnyMap exceptionToAny(final Throwable e, final Collection<String> visitedLines) throws Exception {
    final AnyMap any = FACTORY.createAnyMap();
    any.put("type", e.getClass().getName());
    if (e.getMessage() != null) {
      any.put("message", e.getMessage());
    }
    final AnySeq st = FACTORY.createAnySeq();
    for (final StackTraceElement stElement : e.getStackTrace()) {
      final String line = stElement.toString();
      st.add(line);
      if (!visitedLines.add(line)) {
        st.add("...");
        break;
      }
    }
    any.put("at", st);
    if (e.getCause() != null && e.getCause() != e) {
      any.put("causedBy", exceptionToAny(e.getCause(), visitedLines));
    }
    return any;
  }

}
