/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Andreas Weber (Attensity Europe GmbH) - initial implementation
 **********************************************************************************************************************/

package org.eclipse.smila.workermanager.internal;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.concurrent.CompletionService;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorCompletionService;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

/**
 * Thread pool for executing TaskWorkers.
 */
public class WorkerPool {

  /** local logger. */
  private final Log _log = LogFactory.getLog(getClass());

  /** executor with thread pool. */
  private final ExecutorService _executor;

  /** used to get results after execution is finished. */
  private final CompletionService<WorkerRunner> _completionService;

  /**
   * Worker pool with default configuration.
   */
  public WorkerPool() {
    _executor = Executors.newCachedThreadPool();
    _completionService = new ExecutorCompletionService<WorkerRunner>(_executor);
  }

  /**
   * @return all completed TaskWorker instances
   */
  public Collection<WorkerRunner> getCompleted() {
    final List<WorkerRunner> completed = new ArrayList<WorkerRunner>();
    Future<WorkerRunner> f = null;
    do {
      f = _completionService.poll();
      if (f != null) {
        try {
          completed.add(f.get(0, TimeUnit.MILLISECONDS));
        } catch (final ExecutionException e) {
          // The TaskWorker execution threw an exception.
          // This shouldn't happen, cause all exceptions should be catched in TaskWorker execution.
          _log.error("TaskWorker execution threw an exception", e);
        } catch (final InterruptedException e) {
          // this cannot happen because we use  _completionService.poll())
          _log.warn("InterruptedException getting TaskWorker result", e);
          Thread.currentThread().interrupt();
        } catch (final TimeoutException e) {
          // this cannot happen because we use  _completionService.poll())
          _log.warn("TimeoutException getting TaskWorker result", e);
        }
      }
    } while (f != null);
    return completed;
  }

  /**
   * @param taskWorker
   *          the TaskWorker to be executed.
   * 
   */
  public void submit(final WorkerRunner taskWorker) {
    _completionService.submit(taskWorker);
  }

  /** closes the thread pool. */
  public void close() {
    if (_executor != null) {
      _executor.shutdown();
    }
  }

  /** for getting thread pool info. */
  ExecutorService getThreadPool() {
    return _executor;
  }

}
