/*******************************************************************************
 * Copyright (c) 2008, 2009 empolis GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Juergen Schumacher (empolis GmbH) - initial API and implementation, Andreas Schank (Attensity Europe
 * GmbH) changed the way pipelets are discovered.
 *******************************************************************************/

package org.eclipse.smila.processing;

import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Map;
import java.util.TreeMap;

import org.apache.commons.io.IOUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.smila.common.definitions.ParameterDefinition;
import org.eclipse.smila.common.exceptions.InvalidDefinitionException;
import org.eclipse.smila.datamodel.Any;
import org.eclipse.smila.datamodel.Any.ValueType;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.AnySeq;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.datamodel.ipc.IpcAnyReader;
import org.osgi.framework.Bundle;
import org.osgi.framework.BundleContext;
import org.osgi.framework.BundleEvent;
import org.osgi.framework.SynchronousBundleListener;
import org.osgi.service.component.ComponentContext;

/**
 * Implementation of PipeletTracker service. Registered as a OSGi services by DS. It works as a
 * SynchronousBundleListener to get notified about starting and stopping bundles.
 * 
 * @author jschumacher
 * 
 */
public class PipeletTrackerImpl implements SynchronousBundleListener, PipeletTracker {
  /**
   * Directory name for pipelet description files: "SMILA-INF".
   */
  private static final String DIRECTORY_SMILA_PIPELET_DESCRIPTIONS = "SMILA-INF";

  /**
   * Suffix of filenames for pipelet description files: "SMILA-INF".
   */
  private static final String SUFFIX_SMILA_PIPELET_DESCRIPTIONS = "*.json";

  /**
   * local logger.
   */
  private final Log _log = LogFactory.getLog(PipeletTrackerImpl.class);

  /**
   * currently known pipelet classes: map of bundle name to map of class name to pipelet class.
   */
  private final Map<String, Map<String, Class<? extends Pipelet>>> _knownPipelets =
    new HashMap<String, Map<String, Class<? extends Pipelet>>>();

  /**
   * currently known pipelet descriptions: map of bundle name to map of class name to pipelet description.
   */
  private final Map<String, Map<String, AnyMap>> _knownPipeletDescriptions =
    new HashMap<String, Map<String, AnyMap>>();

  /**
   * registered listener for tracker events.
   */
  private final Collection<PipeletTrackerListener> _listeners = new ArrayList<PipeletTrackerListener>();

  /**
   * IPC Any Reader to read pipelet configuration files.
   */
  private final IpcAnyReader _anyReader = new IpcAnyReader();

  /**
   * {@inheritDoc}
   * 
   * @see org.eclipse.smila.processing.PipeletTracker#getRegisteredPipelets()
   */
  @Override
  public Map<String, Class<? extends Pipelet>> getRegisteredPipelets() {
    final Map<String, Class<? extends Pipelet>> registeredPipelets =
      new TreeMap<String, Class<? extends Pipelet>>();
    for (final Map<String, Class<? extends Pipelet>> bundlePipelets : _knownPipelets.values()) {
      registeredPipelets.putAll(bundlePipelets);
    }
    return registeredPipelets;
  }

  /** {@inheritDoc} */
  @Override
  public Map<String, AnyMap> getRegisteredPipeletDescriptions() {
    final Map<String, AnyMap> registeredPipeletDescriptions = new TreeMap<String, AnyMap>();
    for (final Map<String, AnyMap> bundlePipelets : _knownPipeletDescriptions.values()) {
      registeredPipeletDescriptions.putAll(bundlePipelets);
    }
    return registeredPipeletDescriptions;
  }

  /**
   * {@inheritDoc}
   * 
   * @see org.eclipse.smila.processing.PipeletTracker #addListener(org.eclipse.smila.processing.PipeletTrackerListener)
   */
  @Override
  public void addListener(final PipeletTrackerListener listener) {
    _listeners.add(listener);
    for (final Map<String, Class<? extends Pipelet>> pipeletClassNames : _knownPipelets.values()) {
      listener.pipeletsAdded(pipeletClassNames);
    }
  }

  /**
   * {@inheritDoc}
   * 
   * @see org.eclipse.smila.processing.PipeletTracker
   *      #removeListener(org.eclipse.smila.processing.PipeletTrackerListener)
   */
  @Override
  public void removeListener(final PipeletTrackerListener listener) {
    _listeners.remove(listener);
  }

  /**
   * activate declarative service. It registers the services a bundle listener and searches all currently active bundles
   * for pipelets and notifies all currently known listeners.
   * 
   * @param componentContext
   *          service component context.
   */
  protected void activate(final ComponentContext componentContext) {
    final BundleContext bundleContext = componentContext.getBundleContext();
    bundleContext.addBundleListener(this);
    final Bundle[] bundles = bundleContext.getBundles();
    for (final Bundle bundle : bundles) {
      if ((bundle.getState() & (Bundle.ACTIVE | Bundle.RESOLVED | Bundle.STARTING)) > 0) {
        bundleAdded(bundle);
      }
    }
  }

  /**
   * deactivate declarative service. Removes this as a bundle listener.
   * 
   * @param componentContext
   *          service component context.
   */
  protected void deactivate(final ComponentContext componentContext) {
    final BundleContext bundleContext = componentContext.getBundleContext();
    bundleContext.removeBundleListener(this);
  }

  /**
   * Check newly resolved or stopping bundles for contained Pipelets.
   * 
   * {@inheritDoc}
   * 
   * @see org.osgi.framework.BundleListener#bundleChanged(org.osgi.framework.BundleEvent)
   */
  @Override
  public void bundleChanged(final BundleEvent event) {
    switch (event.getType()) {
      case BundleEvent.RESOLVED:
        bundleAdded(event.getBundle());
        break;
      case BundleEvent.STARTING:
        bundleAdded(event.getBundle());
        break;
      case BundleEvent.STOPPING:
        bundleRemoved(event.getBundle());
        break;
      default:
        // ignore
    }
  }

  /**
   * Check bundle for contained Pipelets. It looks for a folder named "SMILA-INF" in the bundle, that contains a bunch
   * of pipelet definition files containing each at least a class name, that implement Pipelet. Finally listeners are
   * notified about the new pipelet classes.
   * 
   * @param bundle
   *          bundle to examine for pipelets.
   */
  private void bundleAdded(final Bundle bundle) {
    final String bundleName = bundle.getSymbolicName();
    if (_knownPipelets.containsKey(bundleName)) {
      if (_log.isDebugEnabled()) {
        _log.debug("Pipelets from bundle " + bundleName + " have been loaded already, skipping.");
      }
    } else {
      final Map<String, Class<? extends Pipelet>> pipeletClasses = new HashMap<String, Class<? extends Pipelet>>();
      final Map<String, AnyMap> pipeletDescriptionMap = new HashMap<String, AnyMap>();
      @SuppressWarnings("unchecked")
      final Enumeration<URL> pipeletDescriptionUrls =
        bundle.findEntries(DIRECTORY_SMILA_PIPELET_DESCRIPTIONS, SUFFIX_SMILA_PIPELET_DESCRIPTIONS, true);
      if (pipeletDescriptionUrls != null) {
        while (pipeletDescriptionUrls.hasMoreElements()) {
          final URL pipeletDescriptionUrl = pipeletDescriptionUrls.nextElement();
          final String fileName = pipeletDescriptionUrl.getFile();
          try {
            final AnyMap pipeletDescriptionAny = loadPipeletDescription(pipeletDescriptionUrl);
            final String pipeletClassName = pipeletDescriptionAny.getStringValue(KEY_CLASS);
            if (_log.isDebugEnabled()) {
              _log.debug("Found pipelet desciption '" + fileName + "' (class '" + pipeletClassName
                + "') in bundle '" + bundleName + "'.");
            }
            loadPipeletClass(pipeletClasses, pipeletClassName, bundle);
            checkPipeletDescription(pipeletClasses, pipeletDescriptionAny, pipeletClassName);
            pipeletDescriptionMap.put(pipeletClassName, pipeletDescriptionAny);
          } catch (final IOException e) {
            _log.error("Cannot load pipelet description '" + fileName + "' from bundle '" + bundleName + "'.", e);
          }
        }
        if (!pipeletDescriptionMap.isEmpty()) {
          _knownPipeletDescriptions.put(bundleName, pipeletDescriptionMap);
        }
        if (!pipeletClasses.isEmpty()) {
          _knownPipelets.put(bundleName, pipeletClasses);
          for (final PipeletTrackerListener listener : _listeners) {
            listener.pipeletsAdded(pipeletClasses);
          }
        }
      }
    }
  }

  /**
   * Checks the pipelet descriptions. If they define pipelets that could not be loaded or if the parameters section has
   * invalid syntax a sequence of "errors" is attached to the description.
   * 
   * @param pipeletClasses
   *          the map of detected and loaded pipelet classes (up to now)
   * @param pipeletDescriptionAny
   *          the pipelet's description
   * @param pipeletClassName
   *          the class name of the pipelet
   */
  protected void checkPipeletDescription(final Map<String, Class<? extends Pipelet>> pipeletClasses,
    final AnyMap pipeletDescriptionAny, final String pipeletClassName) {
    if (!pipeletClasses.containsKey(pipeletClassName)) {
      // add an error if the pipelet class could not be loaded
      pipeletDescriptionAny.add(KEY_ERRORS,
        DataFactory.DEFAULT.createStringValue("Pipelet class could not be loaded."));
      _log.error("Pipelet '" + pipeletClassName + "' cannot be found.");
    }
    // check the parameters section.
    final Any parameters = pipeletDescriptionAny.get(KEY_PARAMETERS);
    if (parameters != null && !parameters.isEmpty()) {
      if (parameters.getValueType() != ValueType.SEQ) {
        pipeletDescriptionAny.add(KEY_ERRORS,
          DataFactory.DEFAULT.createStringValue("Parameters section is not of Type " + ValueType.SEQ.name()));
        _log.warn("Parameters section of pipelet '" + pipeletClassName + "' is no sequence.");
      } else {
        try {
          ParameterDefinition.parseParameters((AnySeq) parameters);
        } catch (final InvalidDefinitionException e) {
          pipeletDescriptionAny.add(KEY_ERRORS,
            DataFactory.DEFAULT.createStringValue("Parameters section is invalid. " + e.getMessage()));
          _log.warn("Parameters section of pipelet '" + pipeletClassName + "' is invalid.", e);
        }
      }
    }
  }

  /**
   * Reads the description as AnyMap from the given URL.
   * 
   * @param pipeletDescriptionUrl
   *          the URL pointing to the description.
   * @return an AnyMap representing the description.
   * @throws IOException
   *           the description could not be found or parsed.
   */
  private AnyMap loadPipeletDescription(final URL pipeletDescriptionUrl) throws IOException {
    final InputStream pipeletDescriptionStream = pipeletDescriptionUrl.openStream();
    try {
      return (AnyMap) _anyReader.readJsonStream(pipeletDescriptionStream);
    } finally {
      IOUtils.closeQuietly(pipeletDescriptionStream);
    }
  }

  /**
   * load pipelet class from bundle and add it to the map.
   * 
   * @param pipeletClasses
   *          pipelet registry.
   * @param pipeletClassName
   *          name of pipelet.
   * @param bundle
   *          bundle containing pipelet
   */
  private void loadPipeletClass(final Map<String, Class<? extends Pipelet>> pipeletClasses,
    final String pipeletClassName, final Bundle bundle) {
    final String trimmedClassName = pipeletClassName.trim();
    if (trimmedClassName.length() > 0) {
      if (_log.isDebugEnabled()) {
        _log.debug("Found pipelet class name = " + trimmedClassName);
      }
      try {
        @SuppressWarnings("unchecked")
        final Class<? extends Pipelet> pipeletClass = (Class<? extends Pipelet>) bundle.loadClass(trimmedClassName);
        pipeletClasses.put(trimmedClassName, pipeletClass);
        if (_log.isDebugEnabled()) {
          _log.debug("Pipelet class " + trimmedClassName + " loaded.");
        }
      } catch (final ClassNotFoundException ex) {
        _log.error(
          "Pipelet class " + trimmedClassName + " could not be loaded from bundle " + bundle.getSymbolicName(), ex);
      }
    }
  }

  /**
   * remove pipelets found in this bundle from list of known pipelets. Notify listeners about lost pipelets.
   * 
   * @param bundle
   *          bundle being stopped.
   */
  private void bundleRemoved(final Bundle bundle) {
    final String bundleName = bundle.getSymbolicName();
    if (_knownPipelets.containsKey(bundleName)) {
      final Map<String, Class<? extends Pipelet>> pipeletClassNames = _knownPipelets.remove(bundleName);
      for (final PipeletTrackerListener listener : _listeners) {
        listener.pipeletsRemoved(pipeletClassNames);
      }
    }
  }

}
