/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: J�rgen Schumacher, Andreas Weber, Drazen Cindric, Andreas Schank (all Attensity Europe GmbH) - initial
 * implementation
 **********************************************************************************************************************/
package org.eclipse.smila.jobmanager.httphandler;

import java.util.List;

import org.eclipse.smila.datamodel.Any;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.jobmanager.exceptions.ConfigNotFoundException;

/**
 * Implements the handling of HTTP requests on a single job run.
 * 
 * URL pattern: smila/jobmanager/jobs/-jobname-/-jobid-
 */
public class JobRunHandler extends AJobManagerHandler {

  /** The key for the details. */
  public static final String KEY_SHOW_DETAILS = "returnDetails";

  /**
   * {@inheritDoc}
   */
  @Override
  public Object process(final String method, final String requestUri, final Record inputRecord) throws Exception {
    if (isGetRequest(method)) {
      final String jobName = getJobName(requestUri);
      boolean showDetails = false;
      if (inputRecord != null && inputRecord.getMetadata().containsKey(KEY_SHOW_DETAILS)) {
        showDetails = inputRecord.getMetadata().getBooleanValue(KEY_SHOW_DETAILS).booleanValue();
      }
      final Any outputAny = getJobRunDataProvider().getJobRunData(jobName, getJobId(requestUri), showDetails);
      if (outputAny == null) {
        throw new ConfigNotFoundException("No such job run with name '" + jobName + "'.");
      }
      return outputAny;
    } else if (isDeleteRequest(method)) {
      getJobRunEngine().deleteJobRunData(getJobName(requestUri), getJobId(requestUri));
    }
    return null;
  }

  /**
   * Gets the job name from the requestUri.
   * 
   * @param requestUri
   *          the request URI.
   * @return he job name from the requestUri.
   */
  protected String getJobName(final String requestUri) {
    final List<String> uriParts = getDynamicUriParts(requestUri);
    if (uriParts.size() > 0) {
      return uriParts.get(0);
    }
    throw new IllegalArgumentException("Invalid request URI, does not match uri pattern '" + getUriPattern() + "'.");
  }

  /**
   * Gets the job id from the requestUri.
   * 
   * @param requestUri
   *          the request URI.
   * @return he job name from the requestUri.
   */
  protected String getJobId(final String requestUri) {
    final List<String> uriParts = getDynamicUriParts(requestUri);
    if (uriParts.size() > 1) {
      return uriParts.get(1);
    }
    throw new IllegalArgumentException("Invalid request URI, does not match uri pattern '" + getUriPattern() + "'.");
  }

  /**
   * {@inheritDoc}
   * 
   * GET and DELETE are currently the only valid methods.
   */
  @Override
  protected boolean isValidMethod(final String method, final String requestUri) {
    return isGetOrDeleteRequest(method);
  }

}
