/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Juergen Schumacher, Andreas Weber, Drazen Cindric, Andreas Schank (all Attensity Europe GmbH) - initial
 * implementation
 **********************************************************************************************************************/
package org.eclipse.smila.jobmanager.internal;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.eclipse.smila.jobmanager.definitions.BucketDefinition;
import org.eclipse.smila.jobmanager.definitions.DataObjectTypeDefinition;
import org.eclipse.smila.jobmanager.definitions.JobDefinition;
import org.eclipse.smila.jobmanager.definitions.WorkerDefinition;
import org.eclipse.smila.jobmanager.definitions.WorkflowDefinition;
import org.eclipse.smila.jobmanager.exceptions.PersistenceException;
import org.eclipse.smila.jobmanager.persistence.PermanentStorage;

/**
 * Handles read operations of jobmanager definitions.
 */
public class DefinitionAccessor {

  /** definition storage service to use, passed in constructor by job manager. */
  private PermanentStorage _storage;

  /** predefined data object type definitions read from configuration area. */
  private Map<String, DataObjectTypeDefinition> _configuredDataObjectTypes =
    new HashMap<String, DataObjectTypeDefinition>();

  /** predefined worker definitions read from configuration area. */
  private Map<String, WorkerDefinition> _configuredWorkers = new HashMap<String, WorkerDefinition>();

  /** predefined buckets read from configuration area. */
  private Map<String, BucketDefinition> _configuredBuckets = new HashMap<String, BucketDefinition>();

  /** predefined workflows read from configuration area. */
  private Map<String, WorkflowDefinition> _configuredWorkflows = new HashMap<String, WorkflowDefinition>();

  /** predefined jobs read from configuration area. */
  private Map<String, JobDefinition> _configuredJobs = new HashMap<String, JobDefinition>();

  /** */
  public DefinitionAccessor() {
  }

  /**
   * @param storage
   */
  public void setStorage(final PermanentStorage storage) {
    _storage = storage;
  }

  /**
   * @return all data object type names.
   */
  public Collection<String> getDataObjectTypes() {
    return new ArrayList<String>(_configuredDataObjectTypes.keySet());
  }

  /**
   * @param name
   *          the name of the data object type definition
   * @return the data object type definition for the given name, or <code>null</code> if not found
   */
  public DataObjectTypeDefinition getDataObjectType(final String name) {
    return _configuredDataObjectTypes.get(name);
  }

  /**
   * @return all worker names.
   */
  public Collection<String> getWorkers() {
    return new ArrayList<String>(_configuredWorkers.keySet());
  }

  /**
   * @param name
   *          the name of the worker definition
   * @return returns the worker definition with the given name, or <code>null</code> if not found
   */
  public WorkerDefinition getWorker(final String name) {
    return _configuredWorkers.get(name);
  }

  /**
   * @return names of preconfigured and user-defined buckets.
   * @throws PersistenceException
   *           error reading from storage.
   */
  public Collection<String> getBuckets() throws PersistenceException {
    final List<String> bucketNames = new ArrayList<String>(_configuredBuckets.keySet());
    if (_storage != null) {
      final Collection<String> storageBuckets = _storage.getBuckets();
      if (storageBuckets != null) {
        bucketNames.addAll(storageBuckets);
      }
    }
    return bucketNames;
  }

  /**
   * get a bucket from configuration (try first) or storage (if not found in configuration).
   * 
   * @param name
   *          bucket name
   * @return bucket definition.
   * @throws PersistenceException
   *           error reading from storage.
   */
  public BucketDefinition getBucket(final String name) throws PersistenceException {
    if (_configuredBuckets.containsKey(name)) {
      return _configuredBuckets.get(name);
    }
    if (_storage != null) {
      return _storage.getBucket(name);
    }
    return null;
  }

  /**
   * @return names of defined jobs.
   * @throws PersistenceException
   *           error reading from storage
   */
  public Collection<String> getJobs() throws PersistenceException {
    final Set<String> jobNames = new HashSet<String>(_configuredJobs.keySet());
    if (_storage != null) {
      final Collection<String> storageJobs = _storage.getJobs();
      if (storageJobs != null) {
        jobNames.addAll(storageJobs);
      }
    }
    return jobNames;
  }

  /**
   * get a job name from configuration or storage.
   * 
   * @param name
   *          job name
   * @return job definition.
   * @throws PersistenceException
   *           error reading from storage.
   */
  public JobDefinition getJob(final String name) throws PersistenceException {
    if (_configuredJobs.containsKey(name)) {
      return _configuredJobs.get(name);
    }
    if (_storage != null) {
      return _storage.getJob(name);
    }
    return null;
  }
  
  /**
   * @return (unique) names of defined workflows.
   * @throws PersistenceException
   *           error reading from storage
   */
  public Collection<String> getWorkflows() throws PersistenceException {
    final Set<String> workflowNames = new HashSet<String>(_configuredWorkflows.keySet());
    if (_storage != null) {
      final Collection<String> storageWorkflows = _storage.getWorkflows();
      if (storageWorkflows != null) {
        workflowNames.addAll(storageWorkflows);
      }
    }
    return workflowNames;
  }

  /**
   * get a workflow definition from configuration (try first) or storage (if not found in configuration).
   * 
   * @param name
   *          workflow name
   * @return workflow definition.
   * @throws PersistenceException
   *           error reading from storage.
   */
  public WorkflowDefinition getWorkflow(final String name) throws PersistenceException {
    if (_configuredWorkflows.containsKey(name)) {
      return _configuredWorkflows.get(name);
    }
    if (_storage != null) {
      return _storage.getWorkflow(name);
    }
    return null;
  }

  /**
   * @param configuredDataObjectTypes
   */
  public void setConfiguredDataObjectTypes(final Map<String, DataObjectTypeDefinition> configuredDataObjectTypes) {
    this._configuredDataObjectTypes = configuredDataObjectTypes;
  }

  /**
   * @param configuredWorkers
   */
  public void setConfiguredWorkers(final Map<String, WorkerDefinition> configuredWorkers) {
    this._configuredWorkers = configuredWorkers;
  }

  /**
   * @return configured buckets
   */
  public Map<String, BucketDefinition> getConfiguredBuckets() {
    return _configuredBuckets;
  }

  /**
   * @param configuredBuckets
   */
  public void setConfiguredBuckets(final Map<String, BucketDefinition> configuredBuckets) {
    this._configuredBuckets = configuredBuckets;
  }

  /**
   * @return configured workflows
   */
  public Map<String, WorkflowDefinition> getConfiguredWorkflows() {
    return _configuredWorkflows;
  }

  /**
   * @param configuredWorkflows
   */
  public void setConfiguredWorkflows(final Map<String, WorkflowDefinition> configuredWorkflows) {
    this._configuredWorkflows = configuredWorkflows;
  }

  /**
   * @return configured jobs
   */
  public Map<String, JobDefinition> getConfiguredJobs() {
    return _configuredJobs;
  }

  /**
   * @param configuredWorkflows
   */
  public void setConfiguredJobs(final Map<String, JobDefinition> configuredJobs) {
    this._configuredJobs = configuredJobs;
  }

}
