/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Juergen Schumacher, Andreas Weber, Drazen Cindric, Andreas Schank (all Attensity Europe GmbH) - initial
 * implementation
 **********************************************************************************************************************/
package org.eclipse.smila.jobmanager.persistence;

import java.util.Collection;

import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.jobmanager.definitions.BucketDefinition;
import org.eclipse.smila.jobmanager.definitions.JobDefinition;
import org.eclipse.smila.jobmanager.definitions.WorkflowDefinition;
import org.eclipse.smila.jobmanager.exceptions.PersistenceException;

/**
 * Interface for handling persistent jobmanager data.
 * 
 */
public interface PermanentStorage {

  /**
   * Write bucket definition to storage.
   * 
   * @param bucket
   *          bucket definition
   * @throws PersistenceException
   *           error writing.
   */
  void addBucket(BucketDefinition bucket) throws PersistenceException;

  /**
   * get bucket definition from storage.
   * 
   * @param name
   *          bucket name
   * @return bucket definition or <code>null</code> if bucket doesn't exist.
   * @throws PersistenceException
   *           error reading bucket definition
   */
  BucketDefinition getBucket(String name) throws PersistenceException;

  /**
   * Remove bucket definition from storage. Removing a non-existing bucket will be ignored.
   * 
   * @param name
   *          bucket name
   * @throws PersistenceException
   *           error deleting.
   */
  void removeBucket(String name) throws PersistenceException;

  /**
   * Get bucket names defined in storage.
   * 
   * @return bucket names.
   * @throws PersistenceException
   *           error reading.
   */
  Collection<String> getBuckets() throws PersistenceException;

  /**
   * Write job definition to storage.
   * 
   * @param job
   *          job definition
   * @throws PersistenceException
   *           error writing.
   */
  void addJob(JobDefinition job) throws PersistenceException;

  /**
   * get job definition from storage.
   * 
   * @param name
   *          job name
   * @return job definition or <code>null</code> if job doesn't exist.
   * @throws PersistenceException
   *           error reading job definition
   */
  JobDefinition getJob(String name) throws PersistenceException;

  /**
   * @param jobName
   *          job name
   * @return true if a job with this name is defined
   * @throws PersistenceException
   *           error checking existence.
   */
  boolean hasJob(String jobName) throws PersistenceException;

  /**
   * Get job names defined in storage.
   * 
   * @return job names.
   * @throws PersistenceException
   *           error reading.
   */
  Collection<String> getJobs() throws PersistenceException;

  /**
   * Remove job definition from storage. Removing a non-existing job will be ignored.
   * 
   * @param name
   *          job name
   * @throws PersistenceException
   *           error deleting.
   */
  void removeJob(String name) throws PersistenceException;

  /**
   * Write workflow definition to storage.
   * 
   * @param workflow
   *          workflow definition
   * @throws PersistenceException
   *           error writing.
   */
  void addWorkflow(WorkflowDefinition workflow) throws PersistenceException;

  /**
   * get workflow definition from storage.
   * 
   * @param name
   *          workflow name
   * @return workflow definition or <code>null</code> if workflow doesn't exist.
   * @throws PersistenceException
   *           error reading workflow definition
   */
  WorkflowDefinition getWorkflow(String name) throws PersistenceException;

  /**
   * Get workflow names defined in storage.
   * 
   * @return workflow names.
   * @throws PersistenceException
   *           error reading.
   */
  Collection<String> getWorkflows() throws PersistenceException;

  /**
   * Remove workflow definition from storage. Removing a non-existing workflow will be ignored.
   * 
   * @param name
   *          workflow name
   * @throws PersistenceException
   *           error deleting.
   */
  void removeWorkflow(String name) throws PersistenceException;

  /**
   * @param jobName
   *          the job for which to return the job run ids
   * @return the ids of the stored job runs for the given job. If no entries are found an empty list is returned
   * @throws PersistenceException
   *           error accessing store
   */
  Collection<String> getJobRunIds(final String jobName) throws PersistenceException;

  /**
   * Stores the job run data for the given job run. If writing to the storage fails for whatever reason, the data is
   * logged, but no exception is thrown.
   * 
   * @param jobName
   *          the job
   * @param jobRunId
   *          the job run for which to store the data
   * @param jobRunData
   *          the job run data
   */
  void storeJobRun(final String jobName, final String jobRunId, final AnyMap jobRunData);

  /**
   * @param jobName
   *          the job
   * @param jobRunId
   *          the job run for which to check if it's contained in the store
   * @return 'true' if an entry exists for the given job run, otherwise 'false'.
   * @throws PersistenceException
   *           error accessing store
   */
  boolean containsJobRun(final String jobName, final String jobRunId) throws PersistenceException;

  /**
   * @param jobName
   *          the job
   * @param jobRunId
   *          the job run id
   * @return the job run data for the given job run id. If the job run doesn't exist 'null' is returned
   * @throws PersistenceException
   *           error accessing store
   */
  AnyMap getJobRunData(String jobName, String jobRunId) throws PersistenceException;

  /**
   * Deletes the stored data for the given job run.
   * 
   * @param jobName
   *          the job
   * @param jobRunId
   *          the id of the job run for which to delete the data
   * @throws PersistenceException
   *           error accessing store
   */
  void deleteJobRunData(String jobName, String jobRunId) throws PersistenceException;

}
