/*******************************************************************************
 * Copyright (c) 2008, 2012 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Juergen Schumacher (Attensity Europe GmbH) - initial API and implementation
 *******************************************************************************/
package org.eclipse.smila.http.client.attachments;

import java.io.File;
import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.nio.charset.Charset;
import java.util.LinkedHashMap;
import java.util.Map;

import org.apache.http.entity.mime.content.ByteArrayBody;
import org.apache.http.entity.mime.content.ContentBody;
import org.apache.http.entity.mime.content.FileBody;
import org.apache.http.entity.mime.content.InputStreamBody;
import org.apache.http.entity.mime.content.StringBody;
import org.eclipse.smila.http.client.Attachments;

/**
 * Use this class to wrap possible attachment values to provide them to the
 * {@link org.eclipse.smila.http.client.RestClient} in a POST request.
 * 
 * @since 1.1.0
 */
public class AttachmentWrapper implements Attachments {

  private static final Charset CHARSET = Charset.forName("UTF-8");

  private final Map<String, ContentBody> _parts = new LinkedHashMap<String, ContentBody>();

  /** Create empty instance. */
  public AttachmentWrapper() {
  }

  /**
   * Create instance with one byte[] attachment. Adding more attachments is possible.
   * 
   * @param name
   *          attachment name
   * @param attachment
   *          attachment content
   */
  public AttachmentWrapper(final String name, final byte[] attachment) {
    add(name, attachment);
  }

  /**
   * Create instance with one String attachment. Adding more attachments is possible.
   * 
   * @param name
   *          attachment name
   * @param attachment
   *          attachment content
   */
  public AttachmentWrapper(final String name, final String attachment) {
    add(name, attachment);
  }

  /**
   * Create instance with one InputStream attachment. Adding more attachments is possible.
   * 
   * @param name
   *          attachment name
   * @param attachment
   *          attachment content
   */
  public AttachmentWrapper(final String name, final InputStream attachment) {
    add(name, attachment);
  }

  /**
   * Create instance with one File attachment. Adding more attachments is possible.
   * 
   * @param name
   *          attachment name
   * @param attachment
   *          attachment content
   */
  public AttachmentWrapper(final String name, final File attachment) {
    add(name, attachment);
  }

  @Override
  public Map<String, ContentBody> getAttachmentParts() {
    return _parts;
  }

  /**
   * Add one byte[] attachment.
   * 
   * @param name
   *          attachment name
   * @param attachment
   *          attachment content
   */
  public void add(final String name, final byte[] attachment) {
    _parts.put(name, new ByteArrayBody(attachment, MIMETYPE_ATTACHMENT, null));
  }

  /**
   * Add one String attachment.
   * 
   * @param name
   *          attachment name
   * @param attachment
   *          attachment content
   */
  public void add(final String name, final String attachment) {
    try {
      _parts.put(name, new StringBody(attachment, MIMETYPE_ATTACHMENT, CHARSET));
    } catch (final UnsupportedEncodingException ex) {
      throw new IllegalArgumentException("Charset '" + CHARSET + "' unknown. Operation impossible.");
    }
  }

  /**
   * Add one InputStream attachment.
   * 
   * @param name
   *          attachment name
   * @param attachment
   *          attachment content
   */
  public void add(final String name, final InputStream attachment) {
    _parts.put(name, new InputStreamBody(attachment, MIMETYPE_ATTACHMENT, null));
  }

  /**
   * Add one File attachment.
   * 
   * @param name
   *          attachment name
   * @param attachment
   *          attachment content
   */
  public void add(final String name, final File attachment) {
    _parts.put(name, new FileBody(attachment, MIMETYPE_ATTACHMENT));
  }
}
