/**
 * 
 */
package org.eclipse.smila.solr.admin;

import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import org.apache.commons.collections.MapUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.solr.client.solrj.SolrServer;
import org.apache.solr.client.solrj.response.SolrPingResponse;
import org.apache.solr.client.solrj.response.UpdateResponse;
import org.apache.solr.common.params.CoreAdminParams.CoreAdminAction;
import org.apache.solr.common.util.NamedList;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.datamodel.InvalidValueTypeException;
import org.eclipse.smila.datamodel.Value;
import org.eclipse.smila.solr.server.SolrServers;

/**
 * @author pwissel
 * 
 */
public abstract class AbstractSolrAdministration implements SolrAdministration {

  protected final Log _log = LogFactory.getLog(getClass());

  protected final SolrServers _servers;

  /** cache for field names defined in cores. */
  private final Map<String, List<String>> _cachedFieldNames = new HashMap<String, List<String>>();

  public AbstractSolrAdministration(final SolrServers servers) {
    _servers = servers;
  }

  /**
   * @throws SolrAdministrationException
   * @see org.eclipse.smila.solr.admin.SolrAdministration#status()
   */
  @Override
  public AnyMap status() throws SolrAdministrationException {
    final Map<String, String[]> params = new HashMap<String, String[]>();
    params.put(ACTION, convertStringToArray(CoreAdminAction.STATUS.toString()));
    return handle(params);
  }

  /**
   * @throws SolrAdministrationException
   * @see org.eclipse.smila.solr.admin.SolrAdministration#status(java.lang.String)
   */
  @Override
  public AnyMap status(String core) throws SolrAdministrationException {
    final Map<String, String[]> params = new HashMap<String, String[]>();
    params.put(ACTION, convertStringToArray(CoreAdminAction.STATUS.toString()));
    params.put(CORE, convertStringToArray(core));
    return handle(params);
  }

  /**
   * @throws SolrAdministrationException
   * @see org.eclipse.smila.solr.admin.SolrAdministration#create(java.lang.String, java.lang.String)
   */
  @Override
  public AnyMap create(String name, String instanceDir) throws SolrAdministrationException {
    return create(name, instanceDir, null);
  }

  /**
   * @throws SolrAdministrationException
   * @see org.eclipse.smila.solr.admin.SolrAdministration#create(java.lang.String, java.lang.String, java.util.Map)
   */
  @Override
  public AnyMap create(String name, String instanceDir, Map<String, String[]> otherParams)
    throws SolrAdministrationException {
    final Map<String, String[]> params = new HashMap<String, String[]>();
    params.put(ACTION, convertStringToArray(CoreAdminAction.CREATE.toString()));
    params.put(NAME, convertStringToArray(name));
    params.put(INSTANCE_DIR, convertStringToArray(instanceDir));
    if (MapUtils.isNotEmpty(otherParams)) {
      params.putAll(otherParams);
    }
    return handle(params);
  }

  /**
   * @throws SolrAdministrationException
   * @see org.eclipse.smila.solr.admin.SolrAdministration#reload(java.lang.String)
   */
  @Override
  public AnyMap reload(String core) throws SolrAdministrationException {
    final Map<String, String[]> params = new HashMap<String, String[]>();
    params.put(ACTION, convertStringToArray(CoreAdminAction.RELOAD.toString()));
    params.put(CORE, convertStringToArray(core));
    return handle(params);
  }

  /**
   * @throws SolrAdministrationException
   * @see org.eclipse.smila.solr.admin.SolrAdministration#rename(java.lang.String, java.lang.String)
   */
  @Override
  public AnyMap rename(String core, String other) throws SolrAdministrationException {
    final Map<String, String[]> params = new HashMap<String, String[]>();
    params.put(ACTION, convertStringToArray(CoreAdminAction.RENAME.toString()));
    params.put(CORE, convertStringToArray(core));
    params.put(OTHER, convertStringToArray(other));
    return handle(params);
  }

  /**
   * @throws SolrAdministrationException
   * @see org.eclipse.smila.solr.admin.SolrAdministration#swap(java.lang.String, java.lang.String)
   */
  @Override
  public AnyMap swap(String core, String other) throws SolrAdministrationException {
    final Map<String, String[]> params = new HashMap<String, String[]>();
    params.put(ACTION, convertStringToArray(CoreAdminAction.SWAP.toString()));
    params.put(CORE, convertStringToArray(core));
    params.put(OTHER, convertStringToArray(other));
    return handle(params);
  }

  /**
   * @throws SolrAdministrationException
   * @see org.eclipse.smila.solr.admin.SolrAdministration#unload(java.lang.String)
   */
  @Override
  public AnyMap unload(String core) throws SolrAdministrationException {
    return unload(core, false);
  }

  /**
   * @throws SolrAdministrationException
   * @see org.eclipse.smila.solr.admin.SolrAdministration#unload(java.lang.String, boolean)
   */
  @Override
  public AnyMap unload(String core, boolean deleteIndex) throws SolrAdministrationException {
    final Map<String, String[]> params = new HashMap<String, String[]>();
    params.put(ACTION, convertStringToArray(CoreAdminAction.UNLOAD.toString()));
    params.put(CORE, convertStringToArray(core));
    params.put(DELETE_INDEX, convertStringToArray(String.valueOf(deleteIndex)));
    return handle(params);
  }

  /**
   * @throws SolrAdministrationException
   * @see org.eclipse.smila.solr.admin.SolrAdministration#load(java.lang.String)
   */
  @Override
  public AnyMap load(String core) throws SolrAdministrationException {
    final Map<String, String[]> params = new HashMap<String, String[]>();
    params.put(ACTION, convertStringToArray(CoreAdminAction.LOAD.toString()));
    params.put(CORE, convertStringToArray(core));
    return handle(params);
  }

  /**
   * @throws SolrAdministrationException
   * @see org.eclipse.smila.solr.admin.SolrAdministration#mergeIndexes(java.lang.String, java.lang.String[])
   */
  @Override
  public AnyMap mergeIndexes(String core, String... srcCore) throws SolrAdministrationException {
    final Map<String, String[]> params = new HashMap<String, String[]>();
    params.put(ACTION, convertStringToArray(CoreAdminAction.MERGEINDEXES.toString()));
    params.put(CORE, convertStringToArray(core));
    params.put(SRC_CORE, srcCore);
    return handle(params);
  }

  public String[] convertStringToArray(String string) {
    return new String[] { string };
  }

  private AnyMap handle(final Map<String, String[]> params) throws SolrAdministrationException {
    _log.debug("Handle admin request with following parameters: " + convertMapToString(params));
    final NamedList<Object> response = handleRequest(params);
    return handleResponse(response);
  }

  abstract NamedList<Object> handleRequest(Map<String, String[]> parameter) throws SolrAdministrationException;

  private AnyMap handleResponse(NamedList<Object> response) {
    if (response == null) {
      return null;
    }
    final AnyMap map = DataFactory.DEFAULT.createAnyMap();
    return convertNamedListToAnyMap(response, map);
  }

  @SuppressWarnings("unchecked")
  AnyMap convertNamedListToAnyMap(NamedList<Object> list, AnyMap map) {
    final Iterator<Entry<String, Object>> it = list.iterator();
    while (it.hasNext()) {
      Entry<String, Object> entry = it.next();
      final String key = entry.getKey();
      final Object obj = entry.getValue();
      if (obj instanceof NamedList<?>) {
        final AnyMap subMap = map.getMap(key, true);
        convertNamedListToAnyMap((NamedList<Object>) obj, subMap);
      } else {
        try {
          final Value value = DataFactory.DEFAULT.autoConvertValue(obj);
          map.put(key, value);
        } catch (InvalidValueTypeException exception) {
          ; // skip
        }
      }
    }
    return map;
  }

  protected String convertMapToString(Map<String, String[]> map) {
    StringBuilder sb = new StringBuilder();
    for (Entry<String, String[]> entry : map.entrySet()) {
      final String key = entry.getKey();
      sb.append(key);
      sb.append("=");
      final String[] values = entry.getValue();
      int i = 0;
      for (String value : values) {
        sb.append(value);
        i++;
        if (values.length > i) {
          sb.append(",");
        }
      }
      sb.append(";");
    }
    return sb.toString();
  }

  @Override
  public List<String> getFieldNames(String coreName) throws Exception {
    List<String> fieldNames = _cachedFieldNames.get(coreName);
    if (fieldNames == null) {
      synchronized (_cachedFieldNames) {
        fieldNames = readFieldNames(coreName);
        _cachedFieldNames.put(coreName, fieldNames);
      }
    }
    return fieldNames;
  }

  /** actually read field names for given core. */
  protected abstract List<String> readFieldNames(String coreName) throws Exception;

  @Override
  public void clearCoreCaches() {
    synchronized (_cachedFieldNames) {
      _cachedFieldNames.clear();
      _servers.removeAllServers();
    }
  }

  @Override
  public void clearCoreCache(String coreName) {
    synchronized (_cachedFieldNames) {
      _cachedFieldNames.remove(coreName);
      _servers.removeSolrServer(coreName);
    }
  }

  @Override
  public AnyMap optimize(String coreName) throws SolrAdministrationException {
    try {
      final SolrServer server = _servers.getSolrServer(coreName);
      final UpdateResponse response = server.optimize();
      return handleResponse(response.getResponse());
    } catch (Exception exception) {
      throw new SolrAdministrationException("Error while optimize core with name: " + coreName, exception);
    }
  }

  @Override
  public AnyMap ping(String coreName) throws SolrAdministrationException {
    try {
      final SolrServer server = _servers.getSolrServer(coreName);
      final SolrPingResponse response = server.ping();
      return handleResponse(response.getResponse());
    } catch (Exception exception) {
      throw new SolrAdministrationException("Error while ping core with name: " + coreName, exception);
    }
  }

}
