/***********************************************************************************************************************
 * Copyright (c) 2008,2011 empolis GmbH and brox IT Solutions GmbH. All rights reserved. This program and the 
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this 
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Andreas Weber (Attensity Europe GmbH) - initial API and implementation               
 **********************************************************************************************************************/

package org.eclipse.smila.datamodel.ipc;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.util.Iterator;

import org.apache.commons.io.IOUtils;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.ipc.IpcFactory;
import org.eclipse.smila.ipc.IpcStreamWriter;
import org.eclipse.smila.ipc.bon.BinaryFactory;
import org.eclipse.smila.ipc.json.JsonFactory;

/**
 * Utility class for writing Record objects to binary (BON) and JSON.
 * 
 * @author aweber
 */
public class IpcRecordWriter {
  /** attachment names for record (de)serialization. */
  private static final String ATTACHMENT_NAMES = "_attachmentNames";

  /** BON IPC factory. */
  private final IpcFactory _binaryFactory = new BinaryFactory();

  /** Helper to write Record metadata. */
  private final IpcAnyWriter _ipcAnyWriter = new IpcAnyWriter();

  /** JSON IPC factory. */
  private final IpcFactory _jsonFactory;

  /** */
  public IpcRecordWriter() {
    this(false);
  }

  /**
   * @param printPretty
   *          whether JSON output should be formatted.
   */
  public IpcRecordWriter(final boolean printPretty) {
    _jsonFactory = new JsonFactory(printPretty);
  }

  /**
   * @param record
   *          input Record object
   * @return input Record converted to BON
   * @throws IOException
   */
  public byte[] writeBinaryObject(final Record record) throws IOException {
    ByteArrayOutputStream baos = null;
    try {
      baos = new ByteArrayOutputStream();
      writeBinaryStream(record, baos);
      return baos.toByteArray();
    } finally {
      IOUtils.closeQuietly(baos);
    }
  }

  /**
   * @param record
   *          input Record object
   * @param stream
   *          stream to write the output in BON format.
   * @throws IOException
   */
  public void writeBinaryStream(final Record record, final OutputStream stream) throws IOException {
    final IpcStreamWriter writer = _binaryFactory.newStreamWriter(stream);
    writeStreamWithAttachments(record, writer);
  }

  /**
   * @param record
   *          input Record object
   * @return input Record converted to JSON
   * @throws IOException
   */
  public String writeJsonObject(final Record record) throws IOException {
    ByteArrayOutputStream baos = null;
    try {
      baos = new ByteArrayOutputStream();
      writeJsonStream(record, baos);
      return baos.toString(IpcAnyWriter.ENCODING);
    } finally {
      IOUtils.closeQuietly(baos);
    }
  }

  /**
   * @param record
   *          input Record object
   * @param stream
   *          stream to write the output as JSON.
   * @throws IOException
   */
  public void writeJsonStream(final Record record, final OutputStream stream) throws IOException {
    final IpcStreamWriter writer = _jsonFactory.newStreamWriter(stream);
    writeStreamWithoutAttachments(record, writer);
  }

  /**
   * @param record
   *          input Record object
   * @param writer
   *          writer with an underlying stream to write the ouput to.
   * @throws IOException
   */
  public void writeStream(final Record record, final IpcStreamWriter writer) throws IOException {
    if (writer.hasBinarySupport()) {
      writeStreamWithAttachments(record, writer);
    } else {
      writeStreamWithoutAttachments(record, writer);
    }
  }

  /**
   * helper method converting/writing a record with a writer that doesn't support attachments.
   */
  private void writeStreamWithoutAttachments(final Record record, final IpcStreamWriter writer) throws IOException {
    try {
      writer.writeObjectStart();
      final AnyMap recordMetadata = record.getMetadata();
      if (record.hasAttachments()) {
        // JSON does not support attachments, so we just write the attachment names to a system attribute.
        // This can be reconverted to record attachments when re-parsing the stream (see {@IpcRecordReader})
        final Iterator<String> attNames = record.getAttachmentNames();
        while (attNames.hasNext()) {
          recordMetadata.add(ATTACHMENT_NAMES, recordMetadata.getFactory().createStringValue(attNames.next()));
        }
        _ipcAnyWriter.writeAny(writer, recordMetadata);
        recordMetadata.remove(ATTACHMENT_NAMES);
      } else {
        _ipcAnyWriter.writeAny(writer, recordMetadata);
      }
      writer.writeObjectEnd();
    } finally {
      writer.closeWithoutStream(); // do not close underlying stream
    }
  }

  /**
   * helper method converting/writing a record with a writer that supports attachments.
   */
  private void writeStreamWithAttachments(final Record record, final IpcStreamWriter writer) throws IOException {
    writer.writeObjectStart();
    final AnyMap recordMetadata = record.getMetadata();
    _ipcAnyWriter.writeAny(writer, recordMetadata);
    if (record.hasAttachments()) {
      writeAttachments(record, writer);
    }
    writer.writeObjectEnd();
  }

  /**
   * helper method converting/writing attachments.
   */
  private void writeAttachments(final Record record, final IpcStreamWriter writer) throws IOException {
    writer.writeAttachmentsStart();
    final Iterator<String> attachmentNames = record.getAttachmentNames();
    while (attachmentNames.hasNext()) {
      final String attachmentName = attachmentNames.next();
      writer.writeScalarString(attachmentName);
      if (record.getAttachmentAsBytes(attachmentName) != null) {
        writer.writeBinary(record.getAttachmentAsBytes(attachmentName));
      } else {
        throw new IllegalArgumentException("Can't stream record attachment with <null> value for record with id "
          + record.getId());
      }
    }
    writer.writeAttachmentsEnd();
  }

}
