/***********************************************************************************************************************
 * Copyright (c) 2008 empolis GmbH and brox IT Solutions GmbH. All rights reserved. This program and the accompanying
 * materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this distribution,
 * and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Ivan Churkin (brox IT Solutions GmbH) - initial creator Sebastian Voigt (brox IT Solutions GmbH)
 **********************************************************************************************************************/
package org.eclipse.smila.utils.digest;

import java.io.UnsupportedEncodingException;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;

/**
 * Calculates SHA-256 digests of strings or byte arrays.
 */
public final class DigestHelper {

  /** convert byte to unsigned int for hex conversion. */
  private static final int BYTE_TO_INT_MASK = 0xff;

  /** shift-right count to use for converting upper half of byte to hex. */
  private static final int HIGH_NIBBLE_SHIFT = 4;

  /** bit mask to use for converting lower half of byte to hex. */
  private static final int LOW_NIBBLE_MASK = 0xf;

  /** hex digits. */
  private static final char[] HEXDIGITS = { '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', 'a', 'b', 'c', 'd',
    'e', 'f' };

  /**
   * Instantiates a new digest helper.
   */
  private DigestHelper() {
  }

  /** Calculate digest. */
  public static String calculateDigest(final String value) {
    if (value == null) {
      return null;
    }
    final byte[] bytes;
    try {
      bytes = value.getBytes("utf-8");
    } catch (final UnsupportedEncodingException e) {
      throw new RuntimeException("Cannot happen, utf-8 is always known!");
    }
    return calculateDigest(bytes);
  }

  /** Calculate digest. */
  public static String calculateDigest(final byte[] bytes) {
    MessageDigest digest = null;
    try {
      digest = MessageDigest.getInstance("SHA-256");
    } catch (final NoSuchAlgorithmException e1) {
      throw new RuntimeException("Cannot happen, SHA-256 MUST be available!");
    }
    final byte[] hash = digest.digest(bytes);
    return toHexString(hash);
  }

  /** convert bytes to hex string. */
  public static String toHexString(final byte[] hash) {
    final StringBuilder hexHash = new StringBuilder();
    for (int i = 0; i < hash.length; i++) {
      final int hashByte = hash[i] & BYTE_TO_INT_MASK;
      hexHash.append(HEXDIGITS[hashByte >> HIGH_NIBBLE_SHIFT]);
      hexHash.append(HEXDIGITS[hashByte & LOW_NIBBLE_MASK]);
    }
    return hexHash.toString();
  }
}
