/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Andreas Schank (Attensity Europe GmbH) - initial implementation
 **********************************************************************************************************************/
package org.eclipse.smila.jobmanager.persistence.zk;

import java.io.UnsupportedEncodingException;
import java.net.URLDecoder;
import java.net.URLEncoder;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Random;
import java.util.Set;
import java.util.TreeSet;
import java.util.UUID;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.zookeeper.CreateMode;
import org.apache.zookeeper.KeeperException;
import org.apache.zookeeper.KeeperException.NoNodeException;
import org.apache.zookeeper.KeeperException.NodeExistsException;
import org.apache.zookeeper.data.Stat;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.AnySeq;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.datamodel.Value;
import org.eclipse.smila.datamodel.ValueFormatHelper;
import org.eclipse.smila.datamodel.ipc.IpcAnyReader;
import org.eclipse.smila.datamodel.ipc.IpcAnyWriter;
import org.eclipse.smila.jobmanager.JobRunInfo;
import org.eclipse.smila.jobmanager.JobState;
import org.eclipse.smila.jobmanager.definitions.BucketDefinition;
import org.eclipse.smila.jobmanager.definitions.JobManagerConstants;
import org.eclipse.smila.jobmanager.definitions.JobRunDefinitions;
import org.eclipse.smila.jobmanager.definitions.JobRunMode;
import org.eclipse.smila.jobmanager.persistence.JobRunListener;
import org.eclipse.smila.jobmanager.persistence.RunStorage;
import org.eclipse.smila.jobmanager.persistence.RunStorageException;
import org.eclipse.smila.taskmanager.Task;
import org.eclipse.smila.zookeeper.ZkConcurrentMap;
import org.eclipse.smila.zookeeper.ZkConnection;
import org.eclipse.smila.zookeeper.ZooKeeperService;
import org.osgi.service.component.ComponentContext;

/**
 * Component for handling jobmanager run data by using zookeeper.
 * 
 * Zookeeper structure:
 * 
 * <pre>
 * /smila/jobmanager/jobs/&lt;job-name>/workflow-runs/&lt;workflow-run-id>/data/tasks/&lt;task-id>
 * ............................................................................/transient-bulks/&lt;bulkstore+objectId>
 * ......................................................................./finishing
 * 
 * ................................./data/&lt;data-node>              // job counters etc.
 * ................................./worker-data/&lt;workername>      // worker specific counters etc.
 * ................................./jobrun-definitions/              // definitions used in a job run
 * ..................................................../jobdef    
 * ................................................... /wfdef     
 * ..................................................../bucketdef/&lt;bucket-name>    
 * ....................../buckets/&lt;bucket-id>/&lt;job-name>        // jobs triggered by bucket
 * </pre>
 */
public class RunStorageZk implements RunStorage {

  /** prefix for jobmanager data. */
  public static final String JOBMANAGER_PREFIX = "/smila/jobmanager";

  /** workflow runs of a job run. */
  public static final String NODE_WORKFLOW_RUNS = "workflow-runs";

  /** data of a job run. */
  public static final String NODE_BUCKETS = "buckets";

  /** data of a job run. */
  public static final String NODE_JOBS = "jobs";

  /** global data of a job run. */
  public static final String NODE_DATA = "data";

  /** worker specific data of a job run. */
  public static final String NODE_WORKERDATA = "worker-data";

  /** tasks of a workflow step. */
  public static final String NODE_TASKS = "tasks";

  /** transient bulks of a workflow run. */
  public static final String NODE_TRANSIENT_BULKS = "transient-bulks";

  /** root node for stored definitions that are used by a job run. */
  public static final String NODE_RUN_DEFINITIONS = "jobrun-definitions";

  /** stores job definition used by a job run. */
  public static final String NODE_RUN_DEFINITIONS_JOB = "jobdef";

  /** stores workflow definition used by a job run. */
  public static final String NODE_RUN_DEFINITIONS_WORKFLOW = "wfdef";

  /** stores bucket definitions used by a job run. */
  public static final String NODE_RUN_DEFINITIONS_BUCKET = "bucketdef";

  /** indicates whether a workflow run is currently finished. */
  public static final String NODE_WORKFLOW_RUN_FINISHING = "finishing";

  /** "0" in UTF-8 encoding. */
  private static final byte ZERO_AS_UTF_8 = 48;

  /** UTF-8 encoding. */
  private static final String UTF_8 = "UTF-8";

  /** marker for finished workflow runs. */
  private static final String WORKFLOW_RUN_FINISH_PREFIX = "workflow-run-finish-";

  /** simple date format used for creating readable job id and job monitoring. */
  private static final SimpleDateFormat SIMPLE_DATE_FORMAT = ValueFormatHelper.getDefaultDateTimeFormat();

  /** how often should we repeat the zookeeper operation if we are conflicting with changes of others. */
  private static final int NO_OF_TRIES_WHEN_CONFLICT = 100;

  /** maximal time waiting to repeat the zk operation if we are conflicting with changes of others (in millis). */
  private static final int MAX_WAIT_TIME_BETWEEN_TRIES = 10;

  /** we use a random wait time to repeat the zk operation if we are conflicting with changes of others. */
  private final Random _waitTimeBetweenTriesGenerator = new Random(System.nanoTime());

  /** local logger. */
  private final Log _log = LogFactory.getLog(getClass());

  /** zookeeper service. */
  private ZooKeeperService _zkService;

  /** Connection to ZooKeeper server. */
  private ZkConnection _zk;

  /** Any reader. */
  private final IpcAnyReader _anyReader = new IpcAnyReader();

  /** Any serialization. */
  private final IpcAnyWriter _anyWriter = new IpcAnyWriter(false);

  /**
   * @return root path for all stored jobs
   */
  private String getJobsPath() {
    return JOBMANAGER_PREFIX + '/' + NODE_JOBS;
  }

  /**
   * @return root path for given job name
   */
  private String getJobPath(final String jobName) {
    return getJobsPath() + '/' + jobName;
  }

  /**
   * @return root path where to store the definitions for the job (run).
   */
  private String getJobRunDefinitionsPath(final String jobName) {
    return getJobPath(jobName) + '/' + NODE_RUN_DEFINITIONS;
  }

  /**
   * @return path where the job definition for the given job is stored.
   */
  private String getJobRunDefPathForJobDef(final String jobName) {
    return getJobRunDefinitionsPath(jobName) + '/' + NODE_RUN_DEFINITIONS_JOB;
  }

  /**
   * @return path where the workflow definition for the given job is stored.
   */
  private String getJobRunDefPathForWorkflowDef(final String jobName) {
    return getJobRunDefinitionsPath(jobName) + '/' + NODE_RUN_DEFINITIONS_WORKFLOW;
  }

  /**
   * @return path where the bucket definitions for the given job are stored.
   */
  private String getJobRunDefPathForBucketDefs(final String jobName) {
    return getJobRunDefinitionsPath(jobName) + '/' + NODE_RUN_DEFINITIONS_BUCKET;
  }

  /**
   * @param jobName
   *          job name
   * @return path for job run data for given job.
   */
  private String getJobDataPath(final String jobName) {
    return getJobPath(jobName) + '/' + NODE_DATA;
  }

  /**
   * @param jobName
   *          job name
   * @return root path for worker specific job run data.
   */
  private String getJobWorkerDataPath(final String jobName) {
    return getJobPath(jobName) + '/' + NODE_WORKERDATA;
  }

  /**
   * @param jobName
   *          job name
   * @param workerName
   *          worker name
   * @return path for worker specific job run data.
   */

  private String getJobWorkerDataPath(final String jobName, final String workerName) {
    return getJobWorkerDataPath(jobName) + '/' + workerName;
  }

  /**
   * @param jobName
   *          job name
   * @return root path for workflow runs for given job name.
   */
  private String getWorkflowRunsPath(final String jobName) {
    return getJobPath(jobName) + '/' + NODE_WORKFLOW_RUNS;
  }

  /**
   * @param jobName
   *          job name
   * @param workflowRunId
   *          workflow run id
   * @return path for given job and workflow run id
   */
  private String getWorkflowRunPath(final String jobName, final String workflowRunId) {
    return getWorkflowRunsPath(jobName) + '/' + workflowRunId;
  }

  /**
   * @return path where all data for given workflow run is stored.
   */
  private String getWorkflowRunDataPath(final String jobName, final String workflowRunId) {
    return getWorkflowRunPath(jobName, workflowRunId) + "/data";
  }

  /**
   * @return path where task ids for given workflow run are stored.
   */
  private String getWorkflowRunTasksPath(final String jobName, final String workflowRunId) {
    return getWorkflowRunDataPath(jobName, workflowRunId) + '/' + NODE_TASKS;
  }

  /**
   * @return path where transient bulk ids for given workflow run are stored.
   */
  private String getWorkflowRunBulksPath(final String jobName, final String workflowRunId) {
    return getWorkflowRunDataPath(jobName, workflowRunId) + '/' + NODE_TRANSIENT_BULKS;
  }

  /**
   * @param bucketId
   *          bucket Id
   * @return path for node containing job names triggered by this bucket.
   */
  private String getTriggerBucketPath(final String bucketId) {
    return JOBMANAGER_PREFIX + '/' + NODE_BUCKETS + '/' + encode(bucketId);
  }

  /**
   * @param bucketId
   *          bucket Id
   * @param jobName
   *          job name
   * @return path for node describing that a node is triggered by a bucket
   */
  private String getTriggeredJobPath(final String bucketId, final String jobName) {
    return getTriggerBucketPath(bucketId) + '/' + jobName;
  }

  /**
   * @param jobName
   *          the job for which to return the data map
   * @return map view of underlying zookeeper nodes for given job data if the data node currently exists. Else null.
   * @throws Exception
   *           error
   */
  private ZkConcurrentMap getJobDataMap(final String jobName) throws Exception {
    final String jobDataPath = getJobDataPath(jobName);
    try {
      return new ZkConcurrentMap(_zk, jobDataPath);
    } catch (final IllegalArgumentException e) {
      ; // ignore. job may not exist yet/anymore.
    }
    return null;
  }

  /**
   * @param jobName
   *          the job for which to return the data map
   * @return map view of underlying zookeeper nodes for given job data.
   * @throws Exception
   *           error
   */
  private ZkConcurrentMap ensureJobDataMap(final String jobName) throws Exception {
    final String jobDataPath = getJobDataPath(jobName);
    _zk.ensurePathExists(jobDataPath);
    return new ZkConcurrentMap(_zk, jobDataPath);
  }

  /**
   * Checks if the dataMap's job id is equal to jobRunId. Throws an IllegalArgumentException if that is not the case.
   * 
   * @param dataMap
   *          data map of the current job run
   * @param jobRunId
   *          requested job run Id
   */
  private void checkCurrentJobRunId(final ZkConcurrentMap dataMap, final String jobRunId) {
    final String currentId = dataMap.getString(JobManagerConstants.DATA_JOB_ID);
    if (!jobRunId.equals(currentId)) {
      throw new IllegalArgumentException("Current job run is not '" + jobRunId + "' but '" + currentId + "'.");
    }
  }

  /** {@inheritDoc} */
  @Override
  public synchronized String getCurrentTimestamp() {
    return SIMPLE_DATE_FORMAT.format(new Date(System.currentTimeMillis()));
  }

  /**
   * Adds the given value to the given data counter. If that fails, throws no exception, but just logs as warning.
   */
  private Integer addToDataCounter(final ZkConcurrentMap dataMap, final String counter, final int valueToAdd) {
    Integer newValue = null;
    try {
      newValue = dataMap.add(counter, valueToAdd);
      if (newValue == null) {
        _log.warn("Could not update data counter '" + counter + "' with value '" + valueToAdd
          + "', probably due to temporary overload.");
      }
    } catch (final Exception e) {
      _log.warn("Exception while updating data counter '" + counter + "' with value '" + valueToAdd + "': "
        + e.getMessage());
    }
    return newValue;
  }

  /**
   * OSGi Declarative Services service activation method.
   * 
   * @param context
   *          OSGi service component context.
   */
  protected void activate(final ComponentContext context) {
    if (_log.isDebugEnabled()) {
      _log.debug("activate");
    }
    _zk = new ZkConnection(_zkService);
  }

  /**
   * OSGi Declarative Services service deactivation method.
   * 
   * @param context
   *          OSGi service component context.
   */
  protected void deactivate(final ComponentContext context) {
    if (_log.isDebugEnabled()) {
      _log.debug("deactivate");
    }
  }

  /**
   * method for DS to set a service reference.
   * 
   * @param zkService
   *          ZooKeeperService reference.
   */
  public void setZooKeeperService(final ZooKeeperService zkService) {
    _zkService = zkService;
  }

  /**
   * method for DS to unset a service reference.
   * 
   * @param zkService
   *          ZooKeeperService reference.
   */
  public void unsetZooKeeperService(final ZooKeeperService zkService) {
    if (_zkService == zkService) {
      _zkService = null;
    }
  }

  /** {@inheritDoc} */
  @Override
  public void startJobRun(final String jobName, final String jobRunId, final JobRunMode jobRunMode,
    final JobRunDefinitions jobRunDefs) throws RunStorageException {
    try {
      final ZkConcurrentMap dataMap = ensureJobDataMap(jobName);
      final String workflowRunsPath = getWorkflowRunsPath(jobName);
      _zk.ensurePathExists(workflowRunsPath);
      final String currentJobRunId = dataMap.putIfAbsent(JobManagerConstants.DATA_JOB_ID, jobRunId);
      if (!jobRunId.equals(currentJobRunId)) {
        throw new IllegalStateException("Another job run with id '" + currentJobRunId + "' already exists.");
      }
      dataMap.put(JobManagerConstants.DATA_JOB_STATE, JobState.PREPARING.name());

      dataMap.put(JobManagerConstants.DATA_JOB_NO_OF_ACTIVE_WORKFLOW_RUNS, "0");
      dataMap.put(JobManagerConstants.DATA_JOB_NO_OF_STARTED_WORKFLOW_RUNS, "0");
      dataMap.put(JobManagerConstants.DATA_JOB_NO_OF_SUCCESSFUL_WORKFLOW_RUNS, "0");
      dataMap.put(JobManagerConstants.DATA_JOB_NO_OF_FAILED_WORKFLOW_RUNS, "0");
      dataMap.put(JobManagerConstants.DATA_JOB_NO_OF_CANCELED_WORKFLOW_RUNS, "0");

      dataMap.put(JobManagerConstants.DATA_JOB_NO_OF_CREATED_TASKS, "0");
      dataMap.put(JobManagerConstants.DATA_JOB_NO_OF_FAILED_TASKS_NOT_RETRIED, "0");
      dataMap.put(JobManagerConstants.DATA_JOB_NO_OF_FAILED_TASKS_RETRIED, "0");
      dataMap.put(JobManagerConstants.DATA_JOB_NO_OF_RETRIED_TASKS_TTL, "0");
      dataMap.put(JobManagerConstants.DATA_JOB_NO_OF_RETRIED_TASKS_WORKER, "0");
      dataMap.put(JobManagerConstants.DATA_JOB_NO_OF_SUCCESSFUL_TASKS, "0");
      dataMap.put(JobManagerConstants.DATA_JOB_NO_OF_CANCELLED_TASKS, "0");
      dataMap.put(JobManagerConstants.DATA_JOB_NO_OF_OBSOLETE_TASKS, "0");

      dataMap.put(JobManagerConstants.DATA_JOB_RUN_START_TIME, getCurrentTimestamp());
      dataMap.put(JobManagerConstants.DATA_JOB_RUN_MODE, jobRunMode.name());

      storeJobRunDefinitions(jobName, jobRunDefs);

    } catch (final Exception e) {
      throw newRunStorageException("Error while preparing job run data for job '" + jobName + "' with run id '"
        + jobRunId + "'.", e);
    }
  }

  /**
   * @param jobName
   *          job for which to store the definitions
   * @param jobRunDefs
   *          contains job/workflow/bucket definitions that are stored for the given job
   */
  private void storeJobRunDefinitions(final String jobName, final JobRunDefinitions jobRunDefs) throws Exception {
    _zk.ensurePathExists(getJobRunDefPathForBucketDefs(jobName));
    // we only want the relevant information, not the additional stuff here, so call toAny(false) where applicable
    final byte[] jobDefData = _anyWriter.writeBinaryObject(jobRunDefs.getJobDefinition().toAny(false));
    _zk.createNode(getJobRunDefPathForJobDef(jobName), jobDefData);
    final byte[] workflowfDefData = _anyWriter.writeBinaryObject(jobRunDefs.getWorkflowDefinition().toAny(false));
    _zk.createNode(getJobRunDefPathForWorkflowDef(jobName), workflowfDefData);
    for (final BucketDefinition bucketDef : jobRunDefs.getBucketDefinitions()) {
      final byte[] bucketDefData = _anyWriter.writeBinaryObject(bucketDef.toAny());
      _zk.createNode(getJobRunDefPathForBucketDefs(jobName) + '/' + bucketDef.getName(), bucketDefData);
    }
  }

  /** {@inheritDoc} */
  @Override
  public boolean finishJobRun(final String jobName, final String jobRunId) throws RunStorageException {
    try {
      final ZkConcurrentMap dataMap = ensureJobDataMap(jobName);
      checkCurrentJobRunId(dataMap, jobRunId);
      if (!dataMap.replace(JobManagerConstants.DATA_JOB_STATE, JobState.RUNNING.name(), JobState.FINISHING.name())) {
        // we were not the first one... return false, so the caller can react accordingly.
        if (_log.isInfoEnabled()) {
          _log.info("Could not set state to " + JobState.FINISHING + " for job run '" + jobRunId + "' for job '"
            + jobName + "', someone else may already finishing the run");
        }
        return false;
      }
      dataMap.put(JobManagerConstants.DATA_JOB_RUN_FINISH_TIME, getCurrentTimestamp());
    } catch (final Exception e) {
      throw newRunStorageException("Error while setting finish job run data for job '" + jobName
        + "' with run id '" + jobRunId + "'.", e);
    }
    return true;
  }

  /** {@inheritDoc} */
  @Override
  public List<String> cancelJobRun(final String jobName, final String jobRunId) throws RunStorageException {
    try {
      final ZkConcurrentMap dataMap = ensureJobDataMap(jobName);
      checkCurrentJobRunId(dataMap, jobRunId);
      // cancel all workflow runs
      final String workflowRunsRoot = getWorkflowRunsPath(jobName);
      final List<String> workflowRunIds = _zk.getChildrenSorted(workflowRunsRoot);
      for (final String workflowRunId : workflowRunIds) {
        cancelWorkflowRun(jobName, jobRunId, workflowRunId, dataMap);
      }
      return workflowRunIds;
    } catch (final Exception e) {
      throw newRunStorageException("Error while canceling job run data for job '" + jobName + "' with run id '"
        + jobRunId + "'.", e);
    }
  }

  /**
   * Cancels a workflow run, currently processed tasks are canceled too.
   */
  private void cancelWorkflowRun(final String jobName, final String jobRunId, final String workflowRunId,
    final ZkConcurrentMap dataMap) throws RunStorageException {
    try {
      // is the workflow-run still there?
      if (_zk.exists(getWorkflowRunPath(jobName, workflowRunId)) != null) {
        if (_log.isInfoEnabled()) {
          _log.info("Workflow run '" + workflowRunId + "' canceled, preparing finish");
        }
        final boolean doIt = prepareToFinishWorkflowRun(jobName, jobRunId, workflowRunId);
        if (doIt) {
          if (_log.isInfoEnabled()) {
            _log.info("Workflow run '" + workflowRunId
              + "' canceled, aggregating counters and deleting workflow run");
          }
          addToDataCounter(dataMap, JobManagerConstants.DATA_JOB_NO_OF_CANCELED_WORKFLOW_RUNS, 1);
          addToDataCounter(dataMap, JobManagerConstants.DATA_JOB_NO_OF_ACTIVE_WORKFLOW_RUNS, -1);
          deleteWorkflowRunData(jobName, jobRunId, workflowRunId); // delete the data first
          deleteWorkflowRun(jobName, jobRunId, workflowRunId);
        }
        if (_log.isInfoEnabled()) {
          _log.info("Workflow run '" + workflowRunId + "' canceled");
        }
      }
    } catch (final Exception e) {
      _log.warn("Error while canceling workflow run with id '" + workflowRunId + "' for job '" + jobName
        + "' with run id '" + jobRunId + "'.", e);
    }
  }

  /** {@inheritDoc} */
  @Override
  public void deleteJobRun(final String jobName, final String jobRunId) throws RunStorageException {
    try {
      // delete in specific order: job run data (id/state) should be deliverable as long as possible
      _zk.deleteTree(getWorkflowRunsPath(jobName));
      _zk.deleteTree(getJobWorkerDataPath(jobName));
      _zk.deleteTree(getJobRunDefinitionsPath(jobName));
      _zk.deleteTree(getJobPath(jobName));
    } catch (final Exception e) {
      throw newRunStorageException("Error while deleting job run with id '" + jobRunId + "' for job '" + jobName
        + "'.", e);
    }
  }

  /** {@inheritDoc} */
  @Override
  public Collection<String> getCurrentJobs() throws RunStorageException {
    try {
      final String path = getJobsPath();
      _zk.ensurePathExists(path);
      return _zk.getChildrenSorted(path);
    } catch (final Exception e) {
      throw newRunStorageException("Error while getting job names.", e);
    }
  }

  /** {@inheritDoc} */
  @Override
  public String getJobRunId(final String jobName) throws RunStorageException {
    try {
      final ZkConcurrentMap dataMap = getJobDataMap(jobName);
      if (dataMap != null) {
        return dataMap.getString(JobManagerConstants.DATA_JOB_ID);
      } else {
        return null;
      }
    } catch (final Exception e) {
      throw newRunStorageException("Error while getting job run id for job '" + jobName + "'.", e);
    }
  }

  /** {@inheritDoc} */
  @Override
  public AnyMap getJobRunData(final String jobName, final boolean returnDetails) throws RunStorageException {
    try {
      final AnyMap result = DataFactory.DEFAULT.createAnyMap();
      final ZkConcurrentMap dataMap = getJobDataMap(jobName);
      if (dataMap != null) {
        addJobRunDataBasicSections(result, dataMap, jobName);
        addJobRunDataWorkerSection(result, jobName);
        addJobRunDataDefinitionSection(result, jobName, returnDetails);
      }
      return result;
    } catch (final Exception e) {
      throw newRunStorageException("Error while getting job run data for job '" + jobName + "'.", e);
    }
  }

  /**
   * add basic job run data from zk data map and sections for workflow run and task counters.
   */
  private void addJobRunDataBasicSections(final AnyMap result, final ZkConcurrentMap dataMap, final String jobName)
    throws Exception {
    final AnyMap workflowRuns = DataFactory.DEFAULT.createAnyMap();
    final AnyMap tasks = DataFactory.DEFAULT.createAnyMap();
    final Set<String> keySet = dataMap.keySet();
    for (final String key : keySet) {
      final String value = dataMap.getString(key);
      if (value != null) {
        if (key.toLowerCase(Locale.ENGLISH).endsWith("taskcount")) {
          try {
            tasks.put(key, Integer.valueOf(value));
          } catch (final NumberFormatException ex) {
            tasks.put(key, value);
          }
        } else if (key.toLowerCase(Locale.ENGLISH).endsWith("workflowruncount")) {
          try {
            workflowRuns.put(key, Integer.valueOf(value));
          } catch (final NumberFormatException ex) {
            workflowRuns.put(key, value);
          }
        } else {
          result.put(key, value);
        }
      }
    }
    result.put(JobManagerConstants.WORKFLOW_RUN_COUNTER, workflowRuns);
    result.put(JobManagerConstants.TASK_COUNTER, tasks);
  }

  /**
   * add job run data section for worker data.
   */
  private void addJobRunDataWorkerSection(final AnyMap result, final String jobName) throws Exception {
    final String workerDataNode = getJobWorkerDataPath(jobName);
    if (_zk.exists(workerDataNode) != null) {
      final AnyMap workerData = DataFactory.DEFAULT.createAnyMap();
      final List<String> workers = _zk.getChildrenSorted(workerDataNode);
      if (workers != null) {
        for (final String worker : workers) {
          final AnyMap workerAny = getWorkerData(jobName, worker);
          workerData.put(worker, workerAny);
        }
      }
      result.put("worker", workerData);
    }
  }

  /**
   * add job run data section for job/workflow/bucket definitions.
   */
  private void addJobRunDataDefinitionSection(final AnyMap result, final String jobName, final boolean returnDetails)
    throws Exception {
    final String runDefsNode = getJobRunDefinitionsPath(jobName);
    if (_zk.exists(runDefsNode) != null) {
      try {
        final byte[] jobDefBytes = _zk.getData(getJobRunDefPathForJobDef(jobName));
        final AnyMap jobDefAny = (AnyMap) _anyReader.readBinaryObject(jobDefBytes);
        result.put(JobManagerConstants.DATA_JOB_RUN_JOB_DEF, jobDefAny);
        if (returnDetails) {
          final byte[] workflowDefBytes = _zk.getData(getJobRunDefPathForWorkflowDef(jobName));
          final AnyMap workflowDefAny = (AnyMap) _anyReader.readBinaryObject(workflowDefBytes);
          result.put(JobManagerConstants.DATA_JOB_RUN_WORKFLOW_DEF, workflowDefAny);
          final Collection<String> bucketNodes = _zk.getChildrenSorted(getJobRunDefPathForBucketDefs(jobName));
          if (!bucketNodes.isEmpty()) {
            final AnySeq bucketDefs = DataFactory.DEFAULT.createAnySeq();
            result.put(JobManagerConstants.DATA_JOB_RUN_BUCKET_DEFS, bucketDefs);
            for (final String bucketNode : bucketNodes) {
              final byte[] bucketDefBytes = _zk.getData(getJobRunDefPathForBucketDefs(jobName) + '/' + bucketNode);
              final AnyMap bucketDefAny = (AnyMap) _anyReader.readBinaryObject(bucketDefBytes);
              bucketDefs.add(bucketDefAny);
            }
          }
        }
      } catch (final Exception ex) {
        _log.warn("Failed to read job run definition data for job '" + jobName + "'", ex);
      }
    }
  }

  /** {@inheritDoc} */
  @Override
  public AnyMap getWorkflowRunData(final String jobName, final String workflowRunId) throws RunStorageException {
    try {
      final String workflowRunPath = getWorkflowRunPath(jobName, workflowRunId);
      if (_zk.exists(workflowRunPath) == null) {
        throw new RunStorageException("Workflow run '" + workflowRunId + "' does not exist for job '" + jobName
          + "'", false);
      }
      final AnyMap workflowRunData = DataFactory.DEFAULT.createAnyMap();
      final String tasksPath = getWorkflowRunTasksPath(jobName, workflowRunId);
      final Stat tasksStat = _zk.exists(tasksPath);
      if (tasksStat != null) {
        workflowRunData.put(JobManagerConstants.DATA_WORKFLOW_RUN_NO_OF_ACTIVE_TASKS, tasksStat.getNumChildren());
      } else {
        workflowRunData.put(JobManagerConstants.DATA_WORKFLOW_RUN_NO_OF_ACTIVE_TASKS, 0);
      }
      final String bulksPath = getWorkflowRunBulksPath(jobName, workflowRunId);
      final Stat bulksStat = _zk.exists(bulksPath);
      if (bulksStat != null) {
        workflowRunData
          .put(JobManagerConstants.DATA_WORKFLOW_RUN_NO_OF_TRANSIENT_BULKS, bulksStat.getNumChildren());
      } else {
        workflowRunData.put(JobManagerConstants.DATA_WORKFLOW_RUN_NO_OF_TRANSIENT_BULKS, 0);
      }
      return workflowRunData;
    } catch (final Exception e) {
      throw newRunStorageException("Error while getting workflow run data for run '" + workflowRunId + "' in job '"
        + jobName + "'.", e);
    }
  }

  /**
   * @return worker data for given worker in given job. return empty AnyMap if error while reading.
   */
  public AnyMap getWorkerData(final String jobName, final String worker) {
    try {
      final String workerPath = getJobWorkerDataPath(jobName, worker);
      final byte[] workerData = _zk.getData(workerPath);
      if (workerData != null && workerData.length > 0) {
        final AnyMap workerMap = (AnyMap) _anyReader.readBinaryObject(workerData);
        final AnyMap sortedWorkerMap = workerMap.getFactory().createAnyMap();
        if (workerMap.containsKey("warnCount")) {
          sortedWorkerMap.put("warnCount", workerMap.remove("warnCount"));
        }
        final Collection<String> sortedKeys = new TreeSet<String>(workerMap.keySet());
        for (final String key : sortedKeys) {
          sortedWorkerMap.put(key, workerMap.get(key));
        }
        return sortedWorkerMap;
      }
    } catch (final Exception ex) {
      _log.warn("Failed to read worker data for '" + worker + "' in job '" + jobName, ex);
    }
    return DataFactory.DEFAULT.createAnyMap();
  }

  /** {@inheritDoc} */
  @Override
  public JobState getJobState(final String jobName) throws RunStorageException {
    try {
      final ZkConcurrentMap dataMap = getJobDataMap(jobName);
      if (dataMap != null) {
        final String s = dataMap.getString(JobManagerConstants.DATA_JOB_STATE);
        if (s != null) {
          return JobState.valueOf(s);
        }
      }
      return null;
    } catch (final Exception e) {
      throw newRunStorageException("Error while getting job state for job '" + jobName + "'.", e);
    }
  }

  /** {@inheritDoc} */
  @Override
  public JobState getJobState(final String jobName, final String jobRunId) throws RunStorageException {
    try {
      final ZkConcurrentMap dataMap = getJobDataMap(jobName);
      if (dataMap != null) {
        final String runId = dataMap.getString(JobManagerConstants.DATA_JOB_ID);
        if (jobRunId.equals(runId)) {
          final String state = dataMap.getString(JobManagerConstants.DATA_JOB_STATE);
          if (state != null) {
            return JobState.valueOf(state);
          }
        }
      }
      return null;
    } catch (final Exception e) {
      throw newRunStorageException("Error while getting job state for job '" + jobName + "'.", e);
    }
  }

  /** {@inheritDoc} */
  @Override
  public JobRunInfo getJobRunInfo(final String jobName) throws RunStorageException {
    try {
      final ZkConcurrentMap dataMap = getJobDataMap(jobName);
      if (dataMap != null) {
        final String jobRunId = dataMap.getString(JobManagerConstants.DATA_JOB_ID);
        if (jobRunId != null) {
          final String state = dataMap.getString(JobManagerConstants.DATA_JOB_STATE);
          if (state != null) {
            return new JobRunInfo(jobRunId, JobState.valueOf(state));
          }
        }
      }
      return null;
    } catch (final Exception e) {
      throw newRunStorageException("Error while getting job state for job '" + jobName + "'.", e);
    }
  }

  /** {@inheritDoc} */
  @Override
  public void setJobState(final String jobName, final String jobRunId, final JobState jobState)
    throws RunStorageException {
    if (_log.isInfoEnabled()) {
      _log.info("Setting job state for job run '" + jobRunId + "' for job '" + jobName + "' to state " + jobState);
    }
    try {
      final ZkConcurrentMap dataMap = ensureJobDataMap(jobName);
      checkCurrentJobRunId(dataMap, jobRunId);
      dataMap.put(JobManagerConstants.DATA_JOB_STATE, jobState.name());
    } catch (final Exception e) {
      throw newRunStorageException("Error while setting job state '" + jobState + "' for job '" + jobName + "'.", e);
    }
  }

  /** {@inheritDoc} */
  @Override
  public boolean setJobState(final String jobName, final String jobRunId, final JobState expectedState,
    final JobState newState) throws RunStorageException {
    try {
      final ZkConcurrentMap dataMap = ensureJobDataMap(jobName);
      checkCurrentJobRunId(dataMap, jobRunId);
      final boolean result =
        dataMap.replace(JobManagerConstants.DATA_JOB_STATE, expectedState.name(), newState.name());
      if (_log.isInfoEnabled()) {
        _log.info("Changing job state for job run '" + jobRunId + "' for job '" + jobName + "' to state "
          + newState + " while expecting state " + expectedState + " returned result: " + result);
      }
      return result;
    } catch (final Exception e) {
      throw newRunStorageException("Error while setting new job state '" + newState + "' for expected job state '"
        + expectedState + "' of  job '" + jobName + "'.", e);
    }
  }

  /** {@inheritDoc} */
  @Override
  public String startWorkflowRun(final String jobName, final String jobRunId) throws RunStorageException {
    return startWorkflowRun(jobName, jobRunId, JobManagerConstants.DATA_JOB_NO_OF_STARTED_WORKFLOW_RUNS);
  }

  /** {@inheritDoc} */
  @Override
  public String startCompletionWorkflowRun(final String jobName, final String jobRunId) throws RunStorageException {
    return startWorkflowRun(jobName, jobRunId, JobManagerConstants.DATA_JOB_NO_OF_STARTED_COMPLETION_WORKFLOW_RUNS);
  }

  /**
   * starts the workflow run and adds the number of active workflow runs and the start counter.
   * 
   * @param jobName
   *          the job's name
   * @param jobRunId
   *          the job run id
   * @param workflowStartCounter
   *          the counter of started workflow runs to be incremented
   * @return the new workflow run's id
   * @throws RunStorageException
   *           couldn't access or store data
   */
  private String startWorkflowRun(final String jobName, final String jobRunId, final String workflowStartCounter)
    throws RunStorageException {
    String newWorkflowRunId = null;
    try {
      // inc number of workflow runs in job data
      final ZkConcurrentMap dataMap = ensureJobDataMap(jobName);
      final Integer newValue = addToDataCounter(dataMap, workflowStartCounter, 1);
      if (newValue != null) {
        newWorkflowRunId = String.valueOf(newValue);
      } else {
        throw new RunStorageException("Could not create workflow run id for job '" + jobName
          + "', probably due to temporary overload.", true);
      }
      // create zookeeper nodes/path for new workflow run
      final String workflowRunPath = getWorkflowRunPath(jobName, newWorkflowRunId);
      _zk.ensurePathExists(workflowRunPath);
      _zk.ensurePathExists(getWorkflowRunTasksPath(jobName, newWorkflowRunId));
      _zk.ensurePathExists(getWorkflowRunBulksPath(jobName, newWorkflowRunId));
      addToDataCounter(dataMap, JobManagerConstants.DATA_JOB_NO_OF_ACTIVE_WORKFLOW_RUNS, 1);
      if (_log.isDebugEnabled()) {
        _log.debug("Workflow run '" + newWorkflowRunId + "' for job run '" + jobRunId + "' started");
      }
      return newWorkflowRunId;
    } catch (final Exception e) {
      throw newRunStorageException("Error while starting workflow run with id'" + newWorkflowRunId + "' for job '"
        + jobName + "' with run id '" + jobRunId + "'.", e);
    }
  }

  /** {@inheritDoc} */
  @Override
  public boolean prepareToFinishWorkflowRun(final String jobName, final String jobRunId, final String workflowRunId)
    throws RunStorageException {
    try {
      final ZkConcurrentMap dataMap = ensureJobDataMap(jobName);
      // ignore call if the job run is not known, this can happen, if a job is cancelled while a task is being finished.
      final String currentJobRunId = dataMap.getString(JobManagerConstants.DATA_JOB_ID);
      if (currentJobRunId.equals(jobRunId)) {
        final String workflowRunPath = getWorkflowRunPath(jobName, workflowRunId);
        // ignore call if the workflow run is not there anymore
        if (_zk.exists(workflowRunPath) != null) {
          final String finishingFlagPath = workflowRunPath + '/' + NODE_WORKFLOW_RUN_FINISHING;
          if (_zk.exists(finishingFlagPath) == null) {
            // no one else is currently finishing, reset (potential) entry for finisher in job run data
            dataMap.remove(WORKFLOW_RUN_FINISH_PREFIX + workflowRunId);
            // set finishing flag
            try {
              _zk.createNode(finishingFlagPath, null, CreateMode.EPHEMERAL);
            } catch (final KeeperException.NodeExistsException e) {
              if (_log.isInfoEnabled()) {
                _log.info("Someone else is already finishing workflow run '" + workflowRunId + "'");
              }
              return false;
            } catch (final KeeperException.NoNodeException e) {
              if (_log.isInfoEnabled()) {
                _log.info("Someone else already deleted workflow run '" + workflowRunId + "'");
              }
              return false;
            }
            // set (new) job run data map entry for finisher
            final String itSMe = UUID.randomUUID().toString(); // be sure we are the only one to finish 
            final String whoIsIt = dataMap.putIfAbsent(WORKFLOW_RUN_FINISH_PREFIX + workflowRunId, itSMe);
            if (itSMe.equals(whoIsIt) && _zk.exists(workflowRunPath) != null) {
              return true;
            }
          }
        }
      } else {
        _log.warn("Could not prepare to finish workflow run '" + workflowRunId + "' for a job run '" + jobRunId
          + "' of job '" + jobName + "'. The Job Run has already been removed, has probably been cancelled.");
      }
      return false;
    } catch (final Exception e) {
      throw newRunStorageException("Error while preparing to finish workflow run with id '" + workflowRunId
        + "' for job '" + jobName + "' with run id '" + jobRunId + "'.", e);
    }
  }

  /** {@inheritDoc} */
  @Override
  public void successfulWorkflowRun(final String jobName, final String jobRunId, final String workflowRunId)
    throws RunStorageException {
    try {
      final ZkConcurrentMap dataMap = ensureJobDataMap(jobName);
      // ignore the finish if the job run is not known, this can happen, if a job is cancelled while
      // a task is being finished...
      final String currentJobRunId = dataMap.getString(JobManagerConstants.DATA_JOB_ID);
      if (currentJobRunId.equals(jobRunId)) {
        addToDataCounter(dataMap, JobManagerConstants.DATA_JOB_NO_OF_ACTIVE_WORKFLOW_RUNS, -1);
        addToDataCounter(dataMap, JobManagerConstants.DATA_JOB_NO_OF_SUCCESSFUL_WORKFLOW_RUNS, 1);
      } else {
        _log.warn("Finishing workflow run '" + workflowRunId + "' for a job run '" + jobRunId + "' of job '"
          + jobName + "'. The Job Run has already been removed, has probably been cancelled.");
      }
    } catch (final Exception e) {
      throw newRunStorageException("Error while counting successful workflow run with id '" + workflowRunId
        + "' for job '" + jobName + "' with run id '" + jobRunId + "'.", e);
    }
  }

  /** {@inheritDoc} */
  @Override
  public void failedWorkflowRun(final String jobName, final String jobRunId, final String workflowRunId)
    throws RunStorageException {
    try {
      final ZkConcurrentMap dataMap = ensureJobDataMap(jobName);
      // ignore the finish if the job run is not known, this can happen, if a job is cancelled while
      // a task is being finished...
      final String currentJobRunId = dataMap.getString(JobManagerConstants.DATA_JOB_ID);
      if (currentJobRunId.equals(jobRunId)) {
        addToDataCounter(dataMap, JobManagerConstants.DATA_JOB_NO_OF_ACTIVE_WORKFLOW_RUNS, -1);
        addToDataCounter(dataMap, JobManagerConstants.DATA_JOB_NO_OF_FAILED_WORKFLOW_RUNS, 1);
      }
    } catch (final Exception e) {
      throw newRunStorageException("Error while counting failed workflow run with id '" + workflowRunId
        + "' for job '" + jobName + "' with run id '" + jobRunId + "'.", e);
    }
  }

  /** {@inheritDoc} */
  @Override
  public void deleteWorkflowRunData(final String jobName, final String jobRunId, final String workflowRunId)
    throws RunStorageException {

    // remaining workflow run tasks are counted as cancelled
    try {
      final String tasksPath = getWorkflowRunTasksPath(jobName, workflowRunId);
      final int noOfTasks = _zk.getChildrenSorted(tasksPath).size();
      final ZkConcurrentMap dataMap = ensureJobDataMap(jobName);
      addToDataCounter(dataMap, JobManagerConstants.DATA_JOB_NO_OF_CANCELLED_TASKS, noOfTasks);
    } catch (final Exception e) {
      _log.warn("Exception while updating job run data counter '"
        + JobManagerConstants.DATA_JOB_NO_OF_CANCELLED_TASKS + "': " + e.getMessage());
    }

    // remove whole workflow run data tree (incl. tasks)
    try {
      _zk.deleteTree(getWorkflowRunDataPath(jobName, workflowRunId));
    } catch (final Exception e) {
      throw newRunStorageException("Error while deleting data for workflow run with id '" + workflowRunId
        + "' for job '" + jobName + "' with run id '" + jobRunId + "'.", e);
    }
  }

  /** {@inheritDoc} */
  @Override
  public void deleteWorkflowRun(final String jobName, final String jobRunId, final String workflowRunId)
    throws RunStorageException {
    try {
      // remove whole workflow run tree (incl. 'finishing' flag) 
      _zk.deleteTree(getWorkflowRunPath(jobName, workflowRunId));
      // remove data entry indicating that workflow run is currently finished by me
      final ZkConcurrentMap dataMap = ensureJobDataMap(jobName);
      checkCurrentJobRunId(dataMap, jobRunId);
      dataMap.remove(WORKFLOW_RUN_FINISH_PREFIX + workflowRunId);
    } catch (final Exception e) {
      throw newRunStorageException("Error while deleting workflow run with id '" + workflowRunId + "' for job '"
        + jobName + "' with run id '" + jobRunId + "'.", e);
    }
  }

  /** {@inheritDoc} */
  @Override
  public boolean checkAndCleanupActiveWorkflowRuns(final String jobName, final String jobRunId)
    throws RunStorageException {
    final String workflowRunPath = getWorkflowRunsPath(jobName);
    try {
      final Stat stat = _zk.exists(workflowRunPath);
      if (stat == null) {
        return false; // workflow runs path doesn't exist, ignore
      }
      try {
        final List<String> wfRuns = _zk.getChildrenSorted(workflowRunPath);
        if (wfRuns.size() == 0) {
          return false;
        }
        final ZkConcurrentMap dataMap = ensureJobDataMap(jobName);
        boolean foundActiveWfRun = false;
        for (final String wfRunId : wfRuns) {
          // check if workflow run is finished, resp. if someone else is currently finishing the run
          // (see prepareToFinishWorkflowRun())          
          if (_zk.exists(workflowRunPath + '/' + NODE_WORKFLOW_RUN_FINISHING) != null) {
            if (_log.isInfoEnabled()) {
              _log.info("Someone else is already finishing workflow run '" + wfRunId + "'");
            }
            return true;
          }
          if (dataMap.containsKey(WORKFLOW_RUN_FINISH_PREFIX + wfRunId)) {
            // Someone else started the finishing, but was interrupted (e.g. server crash), so we take over. 
            // We can't know whether the workflow run has already been counted, but we assume so. 
            // If not, counters are wrong but that's acceptable.  
            if (_log.isInfoEnabled()) {
              _log.info("Workflow run '" + wfRunId + "' finishing was interrupted, preparing for take over");
            }
            final boolean doIt = prepareToFinishWorkflowRun(jobName, jobRunId, wfRunId);
            if (doIt) {
              if (_log.isInfoEnabled()) {
                _log.info("Taking over the finishing of workflow run '" + wfRunId + "'");
              }
              deleteWorkflowRunData(jobName, jobRunId, wfRunId); // delete data, preserve finishing flag
              deleteWorkflowRun(jobName, jobRunId, wfRunId);
            }
          } else {
            foundActiveWfRun = true;
          }
        }
        return foundActiveWfRun;
      } catch (final KeeperException.NoNodeException e) {
        return false; // workflow runs node was removed in between, ignore
      }
    } catch (final Exception e) {
      throw newRunStorageException("Error while getting info whether job run with id '" + jobRunId
        + "' has active workflow runs for job '" + jobName + "'.", e);
    }
  }

  @Override
  public boolean hasWorkflowRun(final String jobName, final String jobRunId, final String workflowRunId)
    throws RunStorageException {
    final String workflowRunPath = getWorkflowRunPath(jobName, workflowRunId);
    try {
      final Stat stat = _zk.exists(workflowRunPath);
      return (stat != null);
    } catch (final KeeperException e) {
      throw newRunStorageException("Error while getting info job run with id '" + jobRunId
        + "' has active workflow run '" + workflowRunId + "' for job '" + jobName + "'.", e);
    }
  }

  /** {@inheritDoc} */
  @Override
  public boolean hasTasks(final String jobName, final String jobRunId, final String workflowRunId)
    throws RunStorageException {
    try {
      final String tasksPath = getWorkflowRunTasksPath(jobName, workflowRunId);
      if (_zk.exists(tasksPath) != null) {
        try {
          final int noOfTasks = _zk.getChildrenSorted(tasksPath).size();
          return (noOfTasks > 0);
        } catch (final KeeperException.NoNodeException e) {
          ; // someone removed the step in between -> that's ok
        }
      }
    } catch (final Exception e) {
      throw newRunStorageException("Error while getting info whether workflow run with id '" + workflowRunId
        + "' has tasks for job '" + jobName + "' with run id '" + jobRunId + "'.", e);
    }
    return false;
  }

  /**
   * {@inheritDoc}
   * 
   * @throws RunStorageException
   */
  @Override
  public void startTask(final String jobName, final String jobRunId, final String workflowRunId,
    final String stepId, final String taskId) throws RunStorageException {
    startTask(jobName, jobRunId, workflowRunId, stepId, taskId, 0);
  }

  /** {@inheritDoc} */
  @Override
  public void startTask(final String jobName, final String jobRunId, final String workflowRunId,
    final String stepId, final String taskId, final int numberOfRetries) throws RunStorageException {
    try {
      // check whether current workflow-run ist still active
      final ZkConcurrentMap dataMap = ensureJobDataMap(jobName);
      if (dataMap.containsKey(WORKFLOW_RUN_FINISH_PREFIX + workflowRunId)) {
        throw new IllegalStateException("Tried to store a task for a finished workflow run");
      }
      final String taskPath = getWorkflowRunTasksPath(jobName, workflowRunId) + "/" + taskId;
      _zk.createNode(taskPath, String.valueOf(numberOfRetries).getBytes(UTF_8));

      // inc number of created tasks in job data
      addToDataCounter(dataMap, JobManagerConstants.DATA_JOB_NO_OF_CREATED_TASKS, 1);

    } catch (final Exception e) {
      throw newRunStorageException("Error while storing task '" + taskId + "' in step '" + stepId
        + "' of workflow run with id '" + workflowRunId + "' for job '" + jobName + "' with run id '" + jobRunId
        + "'.", e);
    }
  }

  /** {@inheritDoc} */
  @Override
  public void startTasks(final String jobName, final String jobRunId, final String workflowRunId,
    final Collection<Task> tasks) throws RunStorageException {
    final String workflowRunTaskPath = getWorkflowRunTasksPath(jobName, workflowRunId);
    byte[] taskData;
    try {
      taskData = "0".getBytes(UTF_8);
    } catch (final UnsupportedEncodingException e) {
      taskData = new byte[] { ZERO_AS_UTF_8 }; // this is the result of the above...
    }
    try {
      // check whether current workflow-run ist still active
      final ZkConcurrentMap dataMap = ensureJobDataMap(jobName);
      if (dataMap.containsKey(WORKFLOW_RUN_FINISH_PREFIX + workflowRunId)) {
        throw new IllegalStateException("Tried to store a task for a finished workflow run");
      }
      for (final Task task : tasks) {
        final String taskPath = workflowRunTaskPath + "/" + task.getTaskId();
        _zk.createNode(taskPath, taskData);
      }
      // inc number of created tasks in job data
      addToDataCounter(dataMap, JobManagerConstants.DATA_JOB_NO_OF_CREATED_TASKS, tasks.size());

    } catch (final Exception e) {
      throw newRunStorageException("Error while storing tasks of workflow run with id '" + workflowRunId
        + "' for job '" + jobName + "' with run id '" + jobRunId + "'.", e);
    }
  }

  /** {@inheritDoc} */
  @Override
  public void finishTask(final String jobName, final String jobRunId, final String workflowRunId,
    final String stepId, final String taskId, Map<String, Number> workerCounter,
    final Map<String, String> properties) throws RunStorageException {
    deleteTask(jobName, jobRunId, workflowRunId, stepId, taskId,
      JobManagerConstants.DATA_JOB_NO_OF_SUCCESSFUL_TASKS);
    // update worker counters
    if (workerCounter == null) {
      workerCounter = new HashMap<String, Number>(1);
    }
    workerCounter.put(JobManagerConstants.DATA_JOB_NO_OF_SUCCESSFUL_TASKS, 1);
    updateWorkerData(jobName, stepId, workerCounter, properties);
  }

  /** {@inheritDoc} */
  @Override
  public void obsoleteTask(final String jobName, final String jobRunId, final String workflowRunId,
    final String stepId, final String taskId, final Map<String, String> properties) throws RunStorageException {
    deleteTask(jobName, jobRunId, workflowRunId, stepId, taskId, JobManagerConstants.DATA_JOB_NO_OF_OBSOLETE_TASKS);

    // update worker counters
    final Map<String, Number> workerCounter = new HashMap<String, Number>(1);
    workerCounter.put(JobManagerConstants.DATA_JOB_NO_OF_OBSOLETE_TASKS, 1);
    updateWorkerData(jobName, stepId, workerCounter, properties);
  }

  /** {@inheritDoc} */
  @Override
  public void failedTask(final String jobName, final String jobRunId, final String workflowRunId,
    final String stepId, final String taskId, final boolean failedAfterRetry, final Map<String, String> properties)
    throws RunStorageException {

    final Map<String, Number> workerCounter = new HashMap<String, Number>(1);
    if (failedAfterRetry) {
      deleteTask(jobName, jobRunId, workflowRunId, stepId, taskId,
        JobManagerConstants.DATA_JOB_NO_OF_FAILED_TASKS_RETRIED);
      workerCounter.put(JobManagerConstants.DATA_JOB_NO_OF_FAILED_TASKS_RETRIED, 1);
    } else {
      deleteTask(jobName, jobRunId, workflowRunId, stepId, taskId,
        JobManagerConstants.DATA_JOB_NO_OF_FAILED_TASKS_NOT_RETRIED);
      workerCounter.put(JobManagerConstants.DATA_JOB_NO_OF_FAILED_TASKS_NOT_RETRIED, 1);
    }
    // update worker counters
    updateWorkerData(jobName, stepId, workerCounter, properties);
  }

  /** {@inheritDoc} */
  @Override
  public void retriedTask(final String jobName, final String jobRunId, final String workflowRunId,
    final String stepId, final String taskId, final boolean retryByWorker, final Map<String, String> properties)
    throws RunStorageException {

    final Map<String, Number> workerCounter = new HashMap<String, Number>(1);
    if (retryByWorker) {
      deleteTask(jobName, jobRunId, workflowRunId, stepId, taskId,
        JobManagerConstants.DATA_JOB_NO_OF_RETRIED_TASKS_WORKER);
      workerCounter.put(JobManagerConstants.DATA_JOB_NO_OF_RETRIED_TASKS_WORKER, 1);
    } else {
      deleteTask(jobName, jobRunId, workflowRunId, stepId, taskId,
        JobManagerConstants.DATA_JOB_NO_OF_RETRIED_TASKS_TTL);
      workerCounter.put(JobManagerConstants.DATA_JOB_NO_OF_RETRIED_TASKS_TTL, 1);
    }
    // update worker counters
    updateWorkerData(jobName, stepId, workerCounter, properties);
  }

  /**
   * Update counters for given worker. All counters for a single worker are stored as data in the appropriate zk node
   * for the given worker. If an exception occurs, it isn't thrown, but just logged as warning.
   */
  private void updateWorkerData(final String jobName, final String worker,
    final Map<String, Number> workerCounters, final Map<String, String> properties) {
    try {
      _zk.ensurePathExists(getJobWorkerDataPath(jobName));
      final String workerDataNode = getJobWorkerDataPath(jobName, worker);

      int tries = 0;
      do {
        tries++;

        Stat stat = _zk.exists(workerDataNode);
        if (stat == null) {
          // worker data node doesn't exist yet - create empty data map
          final AnyMap data = DataFactory.DEFAULT.createAnyMap();
          final byte[] dataBytes = _anyWriter.writeBinaryObject(data);
          try {
            _zk.createNode(workerDataNode, dataBytes);
            stat = _zk.exists(workerDataNode);
          } catch (final KeeperException.NodeExistsException e) {
            stat = _zk.exists(workerDataNode); // someone created this znode in between, that's ok
          }
        }

        final int version = stat.getVersion();
        final byte[] workerData = _zk.getData(workerDataNode);
        final AnyMap data = (AnyMap) _anyReader.readBinaryObject(workerData);
        for (final Entry<String, Number> counterEntry : workerCounters.entrySet()) {
          final Number valueToAdd = counterEntry.getValue();
          final String counter = counterEntry.getKey();
          if (data.get(counter) != null) {
            // counter exists - add new value
            final Value old = data.getValue(counter);
            Number newValue = null;
            if (old.isDouble()) {
              newValue = Double.valueOf(old.asDouble() + valueToAdd.doubleValue());
            } else if (old.isLong()) {
              newValue = Long.valueOf(old.asLong() + valueToAdd.longValue());
            }
            data.put(counter, newValue);
          } else {
            // counter doesn't exist yet
            data.put(counter, valueToAdd);
          }
        }

        final byte[] dataBytes =
          _anyWriter.writeBinaryObject(adjustWorkerStartAndEndTimeFromProperties(data, properties));
        try {
          _zk.setData(workerDataNode, dataBytes, version);
          return;
        } catch (final KeeperException.NoNodeException e) {
          ; // someone removed the key in between, try again...
        } catch (final KeeperException.BadVersionException e) {
          ; // someone changed the value in between, try again...
        }

        Thread.sleep(_waitTimeBetweenTriesGenerator.nextInt(MAX_WAIT_TIME_BETWEEN_TRIES));
      } while (tries <= NO_OF_TRIES_WHEN_CONFLICT);

      // no success
      if (_log.isWarnEnabled()) {
        _log.warn("Too much conflicts while updating worker data for worker '" + worker + "', data: "
          + workerCounters);
      }

    } catch (final Exception e) {
      _log.warn("Exception while updating worker data for worker '" + worker + "', data '" + workerCounters + "': "
        + e.getMessage());
    }
  }

  /**
   * Adjusts start and end time for each worker from the workers given data map and the task's properties, where the
   * start and end time are kept.
   * 
   * @param data
   *          the workers data node, that will be adjusted according to the start and end times in the properties.
   * @param properties
   *          the task properties
   * @return amended workers data node
   */
  private AnyMap adjustWorkerStartAndEndTimeFromProperties(final AnyMap data, final Map<String, String> properties) {
    if (properties != null) {
      if (properties.containsKey(Task.PROPERTY_START_TIME)) {
        if (data.containsKey(Task.PROPERTY_START_TIME) && data.get(Task.PROPERTY_START_TIME) != null) {
          // since it is in SMILA format it can be compared alpha-numerically...
          final String currentDateTime = data.getStringValue(Task.PROPERTY_START_TIME);
          final String dateFromProperty = properties.get(Task.PROPERTY_START_TIME);
          if (dateFromProperty != null && (dateFromProperty.compareTo(currentDateTime) < 0)) {
            data.put(Task.PROPERTY_START_TIME, dateFromProperty);
          }
        } else {
          data.put(Task.PROPERTY_START_TIME, properties.get(Task.PROPERTY_START_TIME));
        }
      }
      if (properties.containsKey(Task.PROPERTY_END_TIME)) {
        if (data.containsKey(Task.PROPERTY_END_TIME) && data.get(Task.PROPERTY_END_TIME) != null) {
          // since it is in SMILA format it can be compared alpha-numerically...
          final String currentDateTime = data.getStringValue(Task.PROPERTY_END_TIME);
          final String dateFromProperty = properties.get(Task.PROPERTY_END_TIME);
          if (dateFromProperty != null && (dateFromProperty.compareTo(currentDateTime) > 0)) {
            data.put(Task.PROPERTY_END_TIME, dateFromProperty);
          }
        } else {
          data.put(Task.PROPERTY_END_TIME, properties.get(Task.PROPERTY_END_TIME));
        }
      }
    }
    return data;
  }

  /**
   * Deletes the data of a given task for the given step in the given workflow of the given job run. Additionally sets
   * the appropriatetask counter.
   */
  private void deleteTask(final String jobName, final String jobRunId, final String workflowRunId,
    final String stepId, final String taskId, final String taskCounter) throws RunStorageException {
    try {
      final String taskPath = getWorkflowRunTasksPath(jobName, workflowRunId) + "/" + taskId;

      if (_zk.exists(taskPath) != null) {
        // inc number of given task counter
        final ZkConcurrentMap dataMap = ensureJobDataMap(jobName);
        addToDataCounter(dataMap, taskCounter, 1);
      }
      _zk.deleteNode(taskPath); // if node doesn't exist -> exception
    } catch (final Exception e) {
      throw newRunStorageException("Error while deleting task '" + taskId + "' in step '" + stepId
        + "' of workflow run '" + workflowRunId + "' for job '" + jobName + "' with run id '" + jobRunId + "'.", e);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean hasTask(final String jobName, final String workflowRunId, final String taskId)
    throws RunStorageException {
    try {
      final String jobPath = getWorkflowRunTasksPath(jobName, workflowRunId) + "/" + taskId;
      return (_zk.exists(jobPath) != null);
    } catch (final Exception e) {
      throw newRunStorageException("Error while checking existence of task '" + taskId
        + "' of workflow run with id '" + workflowRunId + "' for job '" + jobName + "'.", e);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int getTaskRetries(final String jobName, final String jobRunId, final String workflowRunId,
    final String stepId, final String taskId) throws RunStorageException {
    try {
      final String jobPath = getWorkflowRunTasksPath(jobName, workflowRunId) + "/" + taskId;
      return Integer.parseInt(new String(_zk.getData(jobPath), UTF_8));
    } catch (final Exception e) {
      throw newRunStorageException("Error while getting retry count for task '" + taskId + "' in step '" + stepId
        + "' of workflow run with id '" + workflowRunId + "' for job '" + jobName + "' with run id '" + jobRunId
        + "'.", e);
    }
  }

  /** {@inheritDoc} */
  @Override
  public void addTransientBulk(final String jobName, final String jobRunId, final String workflowRunId,
    final String bulk) throws RunStorageException {
    try {
      final String bulkPath = getWorkflowRunBulksPath(jobName, workflowRunId) + "/" + encode(bulk);
      _zk.createNode(bulkPath, null);
    } catch (final NodeExistsException e) {
      if (_log.isInfoEnabled()) {
        _log.info("Transient bulk '" + bulk + "' in workflow run '" + workflowRunId + "' for job '" + jobName
          + "' with run id '" + jobRunId + "' has been stored before.");
      }
    } catch (final Exception e) {
      throw newRunStorageException("Error while adding bulk '" + bulk + "' in workflow run '" + workflowRunId
        + "' for job '" + jobName + "' with run id '" + jobRunId + "'.", e);
    }
  }

  /** {@inheritDoc} */
  @Override
  public Collection<String> getTransientBulks(final String jobName, final String jobRunId,
    final String workflowRunId) throws RunStorageException {
    final String bulksPath = getWorkflowRunBulksPath(jobName, workflowRunId);
    try {
      try {
        final Collection<String> encodedBulks = _zk.getChildrenSorted(bulksPath);
        final Collection<String> bulks = new ArrayList<String>(encodedBulks.size());
        for (final String encodedBulk : encodedBulks) {
          bulks.add(decode(encodedBulk));
        }
        return bulks;
      } catch (final KeeperException.NoNodeException nne) {
        return Collections.emptyList();
      }
    } catch (final Exception e) {
      throw newRunStorageException("Error while getting bulks of workflow run '" + workflowRunId + "' for job '"
        + jobName + "' with run id '" + jobRunId + "'.", e);
    }
  }

  /** {@inheritDoc} */
  @Override
  public void clear() throws RunStorageException {
    try {
      final List<String> nodes = _zk.getChildrenSorted(JOBMANAGER_PREFIX);
      for (final String node : nodes) {
        try {
          _zk.deleteTree(JOBMANAGER_PREFIX + '/' + node);
        } catch (final KeeperException.NoNodeException e) {
          ; // node was removed in between, ignore.
        }
      }
    } catch (final KeeperException.NoNodeException e) {
      ; // root node doesn't exist, ignore.
    } catch (final Exception e) {
      throw newRunStorageException("Error in clear().", e);
    }
  }

  /** {@inheritDoc} */
  @Override
  public Collection<String> getTriggeredJobs(final String bucketId) throws RunStorageException {
    final String path = getTriggerBucketPath(bucketId);
    try {
      if (_zk.exists(path) != null) {
        return _zk.getChildrenSorted(path);
      }
      return Collections.emptyList();
    } catch (final Exception ex) {
      throw newRunStorageException("Error getting job names triggered by bucket '" + bucketId + "'", ex);
    }
  }

  /** {@inheritDoc} */
  @Override
  public void addJobTrigger(final String bucketId, final String jobName) throws RunStorageException {
    final String path = getTriggeredJobPath(bucketId, jobName);
    try {
      _zk.ensurePathExists(path);
    } catch (final Exception ex) {
      throw newRunStorageException("Error setting job '" + jobName + "' as triggered by bucket '" + bucketId + "'",
        ex);
    }
  }

  /** {@inheritDoc} */
  @Override
  public void removeJobTrigger(final String bucketId, final String jobName) throws RunStorageException {
    final String path = getTriggeredJobPath(bucketId, jobName);
    try {
      if (_zk.exists(path) != null) {
        _zk.deleteNode(path);
      }
    } catch (final NoNodeException ex) {
      ; // ok, maybe someone else has deleted it.
    } catch (final Exception ex) {
      throw newRunStorageException("Error setting job '" + jobName + "' as triggered by bucket '" + bucketId + "'",
        ex);
    }
  }

  /**
   * @param value
   *          a string
   * @return URL encoded version.
   */
  private String encode(final String value) {
    try {
      return URLEncoder.encode(value, "utf-8");
    } catch (final UnsupportedEncodingException ex) {
      throw new IllegalStateException("UTF-8 MUST BE SUPPORTED!", ex);
    }
  }

  /**
   * @param encodedValue
   *          an URL encoded string
   * @return decoded version.
   */
  private String decode(final String encodedValue) {
    try {
      return URLDecoder.decode(encodedValue, "utf-8");
    } catch (final UnsupportedEncodingException ex) {
      throw new IllegalStateException("UTF-8 MUST BE SUPPORTED!", ex);
    }
  }

  /** @return new {@link RunStorageException} with cause and derived recoverable flag. */
  private RunStorageException newRunStorageException(final String message, final Throwable cause) {
    return new RunStorageException(message, cause, isRecoverable(cause));
  }

  /** check if cause exception qualifies for a retry. */
  private boolean isRecoverable(final Throwable cause) {
    if (cause instanceof RuntimeException) {
      // thrown by ZkService/ZkConnection if connection is lost or client cannot get connected.
      return true;
    }
    if (cause instanceof KeeperException.ConnectionLossException
      || cause instanceof KeeperException.OperationTimeoutException
      || cause instanceof KeeperException.SessionExpiredException
      || cause instanceof KeeperException.SessionMovedException) {
      return true;
    }
    return false;
  }

  @Override
  public void registerJobRunListener(final JobRunListener jobRunListener, final String jobName) {
    // register watcher on job node to get informed about job (run) completion (=deletion).
    final JobRunWatcher watcher = new JobRunWatcher(jobRunListener);
    try {
      _zk.exists(JOBMANAGER_PREFIX + "/" + NODE_JOBS + "/" + jobName, watcher);
    } catch (final KeeperException e) {
      _log.warn("Unable to register job run listener resp. set watcher for job " + jobName, e);
    }
  }

}
