/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Juergen Schumacher, Andreas Weber, Drazen Cindric, Andreas Schank (all Attensity Europe GmbH) - initial
 * implementation
 **********************************************************************************************************************/
package org.eclipse.smila.jobmanager.httphandler;

import java.net.MalformedURLException;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;

import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.AnySeq;
import org.eclipse.smila.datamodel.InvalidValueTypeException;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.http.server.HttpExchange;
import org.eclipse.smila.http.server.HttpStatus;
import org.eclipse.smila.http.server.util.URLCreator;
import org.eclipse.smila.jobmanager.JobRunInfo;
import org.eclipse.smila.jobmanager.definitions.DefinitionPersistence;
import org.eclipse.smila.jobmanager.definitions.JobDefinition;
import org.eclipse.smila.jobmanager.definitions.JobManagerConstants;
import org.eclipse.smila.jobmanager.definitions.JobRunMode;
import org.eclipse.smila.jobmanager.exceptions.ConfigNotFoundException;
import org.eclipse.smila.jobmanager.exceptions.IllegalJobStateException;
import org.eclipse.smila.jobmanager.exceptions.JobDependencyException;
import org.eclipse.smila.jobmanager.exceptions.JobManagerException;
import org.eclipse.smila.jobmanager.exceptions.JobRunModeNotAllowedException;
import org.eclipse.smila.jobmanager.taskgenerator.TaskGeneratorException;

/**
 * Implements the handling of HTTP requests on a single job definition. <br>
 * 
 * URL pattern: <code>smila/jobmanager/jobs/-jobname-/</code> <br>
 * 
 * Methods allowed: <code>GET, DELETE, POST</code>
 */
public class JobHandler extends AJobManagerHandler {

  /** key for the showJobRuns request parameter. */
  public static final String PARAM_RETURNRUNS = "returnRuns";

  /** The key for the definition. */
  public static final String KEY_DEFINITION = "definition";

  /** The key for the job id. */
  public static final String KEY_JOB_ID = "jobId";

  /** The key for the state. */
  public static final String KEY_STATE = "state";

  /** The key for the url. */
  public static final String KEY_URL = "url";

  /** The key for the current. */
  public static final String KEY_CURRENT = "current";

  /** The key for the runs. */
  public static final String KEY_RUNS = "runs";

  /** The key for the history. */
  public static final String KEY_HISTORY = "history";

  /** The key for the mode. */
  public static final String KEY_MODE = "mode";

  /**
   * {@inheritDoc}
   */
  @Override
  public Object process(final String method, final String requestUri, final Record inputRecord,
    final HttpExchange exchange) throws Exception {
    final String jobName = getJobName(requestUri);
    final DefinitionPersistence defPersistence = getDefinitionPersistence();
    final String requestHost = getRequestHost(exchange);
    if (isGetRequest(method)) {
      return getJobDefinition(requestHost, requestUri, inputRecord, jobName, defPersistence);
    }
    if (isPostRequest(method)) {
      return startJob(requestHost, requestUri, inputRecord, jobName);
    }
    if (isDeleteRequest(method)) {
      defPersistence.removeJob(jobName);
    }
    return null;
  }

  /** execute POST request: start job. */
  private Object startJob(final String requestHost, final String requestUri, final Record inputRecord,
    final String jobName) throws JobManagerException, MalformedURLException {
    final JobRunMode jobMode = getJobModeFromRecord(inputRecord);
    final String jobId = getJobRunEngine().startJob(jobName, jobMode);
    final AnyMap outputAny = FACTORY.createAnyMap();
    outputAny.put(KEY_JOB_ID, jobId);
    outputAny.put(KEY_URL, URLCreator.create(requestHost, requestUri, jobId).toString());
    return outputAny;
  }

  /** execute GET request: get job definition and job run history. */
  private Object getJobDefinition(final String requestHost, final String requestUri, final Record inputRecord,
    final String jobName, final DefinitionPersistence defPersistence) throws JobManagerException,
    MalformedURLException {
    final boolean returnDetails = getBooleanParameter(inputRecord, JobManagerConstants.KEY_RETURN_DETAILS, false);
    final boolean returnRuns = getBooleanParameter(inputRecord, PARAM_RETURNRUNS, true);
    final AnyMap outputAny = FACTORY.createAnyMap();
    final JobDefinition job = defPersistence.getJob(jobName);
    if (job == null) {
      throw new ConfigNotFoundException("Job '" + jobName + "' not found.");
    }
    outputAny.put(KEY_DEFINITION, job.toAny(returnDetails));
    if (returnRuns) {
      final AnyMap runsAny = FACTORY.createAnyMap();
      outputAny.put(KEY_RUNS, runsAny);
      final AnyMap currentAny = FACTORY.createAnyMap();
      runsAny.put(KEY_CURRENT, currentAny);
      final JobRunInfo jobRunInfo = getJobRunDataProvider().getJobRunInfo(jobName);
      if (jobRunInfo != null) {
        final String jobId = jobRunInfo.getId();
        currentAny.put(KEY_JOB_ID, jobId);
        currentAny.put(KEY_STATE, jobRunInfo.getState().toString());
        currentAny.put(KEY_URL, URLCreator.create(requestHost, requestUri, jobId).toString());
      }
      final Collection<String> completedJobRuns = getJobRunDataProvider().getCompletedJobRunIds(jobName);
      if (completedJobRuns != null) {
        final AnySeq historyAny = FACTORY.createAnySeq();
        final Iterator<String> completedJobRunsIterator = completedJobRuns.iterator();
        while (completedJobRunsIterator.hasNext()) {
          final String completedId = completedJobRunsIterator.next();
          final AnyMap singleAny = FACTORY.createAnyMap();
          singleAny.put(KEY_JOB_ID, completedId);
          singleAny.put(KEY_URL, URLCreator.create(requestHost, requestUri, completedId).toString());
          historyAny.add(0, singleAny);
        }
        runsAny.put(KEY_HISTORY, historyAny);
      }
    }
    return outputAny;
  }

  /** get an optional boolean parameter from the input record. */
  private boolean getBooleanParameter(final Record parameters, final String name, final boolean defaultValue) {
    if (parameters != null) {
      if (parameters.getMetadata().containsKey(name)) {
        try {
          return parameters.getMetadata().getBooleanValue(name);
        } catch (final InvalidValueTypeException ex) {
          ; // no boolean, ignore.
        }
      }
    }
    return defaultValue;
  }

  /**
   * @return the JobMode from the record, if the record is present. If there is no job mode defined, null will be
   *         returned.
   */
  private JobRunMode getJobModeFromRecord(final Record inputRecord) {
    if (inputRecord != null && inputRecord.getMetadata().containsKey(KEY_MODE)) {
      return JobRunMode.valueOf(inputRecord.getMetadata().getStringValue(KEY_MODE).toUpperCase(Locale.US));
    }
    return null;
  }

  /**
   * Adds HTTP result code 400 ("BAD_REQUEST") for IllegalJobStateException to the exception handling of
   * {@link AJobManagerHandler#getErrorStatus(String, String, Throwable)}. <br>
   * 
   * @param method
   *          HTTP method
   * @param requestUri
   *          request URI
   * @param ex
   *          an exception
   * @return error status code.
   */
  @Override
  protected int getErrorStatus(final String method, final String requestUri, final Throwable ex) {
    if (ex instanceof IllegalJobStateException || ex instanceof JobManagerException
      && (ex.getCause() instanceof TaskGeneratorException && ex.getCause().getCause() == null)
      || ex.getCause() instanceof JobRunModeNotAllowedException || ex.getCause() instanceof JobDependencyException) {
      return HttpStatus.BAD_REQUEST;
    }
    return super.getErrorStatus(method, requestUri, ex);
  }

  /**
   * Gets the job name from the requestUri.
   * 
   * @param requestUri
   *          the request URI.
   * @return he job name from the requestUri.
   */
  private String getJobName(final String requestUri) {
    final List<String> uriParts = getDynamicUriParts(requestUri);
    if (uriParts.size() > 0) {
      return uriParts.get(0);
    }
    throw new IllegalArgumentException("Invalid request URI, does not match uri pattern '" + getUriPattern() + "'.");
  }

  /**
   * {@inheritDoc}
   * 
   * GET, DELETE and POST are currently the only valid methods.
   */
  @Override
  protected boolean isValidMethod(final String method, final String requestUri) {
    return isGetOrPostRequest(method) || isDeleteRequest(method);
  }
}
