/*********************************************************************************************************************
 * Copyright (c) 2008, 2012 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Juergen Schumacher (Attensity Europe GmbH) - initial implementation
 **********************************************************************************************************************/
package org.eclipse.smila.solr.server;

import java.io.IOException;
import java.net.MalformedURLException;
import java.text.MessageFormat;

import org.apache.solr.client.solrj.SolrServer;
import org.apache.solr.client.solrj.SolrServerException;
import org.apache.solr.client.solrj.impl.BinaryRequestWriter;
import org.apache.solr.client.solrj.impl.CommonsHttpSolrServer;
import org.apache.solr.client.solrj.response.SolrPingResponse;
import org.eclipse.smila.solr.util.SolrProperties;
import org.eclipse.smila.solr.util.SolrQueryUtils;

/** {@link SolrServer}s manager for non-embedded Solr servers. */
public class SolrHttpServers extends SolrServers {

  private final SolrProperties _properties;

  /** create instance from given properties. */
  public SolrHttpServers(SolrProperties properties) {
    _properties = properties;
  }

  @Override
  public SolrServer getAdminServer() throws SolrServerException {
    try {
      return createHttpSolrServer(getServerUrl());
    } catch (IOException ex) {
      throw new SolrServerException(ex);
    }
  }

  /** Get a HttpSolrServer by core name. Take the server URL from properties or default one. */
  @Override
  protected SolrServer createServer(String coreName) throws SolrServerException {
    try {
      final String url = getServerUrl();
      final CommonsHttpSolrServer server = createHttpSolrServer(url + "/" + coreName);
      pingServer(server);
      return server;
    } catch (IOException ex) {
      throw new SolrServerException(ex);
    }
  }

  /** get base URL of server. */
  private String getServerUrl() {
    return _properties.getServerUrl();
  }

  /** check if the core is running. For multicore servers this will not work for a server without core name. */
  private void pingServer(final CommonsHttpSolrServer server) throws SolrServerException, IOException {
    final SolrPingResponse ping = server.ping();
    if (SolrQueryUtils.responseStatusIsError(ping)) {
      final String msg = MessageFormat.format("Cannot ping given server for URL: {0}. Response is: {1} {2}", //
        ping.getRequestUrl(), ping.getStatus(), ping.toString());
      throw new SolrServerException(msg);
    }
  }

  private CommonsHttpSolrServer createHttpSolrServer(String url) throws MalformedURLException {
    final CommonsHttpSolrServer server = new CommonsHttpSolrServer(url);
    server.setRequestWriter(new BinaryRequestWriter());
    return server;
  }
}
